from os.path import join

import six.moves.http_client as httplib
import requests
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry


class JsonApiException(Exception):
    pass


class JsonApiClient(object):
    def __init__(self, url, oauth_token, session=None):
        self.__url = url
        self.__oauth_token = oauth_token

        if session is not None:
            self.__session = session
        else:
            self.__session = requests.Session()
            retries = Retry(total=3, backoff_factor=1.0, status_forcelist=[429, 500, 502, 503, 504])
            self.__session.mount(self.__url, HTTPAdapter(max_retries=retries))
            self.__session.headers.update({
                'Content-Type': 'application/json',
                'Authorization': 'OAuth {}'.format(self.__oauth_token),
            })

    def __call__(self, url):
        return JsonApiClient(url=join(self.__url, url), oauth_token=self.__oauth_token, session=self.__session)

    def __handle_response(self, r):
        try:
            r.raise_for_status()
        except requests.HTTPError as e:
            raise JsonApiException('API responded with code {}. Body:\n{}'.format(e.response.status_code, e.response.content))
        except Exception as e:
            raise JsonApiException('Failed to process request for url "{}".\nError: {}'.format(self.__url, e))

        if r.status_code == httplib.NO_CONTENT:
            return {}

        return r.json()

    def get(self, **kwargs):
        return self.__handle_response(self.__session.get(self.__url, **kwargs))

    def options(self, **kwargs):
        return self.__handle_response(self.__session.options(self.__url, **kwargs))

    def head(self, **kwargs):
        return self.__handle_response(self.__session.head(self.__url, **kwargs))

    def post(self, json=None, data=None, **kwargs):
        return self.__handle_response(self.__session.post(self.__url, data=data, json=json, **kwargs))

    def put(self, json=None, **kwargs):
        return self.__handle_response(self.__session.put(self.__url, data=json, **kwargs))

    def patch(self, json=None, **kwargs):
        return self.__handle_response(self.__session.patch(self.__url, data=json, **kwargs))

    def delete(self, **kwargs):
        return self.__handle_response(self.__session.delete(self.__url, **kwargs))
