# -*- coding: utf-8 -*-

from sandbox import sdk2
from sandbox import common
import sandbox.common.types.misc as ctm
import sandbox.common.types.resource as ctr

import json
import logging
from .json_api_client import JsonApiClient


class DeployToPlatformTask(object):
    __PLATFORM_API = "https://platform.yandex-team.ru/api/v1"
    __PLATFORM_UI = "https://platform.yandex-team.ru/projects"

    def get_stable_environments(self):
        raise NotImplementedError()

    def get_testing_environments(self):
        raise NotImplementedError()

    @property
    def oauth_token(self):
        return sdk2.Vault.data(self.owner, "OAUTH_TOKEN")

    @property
    def api_client(self):
        return JsonApiClient(url=self.__PLATFORM_API,
            oauth_token=self.oauth_token)

    def get_release_resources(self):
        all_res = sdk2.Resource.find(task=self, state=ctr.State.READY).limit(0)
        resources = [
            {
                'id': r.id,
                'type': r.type.name,
            } for r in all_res if r.type.releasable
        ]
        logging.info('List of resources to release:\n{}'.format(resources))
        return resources

    def update_environment_resources(self, environment):
        release_resources = self.get_release_resources()

        for component in environment["components"]:
            for res in component["sandboxResources"]:
                res_info = sdk2.Resource.find(id=res["id"]).first()
                for release_res in release_resources:
                    if res_info.type.name == release_res["type"]:
                        logging.info("going to update resource %s %s -> %s", res_info.type.name, res["id"], release_res["id"])
                        res["id"] = release_res["id"]

        environment["comment"] = "auto-deploy from sandbox task #{}".format(self.id)
        logging.info("Release payload: %s", json.dumps(environment, indent=4))

        result = self.api_client("environment/upload").post(environment)
        logging.info('Release of task %s has been sent to Platform', self.id)
        logging.info('Release reply: %s', json.dumps(result, indent=4))

        release_link = "{}/{}?version={}".format(self.__PLATFORM_UI, result["objectId"].replace(".", "/", 2), result["version"])
        return "Platform environment deployed: <a href=\"{}\">{}</a>".format(release_link, release_link)

    def update_environments(self, environments):
        update_info = "<br>".join([self.update_environment_resources(
            self.api_client("environment/dump")(environment).get()
            ) for environment in environments])
        self.set_info(update_info, do_escape=False)

    def on_release(self, additional_parameters):
        logging.info("on_release call")
        self.mark_released_resources(additional_parameters["release_status"])
        registry = common.config.Registry()
        if registry.common.installation != ctm.Installation.PRODUCTION:
            logging.info('Current installation is not PRODUCTION but %s, will not send release to Platform',
                         registry.common.installation)
            return

        logging.info(
            'Gathering information about release of task %s with release parameters %r to send it to Platform',
            self.id, additional_parameters
        )

        if additional_parameters['release_status'] == 'stable':
            self.update_environments(self.get_stable_environments())
        elif additional_parameters['release_status'] == 'testing':
            self.update_environments(self.get_testing_environments())
