# -*- coding: utf-8 -*-

import os
import re
import logging
from cgi import escape

from sandbox import common
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import TaskSelector
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.parameters import SandboxIntegerParameter
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk import sandboxapi

from sandbox.projects import resource_types
from sandbox.projects.common import apihelpers
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils
import sandbox.projects.common.constants as consts
import sandbox.projects.common.build.parameters as build_params


class BuildTypes:
    TRUNK = 'trunk'
    BRANCH = 'branch'
    LAST_TAG = 'last_tag'
    EXISTING_TASK = 'existing_task'


BUILD_TYPE_PARAMETER_NAME = 'build_type'
BRANCH_PARAMETER_NAME = 'branch_name'
BUNDLE_TASK_PARAMETER_NAME = 'custom_build_task_id'
REVISION_PARAMETER_NAME = 'svn_revision'


def generate_priemka_parameters(build_task_type):
    class BuildTypeParameter(SandboxStringParameter):
        name = BUILD_TYPE_PARAMETER_NAME
        description = 'Bundle build type'
        choices = [
            ('Make branch from trunk, then tag and build', BuildTypes.TRUNK),
            ('Tag from branch and build', BuildTypes.BRANCH),
            ('Build from last tag', BuildTypes.LAST_TAG),
            ('Use existing build', BuildTypes.EXISTING_TASK)
        ]
        sub_fields = {
            BuildTypes.TRUNK: [REVISION_PARAMETER_NAME],
            BuildTypes.BRANCH: [BRANCH_PARAMETER_NAME, REVISION_PARAMETER_NAME],
            BuildTypes.LAST_TAG: [BRANCH_PARAMETER_NAME],
            BuildTypes.EXISTING_TASK: [BUNDLE_TASK_PARAMETER_NAME],
        }
        default_value = BuildTypes.BRANCH

    class BundleTaskParameter(TaskSelector):
        name = BUNDLE_TASK_PARAMETER_NAME
        description = 'Bundle task'
        task_type = build_task_type

        @classmethod
        def get_custom_parameters(cls):
            result = super(BundleTaskParameter, cls).get_custom_parameters()
            result.update({
                'show_childs': True
            })
            return result

    class BranchParameter(SandboxStringParameter):
        name = BRANCH_PARAMETER_NAME
        description = 'Branch (leave empty for last existing)'

    class RevisionParameter(SandboxIntegerParameter):
        """
            Номер ревизии репозитория
            Значение '0' означает, что будет использоваться HEAD-ревизия
        """
        name = REVISION_PARAMETER_NAME
        description = 'SVN revision'
        required = False

    return BuildTypeParameter, BundleTaskParameter, BranchParameter, RevisionParameter


include_re = re.compile("INCLUDE: r(\d+) ")
review_re = re.compile("REVIEW:\s+\d+")


def get_diff_revisions(branch_url, start_revision, stop_revision):
    """
        Список ревизий из ветки от start_revision до stop_revision, очищенный от служебных сообщений
    """
    result = []
    for commit in Arcadia.log(branch_url, start_revision, stop_revision):
        revision = commit["revision"]
        author = commit["author"]
        merged_revisions = []
        message_lines = []
        for line in commit["msg"].splitlines():
            line = review_re.sub("", line)
            line = line.lstrip(" >.").rstrip()
            if not line:
                continue
            if line.startswith("merge from trunk:"):
                continue
            include_match = include_re.search(line)
            if include_match is not None:
                merged_revisions.append(int(include_match.group(1)))
                continue
            if isinstance(line, unicode):
                line = line.encode("utf-8")
            message_lines.append(escape(line))

        result.append((revision, author, message_lines, merged_revisions))

    return "\n".join((" * {} ({}, {})".format("\n".join(x[2]), x[1], x[0]) for x in result))


class BasePriemkaTask(SandboxTask):
    BUILD_TASK_KEY = 'build_task_id'
    DESCRIPTION_KEY = 'description'
    CHANGELOG_WEB = 'changelog_web'
    CHANGELOG_PLAIN_TEXT = 'changelog_plaintext'

    @classmethod
    def get_project(cls):
        raise NotImplementedError

    @classmethod
    def get_build_task_type(cls):
        """
            Тип задачи сборки бинарников/конфигов
        """
        raise NotImplementedError

    @classmethod
    def get_build_task_ctx(cls):
        """
            Дополнительные параметры для контекста сборочной задачи
        """
        return {}

    def get_main_resource_type(self):
        """
            Тип ресурса, по которому ищется предыдущий релиз
        """
        raise NotImplementedError

    def get_build_task_id(self):
        """
            Задача сборки бинарников/конфигов
        """
        return self.ctx.get(self.BUILD_TASK_KEY)

    def get_description(self):
        """
            Описание подзадач
        """
        return self.ctx.get(self.DESCRIPTION_KEY)

    def get_web_changelog_link(self):
        """
            Ссылка на список изменений в wsvn
        """
        return self.ctx.get(self.CHANGELOG_WEB)

    def get_plain_text_changelog_resource(self):
        """
            Ресурс с changelog в виде текста, пригодного для ручного редактирования
        """
        return self.ctx.get(self.CHANGELOG_PLAIN_TEXT)

    def sync_subtasks(self):
        utils.check_subtasks_fails(stop_on_broken_children=False, fail_on_first_failure=True)

    def branch_and_revision(self, task_id):
        """
            Название ветки и ревизия таска. Ветка - та, из которой был создан собираемый тег
        """
        task_svn_url = channel.sandbox.get_task(task_id).ctx[consts.ARCADIA_URL_KEY]
        parsed_url = Arcadia.parse_url(task_svn_url)
        if parsed_url.tag is None:
            logging.info("Could not get tag name for task %s", task_id)
            return None
        tag = self.get_project().tag_from_path(parsed_url.tag)
        if tag is None:
            logging.info("Could not get project tag for task %s", task_id)
            return None
        revision = parsed_url.revision
        if revision is None:
            revision = Arcadia.info(task_svn_url)['entry_revision']
        return tag.branch, revision

    def create_build_tasks(self):
        if self.BUILD_TASK_KEY not in self.ctx:
            project = self.get_project()
            build_type = self.ctx[BUILD_TYPE_PARAMETER_NAME]
            branch = None
            all_branches = sorted(project.list_branches(), key=lambda x: x.number)
            if build_type == BuildTypes.TRUNK:
                new_branch_number = all_branches[-1].number + 1 if all_branches else 0
                branch = ArcadiaProject.Branch(project, new_branch_number)
                branch.create_from_trunk(self.ctx[REVISION_PARAMETER_NAME])
                self.ctx[BRANCH_PARAMETER_NAME] = branch.basename()
            elif build_type == BuildTypes.BRANCH or build_type == BuildTypes.LAST_TAG:
                eh.ensure(
                    all_branches,
                    "No branches for project {}".format(project.project)
                )
                branch_name = self.ctx[BRANCH_PARAMETER_NAME]
                if branch_name:
                    branch = ArcadiaProject.Branch.from_name(project, branch_name)
                    eh.ensure(
                        branch.number in [existing_branch.number for existing_branch in all_branches],
                        "Non existing branch {}".format(branch_name)
                    )
                else:
                    branch = all_branches[-1]

            tag = None
            if branch is not None:
                all_branch_tags = sorted(branch.list_tags(), key=lambda x: x.number)
                if build_type in [BuildTypes.TRUNK, BuildTypes.BRANCH]:
                    new_tag_number = all_branch_tags[-1].number + 1 if all_branch_tags else 0
                    tag = ArcadiaProject.Tag(project, branch, new_tag_number)
                    revision = self.ctx[REVISION_PARAMETER_NAME] if build_type == BuildTypes.BRANCH else None
                    tag.create_from_branch(revision)
                elif build_type == BuildTypes.LAST_TAG:
                    eh.ensure(
                        all_branch_tags,
                        "No tags for branch {}".format(branch.description())
                    )
                    tag = all_branch_tags[-1]

            if tag is not None:
                self.ctx[self.DESCRIPTION_KEY] = tag.description() + ' priemka'
                build_task_ctx = {
                    consts.ARCADIA_URL_KEY: tag.arcadia_url(),
                    build_params.StripBinaries.name: self.ctx.get(build_params.StripBinaries.name, False)
                }
                build_task_ctx.update(self.get_build_task_ctx())
                self.ctx[self.BUILD_TASK_KEY] = self.create_subtask(
                    self.get_build_task_type(),
                    self.ctx[self.DESCRIPTION_KEY],
                    input_parameters=build_task_ctx,
                    arch=sandboxapi.ARCH_LINUX
                ).id
            else:
                self.ctx[self.DESCRIPTION_KEY] = self.descr
                self.ctx[self.BUILD_TASK_KEY] = self.ctx[BUNDLE_TASK_PARAMETER_NAME]
        else:
            self.ctx[self.DESCRIPTION_KEY] = self.descr

    def create_changelog(self):
        changelog_resource_id = None
        if self.CHANGELOG_PLAIN_TEXT in self.ctx:
            if self.ctx[self.CHANGELOG_PLAIN_TEXT] is None:
                return
            changelog_resource_id = self.ctx[self.CHANGELOG_PLAIN_TEXT]
            if channel.sandbox.get_resource(changelog_resource_id).is_ready():
                return

        old_main_resource = apihelpers.get_last_released_resource(
            self.get_main_resource_type(),
            arch=sandboxapi.ARCH_LINUX
        )
        if old_main_resource is None:
            logging.info("No resources have been released, will not generate changelog")
            return
        old_build_task = channel.sandbox.get_resource(old_main_resource).task_id
        old = self.branch_and_revision(old_build_task)
        new = self.branch_and_revision(self.get_build_task_id())

        self.ctx[self.CHANGELOG_WEB] = None
        self.ctx[self.CHANGELOG_PLAIN_TEXT] = None
        if old is None or new is None:
            logging.info("No branch info for one of tasks, will not generate changelog")
        elif old[0].basename() != new[0].basename():
            logging.info("Different branches for old and new buld, no diff")
        elif old[1] == new[1]:
            logging.info("Binaries built from same revision, no diff")
        else:
            branch = old[0]
            self.ctx[self.CHANGELOG_WEB] = (
                "https://arcadia.yandex.ru/wsvn/arc/branches/"
                "{}/{}?op=log&sr={}&er={}".format(
                    self.get_project().project,
                    branch.basename(),
                    new[1],
                    old[1],
                )
            )

            changelog_path = self.abs_path("changelog")
            changelog_text = get_diff_revisions(branch.svn_url(), old[1], new[1])
            if isinstance(changelog_text, unicode):
                changelog_text = changelog_text.encode('utf-8')
            changelog_text = \
                "<html><head><meta http-equiv=\"Content-Type\" content=\"text/html; charset=utf-8\"></head><body><pre>" + \
                changelog_text + \
                "</pre></body></html>"

            fu.write_file(changelog_path, changelog_text)

            if changelog_resource_id is None:
                changelog_resource_id = self.create_resource(
                    "changelog", changelog_path,
                    resource_types.OTHER_RESOURCE,
                    attributes={'ttl': 'inf'},
                ).id
            self.mark_resource_ready(changelog_resource_id)
            self.ctx[self.CHANGELOG_PLAIN_TEXT] = changelog_resource_id

    def on_execute(self):
        self.create_build_tasks()
        self.create_changelog()
        self.sync_subtasks()

    def _get_build_resource(self, resource, arch):
        build_task_id = self.get_build_task_id()
        return apihelpers.get_task_resource_id(build_task_id, resource, arch)


class ArcadiaProject:
    """
        Класс для работы с ветками и тегами проекта.
        Рассчитан на структуру проекта:
            branches/<имя проекта>/pre-stable-* - ветки
            tags/<имя проекта>/stable-*/r* - теги

        В тестовом режиме структура меняется на:
            branches/junk/<имя проекта>/pre-stable-*
            tags/junk/<имя проекта>/stable-*/r*
    """
    _pre_stable_prefix = 'pre-stable-'
    _stable_prefix = 'stable-'
    _tag_prefix = 'r'
    _entry_not_found = -1

    @common.utils.classproperty
    def _root_url(cls):
        return Arcadia.parent_dir(Arcadia.trunk_url(), 2)  # remove /trunk/arcadia

    @common.utils.classproperty
    def _arc_trunk(cls):
        return Arcadia.parent_dir(Arcadia.trunk_url(), 1)  # remove /arcadia

    def __init__(self, project, test=False):
        self.project = project
        self.test = test
        tag_re_str = os.path.join(self.project, self._stable_prefix + '(\d+)', self._tag_prefix + '(\d+)')
        self._stable_tag_re = re.compile(tag_re_str)

    def _entities_dir(self, entities_type):
        path = Arcadia.append(self._root_url, entities_type)
        if self.test:
            path = Arcadia.append(path, 'junk')
        path = Arcadia.append(path, self.project)
        return path

    @staticmethod
    def _list_entries(svn_dir, entry_prefix):
        entries = Arcadia.list(svn_dir, depth='immediates').splitlines()
        entry_re = re.compile(r"{}\d+/?$".format(entry_prefix))
        entries = [os.path.normpath(x) for x in entries if entry_re.match(x)]
        return entries

    def branches_dir(self):
        return self._entities_dir("branches")

    def tags_dir(self):
        return self._entities_dir("tags")

    class Branch:
        @staticmethod
        def from_name(project, branch_name):
            return ArcadiaProject.Branch(project, int(branch_name[len(ArcadiaProject._pre_stable_prefix):]))

        def __init__(self, project, number):
            self.project = project
            self.number = number

        def basename(self):
            return ArcadiaProject._pre_stable_prefix + str(self.number)

        def tags_dir_basename(self):
            return ArcadiaProject._stable_prefix + str(self.number)

        def tags_dir(self):
            return Arcadia.append(self.project.tags_dir(), self.tags_dir_basename())

        def svn_url(self):
            return Arcadia.append(self.project.branches_dir(), self.basename())

        def arcadia_url(self):
            return Arcadia.append(self.svn_url(), 'arcadia')

        def description(self):
            return os.path.join(self.project.project, self.basename())

        def list_tags(self):
            tag_names = sorted(ArcadiaProject._list_entries(self.tags_dir(), ArcadiaProject._tag_prefix))
            return [ArcadiaProject.Tag.from_name(self.project, self, x) for x in tag_names]

        def create_from_trunk(self, revision=None):
            branch_url = self.svn_url()

            if Arcadia.check(branch_url):
                eh.check_failed("Cannot create branch '{}' - already exists".format(
                    self.description()
                ))

            Arcadia.copy(
                ArcadiaProject._arc_trunk,
                branch_url,
                self.description(),
                user='zomb-sandbox-rw',
                revision=revision
            )

    class Tag:
        @staticmethod
        def from_name(project, branch, tag_name):
            return ArcadiaProject.Tag(project, branch, int(tag_name[len(ArcadiaProject._tag_prefix):]))

        def __init__(self, project, branch, number):
            self.project = project
            self.branch = branch
            self.number = number

        def create_from_branch(self, revision=None):
            tags_dir = self.branch.tags_dir()

            if not Arcadia.check(tags_dir):
                Arcadia.mkdir(
                    tags_dir,
                    user='zomb-sandbox-rw',
                    message="Creating stable tags dir for branch {}".format(self.branch.description())
                )

            Arcadia.copy(
                self.branch.svn_url(),
                self.svn_url(),
                self.description(),
                user='zomb-sandbox-rw',
                revision=revision
            )

        def basename(self):
            return ArcadiaProject._tag_prefix + str(self.number)

        def svn_url(self):
            return Arcadia.append(self.branch.tags_dir(), self.basename())

        def arcadia_url(self):
            return Arcadia.append(self.svn_url(), 'arcadia')

        def description(self):
            return os.path.join(
                self.project.project,
                self.branch.tags_dir_basename(),
                self.basename()
            )

    def list_branches(self):
        branch_names = sorted(self._list_entries(self.branches_dir(), self._pre_stable_prefix))
        return [ArcadiaProject.Branch.from_name(self, x) for x in branch_names]

    def tag_from_path(self, path):
        match = self._stable_tag_re.search(path)
        if match is None:
            return None
        branch = ArcadiaProject.Branch(self, int(match.group(1)))
        tag = ArcadiaProject.Tag(self, branch, int(match.group(2)))
        return tag
