# -*- coding: utf-8 -*-

"""
Support for various C++ code profilers (namely, arcadia profiler
and gperftools)

Maintainer: mvel@

Please obey PEP8 coding standards and Sandbox StyleGuide here
https://wiki.yandex-team.ru/sandbox/codestyleguide
"""

import os
import re
import logging
from sandbox.projects import resource_types
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk import errors as se
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import sandboxapi
from sandbox import sandboxsdk

from sandbox.projects.common import apihelpers
from sandbox.projects.common import file_utils as fu


def get_profiler_environment(use_gperftools=False, executable_path=None, session_name=None, work_dir=None):
    """
        возвращает переменные окружения для загрузки профайлера в процесс раньше других библиотек
        далее эти переменные передаются в run_process
    """
    if work_dir is None:
        work_dir = channel.task.abs_path()

    if use_gperftools:
        libprof_so_resource = apihelpers.get_last_resource_with_attribute(
            resource_type=resource_types.GPERFTOOLS_PROFILER_SO_BINARY,
            attribute_name=None,
            status=sandboxapi.RESOURCE_READY,
            arch=sandboxsdk.util.system_info()['arch']
        )
        if libprof_so_resource is None:
            raise se.SandboxTaskFailureError("Required profiler library GPERFTOOLS_PROFILER_SO_BINARY valid resource was not found")

        libprof_so_path = channel.task.sync_resource(libprof_so_resource)
        logging.info('profiler so path: "%s"', libprof_so_path)
        profile_name = _get_profile_name(
            use_gperftools=use_gperftools,
            executable_path=executable_path,
            session_name=session_name,
        )
        profile_name = os.path.join(work_dir, profile_name)

        return {
            'LD_PRELOAD': libprof_so_path,
            'CPUPROFILE':  profile_name,
            'CPUPROFILE_FREQUENCY': '100',  # good default value
        }

    libprof_so_resource = apihelpers.get_last_resource(resource_types.LIBPROF_SO_BINARY)
    if libprof_so_resource is None:
        raise se.SandboxTaskFailureError("Required profiler library LIBPROF_SO_BINARY valid resource was not found")

    libprof_so_path = channel.task.sync_resource(libprof_so_resource)
    logging.info('profiler so path: "%s"', libprof_so_path)
    return {'LD_PRELOAD': libprof_so_path, }


def read_profile_data(
    executable_path, pid, session_name,
    work_dir=None,
    profile_res_id=None,
    raw_profile_res_id=None,
    use_gperftools=False,
):
    """
        Считывает лог профайлера после остановки процесса и создает ресурсы:
        PROFILE_STAT - вывод профайлера в текстовом (читабельном) виде после обработки скриптом
        из ресурса ANNOTATE_PROFILE_SCRIPT или GPERFTOOLS_PPROF_SCRIPT
        в зависимости от типа профилировщика

        Аналогично, {SVG,PDF}_PROFILE_STAT - аннотированный профиль
        в соответствующих форматах (только для gperftools)

        RAW_PROFILE_STAT - сырой вывод профайлера (бинарный файл в случае gperftools,
        текстовый для аркадийного профайлера)

        :param executable_path: путь к исполняемому файлу для которого запускался профайлер
        :param pid: process id программы
        :param session_name: имя сессии. определяет имена файлов и description создаваемых ресурсов.
            Должно отличаться для каждого запуска.
        :param work_dir: рабочая директория. та же что использовалась при запуске процесса.
            По умолчанию директория таска.
        :param profile_res_id идентификатор ресурса, куда надо сохранить текстовый профиль
            По умолчанию будет просто создан ещё один ресурс
        :param raw_profile_res_id идентификатор ресурса, куда надо сохранить сырой профиль
            По умолчанию будет просто создан ещё один ресурс
        :param use_gperftools использовать профайлер gperftools вместо аркадийного
            По умолчанию выключено

    """
    if work_dir is None:
        work_dir = channel.task.abs_path()

    file_name = _get_profile_name(
        use_gperftools=use_gperftools,
        executable_path=executable_path,
        pid=pid,
        session_name=session_name,
    )

    profile_file = os.path.join(work_dir, file_name)
    if not os.path.exists(profile_file):
        raise se.SandboxTaskFailureError("profile file '{0}' does not exist".format(profile_file))

    res_type = resource_types.GPERFTOOLS_PPROF_SCRIPT if use_gperftools else resource_types.ANNOTATE_PROFILE_SCRIPT
    annotate_script_resource = apihelpers.get_last_resource(resource_type=res_type)
    annotate_script_path = channel.task.sync_resource(annotate_script_resource)

    profile_res = None
    if profile_res_id:
        profile_res = channel.sandbox.get_resource(profile_res_id)

    _annotate_profile(
        annotate_script_path, executable_path, profile_file, session_name,
        result_path=profile_res.path if profile_res else None,
        use_gperftools=use_gperftools,
    )

    # gperftools have nice other options to render profiles
    if use_gperftools:
        # svg profile
        _annotate_profile(
            annotate_script_path, executable_path, profile_file, session_name,
            use_gperftools=True,
            profile_type='svg',
        )

        # callgrind profile (instead of unusable pdf)
        _annotate_profile(
            annotate_script_path, executable_path, profile_file, session_name,
            use_gperftools=True,
            profile_type='callgrind',
        )

    if raw_profile_res_id:
        raw_profile_res = channel.sandbox.get_resource(raw_profile_res_id)
    else:
        raw_profile_res = channel.task.create_resource(
            'Raw profile at session {0}'.format(session_name),
            'profile.raw.{0}.log'.format(session_name),
            resource_types.RAW_PROFILE_STAT,
        )

    os.rename(profile_file, raw_profile_res.path)
    channel.task.mark_resource_ready(raw_profile_res)


def store_profile_data(executable_path, session_name, work_dir):
    """Lightweight version of read_profile_data without legacy"""

    file_name = _get_profile_name(
        use_gperftools=True,
        executable_path=executable_path,
        session_name=session_name,
    )
    profile_file = os.path.join(work_dir, file_name)
    if not os.path.exists(profile_file):
        raise se.SandboxTaskFailureError("profile file '{0}' does not exist".format(profile_file))

    res_type = resource_types.GPERFTOOLS_PPROF_SCRIPT
    annotate_script_resource = apihelpers.get_last_resource(resource_type=res_type)
    annotate_script_path = channel.task.sync_resource(annotate_script_resource)

    for profile_type in ("text", "svg", "callgrind"):
        _annotate_profile(
            annotate_script_path, executable_path, profile_file, session_name,
            use_gperftools=True,
            profile_type=profile_type,
            result_path=os.path.join(work_dir, "gperftools.{}.{}".format(session_name, profile_type))
        )


def _get_profile_name(use_gperftools=False, executable_path=None, pid=None, session_name=None):
    base_name = os.path.basename(executable_path)
    if use_gperftools:
        # we don't use pid here since it is not known yet
        return '{}.{}.gprofile'.format(base_name, session_name)
    else:
        return '{}.{}.0.profile'.format(base_name, str(pid))


def _annotate_profile(
    annotate_script_path,
    executable_path,
    profile_file,
    session_name,
    result_path=None,
    use_gperftools=False,
    profile_type='text',
    drop_zero_samples=True,
):
    cmd = [annotate_script_path]
    if use_gperftools:
        cmd.append('--' + profile_type)

        # do feature detection as old pprof can miss it
        if 'no_multiple_text' in fu.read_file(annotate_script_path):
            cmd.append('--no_multiple_text')

        if profile_type in ('svg', 'pdf', 'callgrind'):
            cmd += [
                '--nodecount=250',
                '--nodefraction=.001',
                '--edgefraction=.0005',
                '--maxdegree=10',
            ]

    cmd += [executable_path, profile_file]

    # fix default extension for text profiles
    ext = profile_type
    if profile_type == 'text':
        ext = 'txt'

    res_types = {
        'text': resource_types.PROFILE_STAT,
        'svg': resource_types.SVG_PROFILE_STAT,
        'pdf': resource_types.PDF_PROFILE_STAT,  # pdf seems to be unusable, disabled by now
        'callgrind': resource_types.CALLGRIND_PROFILE_STAT,
    }

    if result_path is None:
        result_res = channel.task.create_resource(
            'Profile at session {}, {} mode'.format(session_name, profile_type),
            'profile.{}.{}'.format(session_name, ext),
            res_types[profile_type],
        )
        result_path = result_res.path
    else:
        result_res = None

    log_prefix = "profile-{}".format(profile_type)
    with open(result_path, "w") as profile_out:
        process.run_process(
            cmd,
            stdout=profile_out,
            outputs_to_one_file=False,
            log_prefix=log_prefix,
            wait=True,
        )

    if use_gperftools and profile_type == 'text' and drop_zero_samples:
        # Drop lines that have both self and cumulative zero impact
        profile_lines = fu.read_line_by_line(result_path)
        filtered_lines = [line for line in profile_lines if re.search(r' 0\.0% .+ 0\.0% ', line) is None]
        fu.write_lines(result_path, filtered_lines)

    if result_res is not None:
        channel.task.mark_resource_ready(result_res)
