# Warning: This is and experimental library. API is subject to change

# To generate off_cpu profile you will need to setup your system:
#   * Remount tracefs to allow read acces for all users
#    # mount -o remount,mode=755 /sys/kernel/tracing
#   * Relax paranoid level to open access to kernel tracepoints
#    # echo "-1" >/proc/sys/kernel/perf_event_paranoid

import os.path

from sandbox import sdk2
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import paths

from sandbox.projects import resource_types
from sandbox.projects.common import apihelpers
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils

from . import perf
from . import gperftools


FLAMEGRAPH_NAME = 'flamegraph.svg'


class ProfilingType:
    NONE = "none"
    ON_CPU = "on_cpu"
    OFF_CPU = "off_cpu"
    CPI = "cpi"
    LBR = "lbr"
    GPT_CPU = "gpt_cpu"


class ProfilingTask:
    """
        Mixin for all tasks that what to generate FlameGraph
    """

    class Parameters(sdk2.Parameters):
        ProfilingTypeParameter = sdk2.parameters.String("Performance profiling", default_value=ProfilingType.NONE)
        SaveGprof2dot = sdk2.parameters.Bool('Save gprof2dot')

    def profiling_footer(self):
        if self.Parameters.ProfilingTypeParameter == ProfilingType.NONE:
            return

        header = [
            {"key": "num", "title": "&nbsp;"},
        ]

        artefacts = apihelpers.list_task_resources(task_id=self.id, resource_type=resource_types.PROFILE_RESULTS)
        artefacts.sort(key=lambda r: r.description)

        body = {
            "num": [
                lb.HREF_TO_ITEM.format(link=r.proxy_url, name=r.description or "all")
                for r in artefacts
            ]
        }

        if self.Parameters.ProfilingTypeParameter != ProfilingType.GPT_CPU:
            header.append({"key": "flamegraph", "title": "&nbsp;"})
            body["flamegraph"] = [
                lb.HREF_TO_ITEM.format(link=self.__get_svg_path(r.proxy_url), name="flamegraph")
                for r in artefacts
            ]

        if self.Parameters.SaveGprof2dot:
            header.append({"key": "gprof2dot", "title": "&nbsp;"})
            body["gprof2dot"] = [
                lb.HREF_TO_ITEM.format(link=self.__get_gprof2dot_path(r.proxy_url), name="gprof2dot")
                for r in artefacts
            ]

        return {
            "<h3>Profiling results</h3>": {
                "header": header,
                "body": body,
            }
        }

    def _profiling_init(self, component, data_dir):
        if self.Parameters.ProfilingTypeParameter == ProfilingType.NONE:
            return

        paths.make_folder(data_dir, delete_content=True)
        data_path = os.path.join(data_dir, "perf.data")

        if self.Parameters.ProfilingTypeParameter == ProfilingType.ON_CPU:
            perf.perf_record(
                component,
                data_path
            )
        elif self.Parameters.ProfilingTypeParameter == ProfilingType.OFF_CPU:
            perf.perf_record(
                component,
                data_path,
                events=('sched:sched_switch', 'sched:sched_stat_sleep', 'sched:sched_stat_blocked'),
                all_system=True
            )
        elif self.Parameters.ProfilingTypeParameter == ProfilingType.CPI:
            perf.perf_record(
                component,
                data_path,
                events=('stalled-cycles-frontend', 'cpu-cycles')
            )
        elif self.Parameters.ProfilingTypeParameter == ProfilingType.LBR:
            perf.perf_record(
                component,
                data_path,
                lbr=True
            )
        elif self.Parameters.ProfilingTypeParameter == ProfilingType.GPT_CPU:
            component.set_environment(
                gperftools.get_profiler_environment(
                    use_gperftools=True,
                    executable_path=component.binary,
                    session_name=str(component.port),
                    work_dir=data_dir
                )
            )
        else:
            eh.check_failed("Unsupported profiling mode")

    def _profiling_report(self, component, data_dir, description=""):
        if self.Parameters.ProfilingTypeParameter == ProfilingType.NONE:
            return

        data_path = os.path.join(data_dir, "perf.data")
        svg_path = self.__get_svg_path(data_dir)
        stacks_path = self.__get_stacks_path(data_dir)
        self.create_resource(description, data_dir, resource_types.PROFILE_RESULTS)

        if self.Parameters.ProfilingTypeParameter == ProfilingType.LBR:
            report_path = data_path + ".txt"
            perf.perf_report(data_path, report_path, sort_fields=("symbol_from", "symbol_to", "mispredict"))
        elif self.Parameters.ProfilingTypeParameter == ProfilingType.ON_CPU:
            self.__flame_collapse(data_path, stacks_path, demangle=True)
            self.__flame_graph(stacks_path, svg_path, description=description)
            if self.Parameters.SaveGprof2dot:
                self.__gprof2dot(data_path, self.__get_gprof2dot_path(data_dir))
        elif self.Parameters.ProfilingTypeParameter == ProfilingType.OFF_CPU:
            # TODO: filter output by component executable
            self.__flame_collapse(
                data_path,
                stacks_path,
                tool="stackcollapse-perf-sched.awk",
                fields=["time", "comm", "pid", "tid", "event", "ip", "sym", "dso", "trace"]
            )
            self.__flame_graph(stacks_path, svg_path, description=description)
        elif self.Parameters.ProfilingTypeParameter == ProfilingType.CPI:
            events = {
                "cpu-cycles": stacks_path + ".cpu-cycles",
                "stalled-cycles-frontend": stacks_path + ".stalled-cycles",
            }
            for event, event_stacks_path in events.iteritems():
                self.__flame_collapse(data_path, event_stacks_path, event=event, description=event)
            self.__flame_diff(
                events["stalled-cycles-frontend"], events["cpu-cycles"], svg_path,
                description=description,
            )
        elif self.Parameters.ProfilingTypeParameter == ProfilingType.GPT_CPU:
            gperftools.store_profile_data(
                executable_path=component.binary,
                session_name=str(component.port),
                work_dir=data_dir
            )
        else:
            eh.check_failed("Unsupported profiling mode")

    def _profiling_diff(self, data1_dir, data2_dir, data_diff_dir, description=""):
        paths.make_folder(data_diff_dir)
        success = self.__flame_diff(
            self.__get_stacks_path(data1_dir),
            self.__get_stacks_path(data2_dir),
            os.path.join(data_diff_dir, "flamegraph.diff.svg"),
            description=description
        )
        if success:
            self.create_resource(description, data_diff_dir, resource_types.PROFILE_RESULTS)

    def __flame_diff(self, stacks1_path, stacks2_path, svg_path, description=""):
        if not os.path.exists(stacks1_path) or not os.path.exists(stacks2_path):
            return False

        stacks_path = os.path.basename(stacks1_path) + ".delta"
        self.__flame_tool("difffolded.pl", stacks1_path, stacks2_path, stacks_path)
        self.__flame_graph(stacks_path, svg_path, description=description)
        return True

    def __flame_collapse(self, data_path, stacks_path,
                         tool="stackcollapse-perf.pl", fields=[], description="", event=None, demangle=False):
        dump_path = data_path + ".dump"
        perf.perf_script(data_path, dump_path, fields=fields)
        if demangle:
            demangled_path = dump_path + ".demangled"
            perf.run_cppfilt(dump_path, demangled_path)
            dump_path = demangled_path
        if event is not None:
            filtered_path = dump_path + ".filtered"
            self.__flame_tool("filter-perf-events.awk", "-v", "event={}".format(event), dump_path, filtered_path)
            dump_path = filtered_path
        self.__flame_tool(tool, dump_path, stacks_path)

    def __flame_graph(self, stacks_path, svg_path, description):
        self.__flame_tool("flamegraph.pl", stacks_path, svg_path)

    def __flame_tool(self, flame_tool, *flame_args):
        flame_bundle_id = utils.get_and_check_last_released_resource_id(resource_types.FLAME_GRAPH_BUNDLE)
        flame_bundle_dir = self.sync_resource(flame_bundle_id)

        with open(flame_args[-1], "w") as output_file:
            process.run_process(
                (os.path.join(flame_bundle_dir, flame_tool),) + flame_args[:-1],
                stdout=output_file,
                outputs_to_one_file=False,
                log_prefix=flame_tool
            )

    def __gprof2dot(self, data_path, result_dir):
        paths.make_folder(result_dir, delete_content=True)
        dot_filename = os.path.join(result_dir, "perf.data.gprof2dot.svg")
        dot_strip_filename = os.path.join(result_dir, "perf.data.gprof2dot.strip.svg")

        gprof2dot_bundle_id = utils.get_and_check_last_released_resource_id(resource_types.GPROF2DOT_BUNDLE)
        gprof2dot_path = os.path.join(self.sync_resource(gprof2dot_bundle_id), "gprof2dot.py")
        perf.gprof2dot(gprof2dot_path, data_path + ".dump.demangled", dot_filename, dot_strip_filename)

    def __get_stacks_path(self, data_dir):
        return os.path.join(data_dir, "perf.data.collapsed")

    def __get_svg_path(self, data_dir):
        return os.path.join(data_dir, FLAMEGRAPH_NAME)

    def __get_gprof2dot_path(self, data_dir):
        return os.path.join(data_dir, "gprof2dot")
