# Wrappers over Linux perf

from sandbox.sandboxsdk import process


def perf_record(component, data_path, events=(), all_system=False, lbr=False):
    """
        Configure component to run under control of performance tracer

        data_path - output file with profiling data
        fix_permissions - hack to add  group write permissions to output file
        events - tuple of events to process
        all_system - System-wide collection from all CPUs
        lbr - Add LBR data to samples (Doesn't work inside virtual machines)

        Note: Many tools (e.g. script) work with "fp" variant only
        Warning: Use long options only, because of frequent change of meaning of short options, e.g.,
                 '-D' corresponds to '--no-delay' in 3.13 and corresponds to '--delay' in 3.18
    """

    perf_cmd = ['perf', 'record', '--call-graph', 'fp', '--freq', '99', '--output', data_path]

    if all_system:
        perf_cmd += ['--all-cpus']

    if lbr:
        perf_cmd += ['--branch-any']

    for event in events:
        perf_cmd += ["--event", event]

    def cmd_patcher(cmd):
        return perf_cmd + ["--"] + cmd

    component.run_cmd_patcher = cmd_patcher


def perf_script(data_path, dump_path, fields=[]):
    """
        Run perf script to dump events to output file

        Warning: Use long options only, because of frequent change of meaning of short options, e.g.,
              '-f' corresponds to '--fields' in 3.18 and corresponds to '--force' in 4.4
    """

    perf_cmd = ["perf", "script", "--input", data_path]

    if fields:
        perf_cmd += ["--fields", ",".join(fields)]

    with open(dump_path, "w") as dump_file:
        process.run_process(perf_cmd, stdout=dump_file, outputs_to_one_file=False, log_prefix="perf-script")


def perf_report(data_path, dump_path, sort_fields=[]):
    """
        Run perf report in stdio mode to dump stats to specified output file
    """

    perf_cmd = ["perf", "report", "--stdio", "--input", data_path]
    if sort_fields:
        perf_cmd += ["--sort", ",".join(sort_fields)]

    with open(dump_path, "w") as dump_file:
        process.run_process(perf_cmd, stdout=dump_file, outputs_to_one_file=False, log_prefix="perf-report")


def run_cppfilt(input_path, output_path):
    with open(input_path) as inf:
        with open(output_path, 'w') as outf:
            process.run_process(['c++filt'], stdin=inf, stdout=outf, outputs_to_one_file=False, log_prefix="cppfilt")


def gprof2dot(gprof2dot_path, dump_path, dot_filename, dot_strip_filename):
    """
        SEARCH-2441
        :param gprof2dot_path: path to gprof2dot
        :param dump_path: path to perf.data.dump(.demangled)? file
        :param dot_filename: path to dot resource
        :param dot_strip_filename: path to dot resource
        :return: resource id with gprof2dot results
    """

    to_dot_cmd = "{} -n 0.1 -e 0.1 -f perf -c print -w < {} | dot -Tsvg".format(
        gprof2dot_path,
        dump_path
    )
    to_dot_strip_cmd = "{} -n 0.1 -e 0.1 -f perf -c print -w -s < {} | dot -Tsvg".format(
        gprof2dot_path,
        dump_path
    )
    with open(dot_filename, "w") as f:
        process.run_process(to_dot_cmd, stdout=f, shell=True, outputs_to_one_file=False, log_prefix="gprof2dot")
    with open(dot_strip_filename, "w") as f:
        process.run_process(
            to_dot_strip_cmd,
            stdout=f,
            shell=True,
            outputs_to_one_file=False,
            log_prefix="gprof2dot_strip",
        )
