import logging

from sandbox.projects.common.base_search_quality.threadPool import FakeObj, get_cpu_chunks, ProcessWorker, ThreadWorker, join_workers, ThreadPoolFailureError
from sandbox.projects.common import utils
from sandbox.common import errors


def ProcessData(
    func,
    listToProcess,
    params,
    timeout=None,
    defaultExceptionType=errors.TaskError,
    useProcesses=True,
    numberOfProcesses=1,
    ctx=None,
    log=logging
):
    if params is None:
        params = FakeObj()
    params.SaveErrorsInDir = False

    if ctx:
        useProcesses = ctx.get('use_multiprocessing', False)

        numberOfProcesses = ctx.get('process_count', 0)
        if numberOfProcesses == 0:
            numberOfProcesses = None

        if ctx.get('save_multiprocessing_errors_in_files', False):
            params.SaveErrorsInDir = utils.create_misc_resource_and_dir(
                ctx, 'multiprocessing_errors_resource_id', 'multiprocessing errors', 'multiprocessing_errors')

    if log is not None:
        log.info("threadPool - processing started")

    workers = []

    for chunk in get_cpu_chunks(listToProcess, numberOfProcesses):
        if useProcesses:
            worker = ProcessWorker(func, chunk, params)
        else:
            worker = ThreadWorker(func, chunk, params)

        worker.start()

        workers.append(worker)

    if log is not None:
        log.info("threadPool - workers started")

    join_workers(workers, timeout)

    if log is not None:
        log.info("threadPool - workers ended")

    results = []

    for worker in workers:
        error = worker.GetError()

        if error:
            exception, msg, timestamp = error

            if isinstance(exception, errors.TaskFailure):
                e = ThreadPoolFailureError
            else:
                e = defaultExceptionType
            raise e("An error occured in worker ({}):\n{}".format(worker, msg))

        resultsChunk = worker.GetResults()

        if type(resultsChunk) == list:
            resultsDescr = 'list of {} items'.format(len(resultsChunk))
            results.extend(resultsChunk)
        elif type(resultsChunk) == tuple:
            resultsDescr = 'list of {} items'.format(len(resultsChunk[0]))
            for (idx, elems) in enumerate(resultsChunk):
                if len(results) <= idx:
                    results.append([])

                results[idx].extend(elems)
        else:
            resultsDescr = str(type(resultsChunk))
            results.append(resultsChunk)

        if log is not None:
            if useProcesses:
                workerDescr = '{} (PID: {})'.format(worker, worker.pid)
            else:
                workerDescr = str(worker)
            log.info('results from worker %s: %s', workerDescr, resultsDescr)

    if log is not None:
        log.info("ThreadPool - processing ended")

    return results
