import logging
import time

from sandbox.projects import resource_types
from sandbox.projects.resource_types import releasers as resource_releasers
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.process import run_process
from sandbox import sdk2

from sandbox.projects.common.search import components as sc
from sandbox.projects.common.search import config as sconf
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils

from . import ThreadPool as TP
from . import executable_types as ex_types


GROUP_NAME = 'Wizard Proxy Params'


class PROXY_WIZARD_LOCAL_TRIE(sdk2.Resource):
    """
        Trie for proxywizard
    """
    releasable = True
    releasers = ['petrk', 'algolix', 'pazdnikov'] + resource_releasers.collections_releasers
    auto_backup = True
    any_arch = True


class PROXY_WIZARD_LOCAL_TRIE_BUILDER(sdk2.Resource):
    """
        Trie builder for proxywizard
    """
    releasable = True
    releasers = ['yulddika', 'petrk', 'algolix', 'pazdnikov'] + resource_releasers.collections_releasers
    auto_backup = True
    any_arch = True


class CheckSnippetTypesParameter(parameters.SandboxStringParameter):
    name = 'proxy_wizard_check_snippet_types'
    description = 'Snippet types to find and check (\s+ is separator)'
    group = GROUP_NAME
    required = True


class ResponsesResultParameter(parameters.LastReleasedResource):
    resource_type = resource_types.PROXY_WIZARD_RESPONSES_RESULT
    group = GROUP_NAME


class Responses1ResultParameter(ResponsesResultParameter):
    name = 'proxy_wizard_responses1_resource_id'
    description = 'Proxy Wizard Responses1 Resource'


class Responses2ResultParameter(ResponsesResultParameter):
    name = 'proxy_wizard_responses2_resource_id'
    description = 'Proxy Wizard Responses2 Resource'


class CgiParameter(parameters.SandboxStringParameter):
    multiline = True
    name = 'proxy_wizard_cgi_params'
    description = 'CGI parameters (k=v\\nk1=v1)'
    group = GROUP_NAME
    required = False


class HostnameParameter(parameters.SandboxStringParameter):
    name = 'proxy_wizard_host_name'
    description = 'Hostname ({tld} substitute into tld from queries file)'
    default_value = 'hamster.yandex.{tld}'
    group = GROUP_NAME
    required = True


class CollectionParameter(parameters.SandboxStringParameter):
    name = 'proxy_wizard_collection_name'
    description = 'Collection name'
    default_value = 'yandsearch'
    group = GROUP_NAME
    required = True


class QueriesResourceParameter(parameters.LastReleasedResource):
    name = 'proxy_wizard_queries_resource_id'
    description = 'Queries (cgi_params \\t tld)'
    resource_type = resource_types.PROXY_WIZARD_PLAIN_QUERIES
    group = GROUP_NAME


class ConfigResourceParameter(parameters.LastReleasedResource):
    name = 'proxy_wizard_config_resource_id'
    description = 'Config'
    resource_type = resource_types.PROXY_WIZARD_CONFIG
    group = GROUP_NAME


class MssResourceParameter(parameters.SandboxStringParameter):
    name = 'proxy_wizard_mss_name'
    description = 'MSS name for subst when test responses'
    group = GROUP_NAME
    required = True


class GeoDBDataParameter(parameters.LastReleasedResource):
    name = 'resource_proxy_wizard_geodb_data_id'
    description = 'GeoDB data'
    group = GROUP_NAME
    resource_type = (resource_types.GEODB_DATA)


class BinaryProxyWizardParameter(parameters.LastReleasedResource):
    name = 'resource_proxy_wizard_bin_id'
    description = 'Executable'
    group = GROUP_NAME
    resource_type = (
        ex_types.PROXY_WIZARD_TRAFFIC_EXECUTABLE,
        ex_types.PROXY_WIZARD_TV_EXECUTABLE,
        ex_types.PROXY_WIZARD_AFISHA_EXECUTABLE,
        ex_types.PROXY_WIZARD_FASTRES_EXECUTABLE,
    )


class ProcessCountParameter(parameters.SandboxIntegerParameter):
    name = 'process_count'
    description = 'Process count'
    default_value = 0


class SearchExecutable(sc.SearchExecutableComponent):
    def __init__(
        self, myname, work_dir, binary, config_file,
        config_params="",
        ctx=None,
        config_class=sconf.SimpleConfigBase,
        config_kws={},
    ):
        self.name = myname
        self.ctx = ctx
        self.config_kws = config_kws

        super(SearchExecutable, self).__init__(
            work_dir=work_dir,
            config_file=config_file,
            config_class=config_class,
            config_params=config_params,
            port=0,
            binary=binary
        )

    def process_queries(self, queries, params):
        with utils.TimeCounter('processing queries'):
            responses = self.use_component(
                lambda: TP.ProcessData(params.ProcessQueries, queries, params, ctx=self.ctx)
            )

        return responses

    def _get_run_cmd(self, config_path):
        args = []
        for k, v in self.config_kws.iteritems():
            args.append("-V")
            args.append("{}={}".format(k, v))

        return args + [config_path]

    def wait(self):
        cmd = ['lsof', '-Pan', '-p', self.process.pid, '-i4', '-sTCP:LISTEN', '-Fn', '|', 'awk', '-F:', "'$1 == \"n*\" { print $2 }'"]

        # wait until the daemon start listening
        for x in range(5):
            out = run_process(cmd, shell=True, outs_to_pipe=True, check=False, wait=False)
            port = ''.join(out.stdout.readlines()).strip()
            if port:
                break
            time.sleep(1)

        eh.ensure(port, 'Unable to obtain a listening port')

        self.port = port
        logging.info('%s is listening on %s', self.name, self.port)
