# -*- coding: utf-8 -*-

import hashlib
import logging


def make_urls(url, index_params=['text', 'lr'], add_params='', exclude_from_fetch_params=[]):
    """
        Generates 3 values:
          1. index URL (URL with only index_params remained. All other URL parameters are filtered).
          2. fetch URL (original URL with 'add_params' appended).
          3. index URL hash.

        Index URL parameters are filtered. Filter behaviour is controlled by index_params parameter.
        Fetch URL parameters are not filtered at all. Fetch URL is just a formatted string: {url}&{add_params}.

        :param url: basic url for generation.
        :param index_params: list of parameter names that should be kept during index URL generation.
                             when list is None: copy all parameters during index URL from fetch URL generation.
                             when list is empty: drop all parameters during index URL from fetch URL generation.
        :param add_params: additional parameters that should be added to fetch UTL.
        :param exclude_from_fetch_params: parameters that should be excluded from fetch url

        :rtype: tuple
        :return: (index_url, fetch_url, url_hash)
    """

    # Generate filter for filtering parameters for index URL generation:
    def filter_index_param(param):
        res = param[0] in index_params
        return res

    # Generate filter for filtering parameters for fetch URL generation:
    def filter_fetch_param(param):
        res = param[0] not in exclude_from_fetch_params
        return res

    def apply_filter(url, filter_func):
        # Prepare parameter list to filtration.
        url_head, url_params_str = url.split('?')
        url_params = url_params_str.split('&')  # list of <a>=<b> strings
        tuple_params = map(lambda s: s.split('=', 1), url_params)  # list of (<a>, <b>) tuples

        # Filter parameters list for index file:
        tuple_params = filter(filter_func, tuple_params)

        # Make new URL for index file:
        params_str = '&'.join('{0}={1}'.format(k, v) for k, v in tuple_params)
        return '{0}?{1}'.format(url_head, params_str)

    if index_params:
        index_filter = filter_index_param
    elif index_params is None:
        index_filter = lambda param: True
    else:
        index_filter = lambda param: False

    if exclude_from_fetch_params:
        fetch_filter = filter_fetch_param
    else:
        fetch_filter = lambda param: True

    # Add 'add_params' to original URL:
    if add_params:
        url = '{0}&{1}'.format(url, add_params)

    # Fetch URL is a result of concatenation original URL with add_params.
    # Fetch URL parameters will be filtered a little.
    fetch_url = apply_filter(url, fetch_filter)
    index_url = apply_filter(url, index_filter)

    url_hash = hashlib.md5(index_url).hexdigest()

    logging.debug("make_urls({0}, {1}, {2}, {3})."
                  "\n\tIndex: '{4}'"
                  "\n\tFetch: '{5}'"
                  "\n\thash: '{6}'".format(url, index_params, add_params, exclude_from_fetch_params,
                                           index_url, fetch_url, url_hash))

    return index_url, fetch_url, url_hash
