#!/usr/bin/env python

import json
import logging
from six.moves.urllib import (
    request,
    parse,
    error
)


_logger = logging.getLogger(__name__)


class RestApi(object):
    def __init__(self, token, netloc, api_root, scheme="https"):
        self._token = token
        self._netloc = netloc
        self._api_root = api_root
        self._scheme = scheme

    def _get(self, method, query={}):
        url = self._make_url(method, query)
        headers = self._make_headers()

        _logger.debug("GET {}".format(url))

        r = request.Request(url, headers=headers)
        response = request.urlopen(r)
        response_str = response.read()

        _logger.debug("Response: {}".format(response_str))

        return json.loads(response_str)

    def _post(self, method, data={}):
        url = self._make_url(method)
        json_data = json.dumps(data)
        headers = self._make_headers_for_json()

        _logger.debug("POST {}".format(url))
        _logger.debug("Request: {}".format(json_data))

        r = request.Request(url, data=json_data, headers=headers)
        try:
            response = request.urlopen(r)
        except error.HTTPError as e:
            _logger.error('Error: {}'.format(e.read()))
            raise

        response_str = response.read()

        _logger.debug("Response: {}".format(response_str))

        return json.loads(response_str)

    def _make_url(self, method, query={}):
        obj = parse.ParseResult(
            scheme=self._scheme,
            netloc=self._netloc,
            path=parse.urljoin(self._api_root, method),
            params="",
            query=parse.urlencode(query),
            fragment=""
        )
        return parse.urlunparse(obj)

    def _make_headers(self):
        return {
            "Authorization": ' '.join(["OAuth", self._token])
        }

    def _make_headers_for_json(self):
        return {
            "Authorization": ' '.join(["OAuth", self._token]),
            "Content-Type": "; ".join(["application/json", "charset=utf-8"]),
            "Accept":       "; ".join(["application/json", "charset=utf-8"])
        }
