import os
import shutil
import tempfile
import logging

from sandbox.sandboxsdk import ssh
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.parameters import SandboxBoolParameter


class ShouldMakeNewRelease(SandboxBoolParameter):

    name = "should_make_new_release_prop"
    description = "Make new release:"
    default_value = False
    required = True


_VERSION_FILE = "VERSION"


def add_version_file(path, version):
    with open(os.path.join(path, _VERSION_FILE), 'w') as version_file:
        version_file.write('{}'.format(version))


_INIT_RELEASE_DIR_MESSAGE = "Initialize new {} release dir at {}"
_MAKE_NEW_RELEASE_MESSAGE = "Make a new {} release at version {} from stable version {}"
_COPY_RELEASE_BRANCH_MESSAGE = "Copy branch {} to {}; no merge"

_STABLE_BRANCH_NAME = "stable"
_MINIMAL_VERSION_STRING_SIZE = 4
_MINOR_VERSION_STRING_SIZE = 3


def make_copy_branch_message(src, dst):
    return _COPY_RELEASE_BRANCH_MESSAGE.format(src, dst)


def make_new_release_branch(version_string):
    return _STABLE_BRANCH_NAME + '-' + version_string


def get_stable_name(current_path):
    """ Gets the name of the stable branch from the branch field of ArcadiaURL,
    if the field is None, stable-0 returned,
    the last component of the slash separated name is returned otherwise"""

    if current_path.branch is None:
        raise ValueError("Url has no branch: {}".format(current_path))

    branch = current_path.branch
    return branch.rsplit('/')[-1] if '/' in branch else branch


def parse_stable_version(branch):
    name_parts = branch.split('-')
    if name_parts[0] != _STABLE_BRANCH_NAME or len(name_parts) != 2:
        raise ValueError("Releases only allowed from stable branches: {}".format(branch))
    return int(name_parts[1])


def parse_major_minor_version(version_string):
    """Parse mjaor and minor version from a string in formatted like \d+\d{3},
    last three digits are minor version (padded with zeroes on the left), anything before that - major versionr.
    If the string passed is nonconformant then versions 0, 0 are returned"""

    if len(version_string) < _MINIMAL_VERSION_STRING_SIZE:
        raise ValueError('Malformed version string: {}'.format(version_string))
    return int(version_string[:len(version_string) - _MINOR_VERSION_STRING_SIZE]), int(version_string[-_MINOR_VERSION_STRING_SIZE:])


def make_version_string(major_version, minor_version):
    return str(major_version) + str(minor_version).zfill(_MINOR_VERSION_STRING_SIZE)


class Releaser(object):

    def __init__(self, service_name, vault_name, vault_owner, release_dir):
        self._service_name = service_name
        self._task = None
        self._vault_name = vault_name
        self._vault_owner = vault_owner
        self._release_dir = release_dir
        self._service_dir = os.path.dirname(release_dir)
        self._release_folder_name = os.path.basename(release_dir)
        self._version_file = os.path.join(release_dir, _VERSION_FILE)

    def set_task(self, task):
        self._task = task

    def _make_new_release_message(self, current_version, stable_version):
        return _MAKE_NEW_RELEASE_MESSAGE.format(self._service_name, current_version, stable_version)

    def _add_true_committer(self, message):
        return message + '; performed on behalf of {}'.format(self._task.author)

    def _commit_to_tree(self, path, message):
        with ssh.Key(self._task, self._vault_owner, self._vault_name):
            return Arcadia.commit(path, self._add_true_committer(message), self._vault_owner)

    def _bump_release_version(self, local_path, major_stable_version):
        """Gets the current release version and increments it, storing the result.
        If the stable branch version has changed then changes the current major stable version,
        increments minor one otherwise"""
        major_version = 0
        minor_version = 0
        release_version_file_path = os.path.join(local_path, _VERSION_FILE)
        with open(release_version_file_path) as release_version_file:
            major_version, minor_version = parse_major_minor_version(release_version_file.read().strip())

        if major_stable_version is not None and major_version != major_stable_version:
            major_version = major_stable_version
            minor_version = 1
        else:
            minor_version += 1

        version_string = make_version_string(major_version, minor_version)
        with open(release_version_file_path, "w") as release_version_file:
            release_version_file.write(version_string)
        self._commit_to_tree(release_version_file_path, self._make_new_release_message(version_string, major_stable_version))
        return version_string

    def _make_release_dir(self, local_release_dir):
        if os.path.exists(local_release_dir) and not os.path.isdir(local_release_dir):
            os.remove(local_release_dir)
        os.mkdir(local_release_dir)

    def _make_new_release_version_file(self, local_release_dir):
        release_version_file_path = os.path.join(local_release_dir, _VERSION_FILE)
        with open(release_version_file_path, "w") as release_version_file:
            release_version_file.write("0" * _MINIMAL_VERSION_STRING_SIZE)
        return release_version_file_path

    def init_release_dir(self):
        """Initializes the release directory with release version file and commits it to the repository,
        if one has not existed before"""
        local_path = None
        should_make_directory = not Arcadia.check(self._release_dir)
        should_make_version_file = not Arcadia.check(self._version_file)

        if not should_make_version_file:
            return

        local_path = tempfile.mkdtemp()
        local_release_dir = local_path

        if should_make_directory:
            logging.info("Release directory at {} is not initialized, making one".format(self._release_dir))
            Arcadia.checkout(self._service_dir, local_path, depth="immediates")
            local_release_dir = os.path.join(local_path, self._release_folder_name)
            self._make_release_dir(local_release_dir)

        if should_make_version_file:
            if not should_make_directory:
                Arcadia.checkout(self._release_dir, local_release_dir, depth="immediates")
            release_file = self._make_new_release_version_file(local_release_dir)
            if local_release_dir == local_path:
                local_release_dir = release_file

        Arcadia.add(local_release_dir)

        self._commit_to_tree(local_path, _INIT_RELEASE_DIR_MESSAGE.format(self._service_name, self._release_dir))
        shutil.rmtree(local_path)

    def get_current_release_version(self):
        """Downloads release version file and get the release version string from it"""
        temp_filename = ""
        with tempfile.NamedTemporaryFile(delete=False) as local_file:
            temp_filename = local_file.name

        Arcadia.export(self._version_file, temp_filename)
        result = "0" * _MINIMAL_VERSION_STRING_SIZE
        with open(temp_filename) as local_file:
            result = local_file.read().strip()
        os.remove(temp_filename)
        return result

    def make_new_release_if_needed(self, url):
        """If specified in the task, increments the release version and copies the specified branch to a new release branch"""
        if self._task.ctx.get(ShouldMakeNewRelease.name):
            current_path = Arcadia.parse_url(url)
            branch_name = get_stable_name(current_path)
            local_path = tempfile.mkdtemp()
            Arcadia.checkout(self._release_dir, local_path, depth="files")
            release_version = self._bump_release_version(local_path, parse_stable_version(branch_name))
            shutil.rmtree(local_path)
            new_release_branch_name = make_new_release_branch(release_version)
            copied_release_branch_path = os.path.join(self._release_dir, new_release_branch_name)
            with ssh.Key(self._task, self._vault_owner, self._vault_name):
                Arcadia.copy(os.path.dirname(url),
                             copied_release_branch_path,
                             self._add_true_committer(make_copy_branch_message(branch_name, copied_release_branch_path)),
                             user=self._vault_owner,
                             revision=current_path.revision)

    def add_release_file(self, rerankd_path):
        release_version = self.get_current_release_version()
        add_version_file(rerankd_path, release_version)
