import os
import os.path
from os.path import join as pj
import time
import shutil
import re

from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk import network
from sandbox.sandboxsdk.paths import get_unique_file_name, make_folder, get_logs_folder
from sandbox.sandboxsdk.errors import SandboxTaskFailureError


class REMRunner(object):
    def __init__(self, arcadia_root, root_dir, log_dir=None, environ={}):
        self.arcadia_root = arcadia_root
        self.root_dir = os.path.abspath(root_dir)
        self.tmp_dir = get_unique_file_name(self.root_dir, 'tmp')
        self.log_dir = log_dir or get_logs_folder()
        self.subprocesses = []
        self.server_port = self.find_port(range(20000, 21000))
        self.rem_url = 'http://localhost:' + str(self.server_port)
        self.ctx = dict()
        self.scripts_dir = get_unique_file_name(self.root_dir, 'rem')
        self.environ = environ

    def get_scripts_dir(self):
        return self.scripts_dir

    def rem_tool_path(self):
        return pj(self.get_scripts_dir(), 'client', 'rem-tool')

    def rem_tool_prefix(self):
        return self.rem_tool_path() + ' -u ' + self.rem_url + ' '

    def with_rem_tool(self, cmd, *args, **kwargs):
        kwargs["shell"] = True
        return run_process(self.rem_tool_prefix() + cmd, *args, **kwargs)

    def get_pythonpaths(self):
        return [
            self.get_scripts_dir(),
            os.path.join(self.get_scripts_dir(), 'client')
        ]

    def setup_server(self):
        self.create_directores()
        self.checkout_scripts()
        self.write_config()
        self.run_server()

    def teardown_server(self):
        for p in self.subprocesses:
            p.terminate()
        alive = self.subprocesses
        time.sleep(5)
        while alive:
            alive = [p for p in alive if p.poll() is None]
            for p in alive:
                p.kill()

    def create_directores(self):
        make_folder(self.root_dir, delete_content=True)
        make_folder(self.tmp_dir)

    def checkout_scripts(self):
        ar, rev = self.arcadia_root, 'HEAD'
        if '@' in ar:
            ar, rev = ar.split('@', 1)

        Arcadia.export('%s/rem/server@%s' % (ar, rev), self.scripts_dir)
        Arcadia.export('%s/rem/client@%s' % (ar, rev), self.scripts_dir + '/client')

    def write_config(self):
        shutil.copy(pj(self.scripts_dir, 'setup_env-default.sh'), pj(self.scripts_dir, 'setup_env.sh'))
        with open(pj(self.scripts_dir, 'network_topology.cfg'), 'w') as ntf:
            print >> ntf, '[servers]\nlocalhost = http://localhost:{}, http://localhost:{}'.format(self.server_port, self.server_port+1)

        with open(pj(self.scripts_dir, 'rem.cfg-default'), 'r') as src, open(pj(self.scripts_dir, 'rem.cfg'), 'w') as dst:
            section = ''
            for l in src:
                o = l
                m = re.search(r'^\[([^[]+)\]', l)
                if m:
                    section = m.group(1)
                if l.startswith("project_dir"):
                    o = "project_dir = {}".format(self.scripts_dir)
                elif l.startswith("port"):
                    o = "port = {}".format(self.server_port)
                elif l.startswith("readonly_port"):
                    o = "readonly_port = {}".format(self.server_port + 1)
                elif l.startswith("system_port"):
                    o = "# system_port = UNUSED"  # we don't do remote tags
                elif l.startswith("network_topology"):
                    o = "network_topology = file://{}/network_topology.cfg".format(self.scripts_dir)
                elif l.startswith("network_hostname"):
                    o = "network_hostname = localhost"  # we don't do remote tags
                elif l.startswith("send_emails"):
                    o = "send_emails = no"  # we don't want to send mail
                elif section == 'log' and l.startswith("dir"):
                    o = 'dir = {}'.format(self.log_dir)
                print >> dst, o

    def run_server(self):
        pwd = os.getcwd()
        try:
            os.chdir(self.scripts_dir)
            cmd = ['/skynet/python/bin/python', pj(self.scripts_dir, 'rem-server.py'), 'start']
            pp = os.environ['PYTHONPATH'] if 'PYTHONPATH' in os.environ else ''
            penv = {
                'PYTHONPATH': ':'.join([pp, self.scripts_dir, pj(self.scripts_dir, 'client')]),
                'TMPDIR': self.tmp_dir,
                'TMP': self.tmp_dir,
            }
            penv.update(self.environ)
            self.subprocesses.append(run_process(cmd, environment=penv, shell=False, wait=False, check=False, log_prefix='rem-server'))
            self.service_url = 'http://localhost:{}'.format(self.server_port)
        finally:
            os.chdir(pwd)

    def find_port(self, itr):
        for p in itr:
            if (network.is_port_free(p)):
                return p
        raise SandboxTaskFailureError("Could not find free port for REM process")
