# -*- coding: utf-8 -*-

import json
import os
import time
import socket
import sys
import tarfile
import logging
import re
from glob import glob

import requests
from sandbox.common.fs import make_folder
from sandbox.projects import resource_types
from sandbox.projects.common import apihelpers
from sandbox.projects.common.report_renderer import resources as rr_resources
from sandbox.projects.common.utils import sync_resource
from sandbox.projects.common.utils import wait_searcher_start, wait_searcher_stop
from sandbox.projects.common.utils import gimme_port
from sandbox.sandboxsdk.paths import copy_path
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

# Масимальное время ожидания поднятия worker'ов report-renderer в секундах
MAX_WORKERS_START_TIME = 60


class ReportRendererProvider(object):

    TEMPLATE_PACKAGE_NAME = 'report-templates'
    TEMPLATE_PACKAGE_NAME_PATTERN_RE = re.compile('report-(?:\w+-)?templates(?:-.+)?')

    def __init__(
            self,
            report_renderer_resource_id,
            templates_resource_id,
            port=13300,
            admin_port=13301,
            workers=16,
            apphost=False,
            logs='logs',
            task=None,
            numa_node=-1,
            dump_start_end_heap_stats=False,
            heap_dump_interval=None,
            log_attrs=None):
        """
        :param report_renderer_resource_id:
        :param templates_resource_id:
        :param port:
        :param admin_port:
        :param workers:
        :param apphost: Bool. Запуск в режиме аппхостового сервиса.
        :param logs:
        :param task:
        :param numa_node:
        :param dump_start_end_heap_stats: Bool. Если включен, то в начале и в конце работы скрипта будут сняты дампы
        памяти.
        :param heap_dump_interval: Float|None. Если указано значение больше 0, то раз в указанное время будет сниматься
        дамп памяти. Работает независимо от от dump_start_end_heap_stats.
        :param log_attrs: Dict|None. Словарь с атрибутами, добавляемый к ресурсам логов
        """
        self.report_renderer_resource_id = report_renderer_resource_id
        self.templates_resource_id = templates_resource_id
        self.rr_port = port
        self.rr_admin_port = admin_port
        self.rr_workers = workers
        self.apphost = apphost

        self.task = task
        self.numa_node = numa_node
        self.unistat_file_path = 'unistat-{}.json'.format(port)
        self.dump_start_end_heap_stats = dump_start_end_heap_stats
        self.heap_dump_interval = heap_dump_interval
        self.log_attrs = log_attrs

        self.process = None
        self.heap_dumper_process = None

        if not self.rr_port:
            self.rr_port = gimme_port()

        if self.apphost:
            self.rr_admin_port = self.rr_port - 1
        elif not self.rr_admin_port:
            self.rr_admin_port = gimme_port()

        self.rr_logs = os.path.join(os.getcwd(), logs)

    @property
    def rr_v8_opts(self):
        return '--min_semi_space_size=64 --max_semi_space_size=64 --max_old_space_size=1024'

    @property
    def rr_shed_policy(self):
        return 'none'

    @property
    def rr_templates_version(self):
        return 'unknown'

    @property
    def rr_cpu_profile(self):
        return 'ondemand'

    @property
    def rr_backlog(self):
        return 16

    @property
    def rr_node_args(self):
        return ''

    @property
    def rr_luster_path(self):
        return '{dir}/report-renderer/node_modules/luster/bin/luster.js'.format(dir=os.getcwd())

    @property
    def rr_luster_config_path(self):
        return '{dir}/report-renderer/config.js'.format(dir=os.getcwd())

    @property
    def rr_routes_json(self):
        return '{dir}/report-templates/routes.js*'.format(
            dir=os.getcwd(),
        )

    @property
    def rr_templates_place(self):
        return '{dir}/report-templates'.format(dir=os.getcwd())

    @property
    def rr_templates_filter(self):
        return ''

    @property
    def rr_custom_log(self):
        return ''

    @property
    def rr_service_properties(self):
        # Should return string with correct JSON
        return ''

    @property
    def rr_geobase_path(self):
        return ''

    def get_run_args(self):
        run_args = r'''{node_args} {v8_opts} {luster_path} {luster_config}
            --port={port} --admin-port={admin_port} --logs={logs_dir}
            --workers={workers} --templates-package={templates} --templates-version={templates_version}
            --sched-policy={sched_policy} --cpu-profile={cpu_profile}
            --lsock-backlog={backlog}'''.format(
            v8_opts=self.rr_v8_opts,
            templates_version=self.rr_templates_version,
            sched_policy=self.rr_shed_policy,
            cpu_profile=self.rr_cpu_profile,
            backlog=self.rr_backlog,
            workers=self.rr_workers,
            node_args=self.rr_node_args,
            luster_path=self.rr_luster_path,
            luster_config=self.rr_luster_config_path,
            templates=self.rr_routes_json,
            templates_filter=self.rr_templates_filter,
            port=self.rr_port,
            admin_port=self.rr_admin_port,
            logs_dir=self.rr_logs,
        )

        if self.apphost:
            run_args += ' --apphost-service --loop-queue-size 1'

        templates_filter = self.rr_templates_filter
        if templates_filter:
            run_args += ' --templates-filter={templates_filter}'.format(templates_filter=templates_filter)

        custom_log = self.rr_custom_log
        if custom_log:
            run_args += ' --custom-log={custom_log}'.format(custom_log=custom_log)

        service_properties = self.rr_service_properties
        if service_properties:
            run_args += ' --service-properties=\'{service_properties}\''.format(service_properties=service_properties)

        geobase_path = self.rr_geobase_path
        if geobase_path:
            run_args += ' --geobase={geobase_path}'.format(geobase_path=geobase_path)

        return run_args

    def alive(self):
        if not self.process:
            return False
        return self.process.poll() is None

    def prepare_resources(self):
        self.prepare_templates()
        self.prepare_report_renderer()
        self.prepare_clickdaemon_keys()

    def prepare_templates(self):
        self.unarchive_resource(self.templates_resource_id, self.TEMPLATE_PACKAGE_NAME)
        return self.task.path(self.TEMPLATE_PACKAGE_NAME)

    def unarchive_resource(self, resource_id, extract_path):
        with tarfile.open(sync_resource(resource_id)) as tar:
            tar.extractall(path=extract_path)

    def prepare_report_renderer(self):
        self.unarchive_resource(self.report_renderer_resource_id, '.')

    def prepare_clickdaemon_keys(self):
        res = apihelpers.get_last_resource(resource_type=resource_types.CLICKDAEMON_KEYS)

        if not res:
            raise SandboxTaskFailureError('CLICKDAEMON_KEYS resource is missing')

        copy_path(sync_resource(res.id), './clickdaemon-keys.json')

    def _get_heap_snapshot_url(self):
        return 'http://localhost:{}/admin?action=heapsnapshot'.format(self.rr_admin_port)

    def _make_heap_snapshot(self):
        endpoint = self._get_heap_snapshot_url()
        logging.info('Using heap snapshot endpoint: %s', endpoint)

        result = requests.get(endpoint)
        logging.info('Heap snapshot request result: %s: %s', result.status_code, result.text)

    def start(self):
        environment = dict(os.environ)

        self.prepare_resources()

        make_folder(self.rr_logs)

        args = self.get_run_args()
        ynode = os.path.join(os.getcwd(), 'ynode/bin/ynode')
        cmd = ynode + ' ' + args
        if self.numa_node != -1:
            cmd = "numactl --cpunodebind={node} --membind={node} {ynode} {args}".format(
                node=self.numa_node,
                ynode=ynode,
                args=args)

        logging.info('Starting report-renderer with command %s' % cmd)

        if not self.alive():
            self.process = run_process(
                cmd.split(),
                outputs_to_one_file=False,
                wait=False,
                log_prefix='report-renderer_{}'.format(self.rr_port),
                environment=environment,
                work_dir=self.rr_templates_place,
            )

            logging.info(
                '===waiting for report-renderer to start {}:{}'.format(
                    socket.gethostname(),
                    self.rr_port))
            wait_searcher_start(
                '127.0.0.1',
                self.rr_admin_port,
                subproc_list=[self.process],
                timeout=5 * 60
            )

            logging.info('Waiting report-renderer to start workers')
            self._wait_for_all_workers()
            logging.info('Report-renderer has started all workers')

            if self.dump_start_end_heap_stats:
                self._make_heap_snapshot()

            if self.heap_dump_interval > 0:
                dumper_path = os.path.join(os.path.dirname(__file__), 'heap_dumper.py')
                logging.info("Starting heap dumper")
                self.heap_dumper_process = run_process(
                    [sys.executable, dumper_path, str(self._get_heap_snapshot_url()), str(self.heap_dump_interval)],
                    outputs_to_one_file=True,
                    log_prefix='heap_dumper',
                    wait=False)
                logging.info("Heap dumper has started")
            logging.info('Report-renderer has started')
        else:
            logging.warn('report-renderer already started')

    def _get_workers_count(self):
        try:
            data = json.loads(requests.get('http://localhost:{}/admin?action=stats'.format(self.rr_admin_port)).text)
        except Exception as e:
            msg = 'Report renderer could not start.\n\nRequest error:\n{}'.format(e)

            js_errors = self._save_js_errors()
            if js_errors:
                msg += (
                    '\n\nThere are JS errors (see js_errors.txt).\n'
                    "JS errors' tail:\n" + '\n'.join(js_errors[-3:])
                )

            raise SandboxTaskFailureError(msg)

        workers_total = 1000000000
        workers_ready = 0
        for (metric, value) in data:
            if metric == 'master.workers_total_ammv':
                workers_total = int(value)
            if metric == 'master.workers_ready_ammv':
                workers_ready = int(value)
        return workers_total, workers_ready

    def _wait_for_all_workers(self):
        wait_up_to = time.time() + MAX_WORKERS_START_TIME
        has_started = False
        while (not has_started) and (time.time() < wait_up_to):
            (total, ready) = self._get_workers_count()
            if ready < total:
                logging.debug('Started {} of {} workers'.format(ready, total))
                time.sleep(1)
            else:
                has_started = True
        if not has_started:
            raise SandboxTaskFailureError('Could not start all workers in {} seconds'.format(MAX_WORKERS_START_TIME))

    def _save_unistat(self):
        try:
            res = requests.get('http://localhost:{}/admin?action=stats'.format(self.rr_admin_port))
        except requests.ConnectionError as e:
            logging.error('Unable to request unistat from report-renderer.\n%s', e)
            raise

        with open(self.unistat_file_path, 'w') as fh:
            fh.write(res.text)

    def _create_unistat_resource(self):
        res = self.task.create_resource(
            self.unistat_file_path,
            self.unistat_file_path,
            resource_types.UNISTAT_RESPONSE
        )
        self.task.mark_resource_ready(res)

    def _create_heap_dump_resources(self):
        dumps = glob('**/*.heapsnapshot')
        if dumps:
            logging.info('Heap dumps are found: %s', dumps)
        else:
            logging.debug('No "heapsnapshot" files to save')
            return

        os.mkdir('heapsnapshots')
        for file in dumps:
            logging.debug('Adding {} to heapsnapshotы archive'.format(file))
            with tarfile.open('heapsnapshots/{}.tar.bz2'.format(file), 'w:bz2') as tar:
                tar.add(file)
        res = self.task.create_resource(
            'heapsnapshots',
            'heapsnapshots',
            resource_types.V8_COMPRESSED_HEAP_SNAPSHOT_DIR
        )
        self.task.mark_resource_ready(res)

    def _save_js_errors(self):
        errors = self._get_js_errors()
        if not errors:
            return

        with open('js_errors.txt', 'w') as fp:
            fp.writelines(errors)

        res = self.task.create_resource(
            'js_errors.txt',
            'js_errors.txt',
            rr_resources.REPORT_RENDERER_JS_ERRORS,
        )

        self.task.mark_resource_ready(res)

        return errors

    def _get_js_errors(self):
        try:
            with open('logs/current-report-renderer_debug-{}'.format(self.rr_port)) as fp:
                debug_log = (json.loads(line) for line in fp.readlines() if 'Error:' in line)
                return [event['message'] for event in debug_log]
        except Exception as e:
            logging.error("Can't receive JS errors: %s", e)

    def save_resources(self):
        # save application logs
        application_logs = [
            ('blockstat', resource_types.REPORT_RENDERER_BLOCKSTAT_LOG),
            ('renderer-request-event', rr_resources.REPORT_RENDERER_REQUEST_LOG),
            ('renderer-profile-event', rr_resources.REPORT_RENDERER_PROFILE_LOG),
            ('renderer-debug-event', rr_resources.REPORT_RENDERER_DEBUG_LOG),
            ('renderer-body-dump-event', rr_resources.REPORT_RENDERER_BODY_DUMP_LOG),
            ('common-failure-event', rr_resources.REPORT_RENDERER_COMMON_FAILURE_LOG),
        ]
        for log_name, log_type in application_logs:
            file_name = 'logs/current-report-renderer_{}-{}'.format(log_name, self.rr_port)
            # access log become request log in newer versions of report-renderer
            if os.path.isfile(file_name):
                res = self.task.create_resource(
                    'current-report-renderer_{}-{}.log'.format(log_name, self.rr_port),
                    file_name,
                    log_type,
                    attributes=self.log_attrs,
                )
                self.task.mark_resource_ready(res)

        self._create_heap_dump_resources()
        self._create_unistat_resource()

        # TODO: save v8 log
        # TODO: save profiling logs

    def stop(self):
        logging.info('stopping report-renderer')

        # Для получения unistat, нужен поход в ручку RR. Следовательно, он должен быть еще живой.
        self._save_unistat()

        if self.dump_start_end_heap_stats:
            self._make_heap_snapshot()

        if self.heap_dump_interval > 0:
            logging.info("Stopping heap dumper")
            self.heap_dumper_process.kill()
            self.heap_dumper_process = None
            logging.info("Heap dumper is stopped")

        if not self.alive():
            raise Exception('report-renderer is dead')
        else:
            requests.get('http://localhost:{}/admin?action=shutdown'.format(self.rr_admin_port))
            wait_searcher_stop(
                socket.gethostname(),
                self.rr_admin_port,
                timeout=2 * 60
            )
            self.process = None

    def __enter__(self):
        self.start()
        return self

    def __exit__(self, type, value, traceback):
        self.stop()
