# -*- coding: utf-8 -*-

import functools
import logging
import requests
from requests.packages.urllib3.exceptions import InsecureRequestWarning
from sandbox.projects.common import decorators


logger = logging.getLogger(__name__)


def disable_insecure_warnings():
    requests.packages.urllib3.disable_warnings(InsecureRequestWarning)


class ServerError(requests.HTTPError):
    pass


class EmptyResponseError(Exception):
    pass


NO_RETRY_CODES = [
    403,
    404,
]


def check_status_code(response):
    if response.status_code >= 500:
        raise ServerError(
            "Server Error [{}]: for url: {}. Reason: {}".format(response.status_code, response.url, response.content),
            response=response
        )
    elif response.status_code >= 400:
        raise requests.HTTPError(
            "Client Error [{}]: for url: {}. Reason: {}".format(response.status_code, response.url, response.content),
            response=response
        )


def get(url, log_request=True, log_bad_response=True, **kwargs):
    """
    Simplify `requests.get` method for every-day usage.

    - Add non-infinite timeout (see SEARCH-4429)
    - Disable certificate verification
    - Provide useful request/response logging

    @param url: request url
    @param log_request: dump request url into log (enabled by default)
    @param log_bad_response: dump response into log in case of 4xx or 5xx answers (enabled by default)
    """
    if log_request:
        logger.debug('[requests_wrapper][get] Requesting `%s`', url)
    response = functools.partial(requests.get, timeout=70, verify=False)(url, **kwargs)
    if log_bad_response and response is not None and response.status_code >= 400:
        logger.debug(
            '[requests_wrapper][get] HTTP ERROR %s on request `%s`:\n%s\n',
            response.status_code, url, response.content,
        )
    return response


@decorators.retries(max_tries=3, delay=5, backoff=3)
def get_r(
    url,
    log_request=True,
    log_bad_response=True,
    raise_for_empty_response=False,
    no_retry_on_http_codes=None,
    **kwargs  # noqa
):
    """
    Simplify `requests.get` method for every-day usage.

    - Add non-infinite timeout (see SEARCH-4429)
    - Disable certificate verification
    - Provide useful request/response logging
    - Embedded `raise_for_status` call
    - Retries included

    @param log_request: dump request url into log (enabled by default)
    @param log_bad_response: dump response into log in case of 4xx or 5xx answers (enabled by default)
    @param raise_for_empty_response: retry on empty response, raise Exception
        when retries limit exceeded. Added as a result of WEBREPORT-673 investigation.
    @param no_retry_on_http_codes: list of http codes that should NOT be retried
        (empty by default)
    @return: requests response
    """
    if no_retry_on_http_codes is None:
        no_retry_on_http_codes = []

    r = get(
        url=url,
        log_request=log_request,
        log_bad_response=log_bad_response,
        **kwargs  # noqa
    )

    raise_for_status = True
    if r.status_code in no_retry_on_http_codes:
        raise_for_status = False

    if raise_for_status:
        r.raise_for_status()

    if raise_for_empty_response and not r.text:
        raise EmptyResponseError("Empty response received. ")

    return r


def post(
    url,
    log_request=True,
    log_bad_response=True,
    raise_for_empty_response=False,
    timeout=40,
    **kwargs
):
    """
    Simplify `requests.post` method for every-day usage (SEARCH-4429).

    - Add non-infinite timeout (see SEARCH-4429)
    - Disable certificate verification
    - Provide useful request/response logging

    @param log_request: dump request url into log (enabled by default)
    @param log_bad_response: dump response into log in case of 4xx or 5xx answers (enabled by default)
    @param raise_for_empty_response: retry on empty response, raise Exception
        when retries limit exceeded. Added as a result of WEBREPORT-673 investigation.
    @return: requests response
    """
    if log_request:
        logger.debug('[requests_wrapper][post] Requesting %s', url)

    response = functools.partial(requests.post, timeout=timeout, verify=False)(url, **kwargs)
    if log_bad_response and response is not None and response.status_code >= 400:
        logger.debug(
            '[requests_wrapper][get] HTTP ERROR %s on request `%s`:\n%s\n',
            response.status_code, url, response.content,
        )

    if raise_for_empty_response and not response.text:
        raise EmptyResponseError("Empty response received. ")

    return response


@decorators.retries(max_tries=3, delay=5, backoff=3)
def post_r(
    url,
    log_request=True,
    log_bad_response=True,
    raise_for_empty_response=False,
    **kwargs
):
    """
    Simplify `requests.post` method for every-day usage.

    - Add non-infinite timeout (see SEARCH-4429)
    - Disable certificate verification
    - Provide useful request/response logging
    - Embedded `raise_for_status` call
    - Retries included

    Please be careful with POST requests retrying!

    @param log_request: dump request url into log (enabled by default)
    @param log_bad_response: dump response into log in case of 4xx or 5xx answers (enabled by default)
    @param raise_for_empty_response: retry on empty response, raise Exception
        when retries limit exceeded. Added as a result of WEBREPORT-673 investigation.
    @return: requests response
    """
    r = post(
        url=url,
        log_request=log_request,
        log_bad_response=log_bad_response,
        raise_for_empty_response=raise_for_empty_response,
        **kwargs  # noqa
    )
    r.raise_for_status()
    return r


def patch(url, **kwargs):
    return functools.partial(requests.patch, timeout=40, verify=False)(url, **kwargs)


@decorators.retries(max_tries=3, delay=5, backoff=3)
def patch_r(url, **kwargs):
    r = patch(url, **kwargs)
    r.raise_for_status()
    return r


@decorators.retries(max_tries=3, delay=5, backoff=3)
def put(url, **kwargs):
    """
    Simplify `requests.put` method for every-day usage.

    - Set default non-zero timeout (SEARCH-4429)
    - Disable certificate verification.

    @return request response
    """
    return functools.partial(requests.put, timeout=40, verify=False)(url, **kwargs)


def sget(session, url, **kwargs):
    """
    Simplify `session.get` method for every-day usage.

    - Set default non-zero timeout (SEARCH-4429)
    - Disable certificate verification.

    @return request response
    """
    return functools.partial(session.get, timeout=15, verify=False)(url, **kwargs)


@decorators.retries(max_tries=3, delay=5, backoff=3)
def sget_r(session, url, **kwargs):
    """
    Simplify `session.get` method for every-day usage.

    - Set default non-zero timeout (SEARCH-4429)
    - Disable certificate verification.
    - Embedded `raise_for_status` call
    - Retries included.

    @return request response
    """
    r = sget(session, url, **kwargs)
    r.raise_for_status()
    return r
