import logging
import re
from collections import defaultdict
from typing import (  # noqa: UnusedImport
    Any,
    AnyStr,
    Optional,
    Tuple,
)

from sandbox.common import rest
from sandbox.common.types import resource as ctr
from sandbox.common.types import task as ctt


LOGGER = logging.getLogger(__name__)
RELEASED_ATTR_NAME = "released"
RELEASED_TASK_STATUS = "RELEASED"
RELEASED_AS_RE = re.compile(r"Released as ([a-z]+)")


def by_max_attr_value(resource_type, attr_name, last_n=5):
    # type: (Any, AnyStr, int) -> Optional[int]
    """
    Get resource of specified resource type with max attribute value of specified attribute name.

    Get last_n resources of specified resource type and attribute name, sorted by descending id.
    Choose resource with max attribute value and return it as int.
    """
    result = None
    try:
        resources = rest.Client().resource.read(
            type=resource_type,
            state=ctr.State.READY,
            attr_name=attr_name,
            limit=last_n,
        )["items"]
        if resources:
            result = int(max(resources, key=lambda x: x["attributes"].get(attr_name))["id"])
    except LookupError:
        LOGGER.warning("Resource %s not found", resource_type)
    LOGGER.info("RESOURCE BY MAX ATTR VALUE RESULT = %s", result)
    return result


def by_last_released_task(resource_type, attrs=None, stage=ctt.ReleaseStatus.STABLE, last_n=100):
    # type: (Any, Optional[dict], ctt.ReleaseStatus, int) -> Tuple[Optional[int], str]
    """
    Get resource of specified resource type, released to specified stage most lately with its release time.

    Get last_n resources of specified resource type, released to all stages, sorted by descending id.
    Get tasks audit for these resources.
    Filter released to specific stage times
    Choose resource with max released task time and return it (as int) with time (as human readable str).
    """
    result = None, ""
    sb_rest_client = rest.Client()
    resources = sb_rest_client.resource.read(
        type=resource_type,
        status=ctr.State.READY,
        attr_name=RELEASED_ATTR_NAME,
        attrs=attrs,
        limit=last_n,
    )["items"]
    LOGGER.debug("Got resource candidates: %s", [(i["id"], i["attributes"][RELEASED_ATTR_NAME]) for i in resources])
    if not resources:
        LOGGER.warning("No released resource for %s with attrs %s", resource_type, attrs)
        return result
    # assume, that there are no same resources in one task
    task_resource_index = {i["task"]["id"]: int(i["id"]) for i in resources}
    audit = sb_rest_client.task.audit.read(id=list(task_resource_index.keys()))
    released_to = defaultdict(list)
    for audit_item in audit:
        if audit_item.get("status") == RELEASED_TASK_STATUS:
            stage_match = RELEASED_AS_RE.match(audit_item.get("message", ""))
            if stage_match:
                released_to[stage_match.group(1)].append((audit_item["task_id"], audit_item["time"]))
    LOGGER.debug("Got released statuses from audit: %s", released_to)
    cancelled_tasks = {
        task_id: time for task_id, time in
        sorted(released_to[str(ctt.ReleaseStatus.CANCELLED)], key=lambda x: x[1])
    }  # sort by time, because we need to store last cancelled release time for task
    for task_id, time in reversed(sorted(released_to[str(stage)], key=lambda x: x[1])):
        if task_id not in cancelled_tasks or time > cancelled_tasks[task_id]:
            result = task_resource_index[task_id], time
            break
    LOGGER.info("RESOURCE BY LAST RELEASED TASK = %s", result)
    return result
