# -*- coding: utf-8 -*-

import os
import stat
import tempfile
import subprocess

from sandbox.sandboxsdk import process


class Rsync(object):

    def __init__(self, user=None, password=None, password_file=None):
        if password and password_file:
            msg = "{} constructor can accept only one of arguments - password or password_file".format(self.__class__.__name__)
            raise RuntimeError(msg)

        self.user = user
        self.password_file = password_file
        # saves reference to tempfile to prevent the premature removal if password is presented
        self.temp_password_file = None
        if password:
            self.temp_password_file = tempfile.NamedTemporaryFile()
            self.temp_password_file.write(password)
            self.temp_password_file.flush()
            self.temp_password_file.seek(0)
            self.password_file = self.temp_password_file.name
            # rsync requires password file to be not other-accessible
            os.chmod(self.password_file, stat.S_IRUSR | stat.S_IRGRP)

    def _get_cmd(self, rsync_path, local_path="", args=[]):
        """
        Builds rsync command. Automatically append secret file (--password-file) to command and user's name
        if specified in constructor.

        :return: :class:`list` the sequence of arguments
        """
        cmd = ["rsync"]
        if self.password_file:
            cmd.extend(["--password-file", self.password_file])
        cmd.extend(args)
        # Set secret user for secret file
        rsync_path = rsync_path.replace("rsync://", "rsync://%s@" % self.user)
        cmd.append(rsync_path)
        if local_path:
            cmd.append(local_path)
        return cmd

    def get_file(self, rsync_path, local_path="./"):
        """
        Copies file from rsync_path to local_path via rsync.

        :param rsync_path: Rsync path to file in format rsync://SHARE_NAME/path_relative_to_the_share
        :param local_path: Path where the file will be copied
        :return: :class:`subprocess.Popen`
        """
        cmd = self._get_cmd(rsync_path, local_path, args=["-L"])
        return process.run_process(cmd, wait=True, log_prefix="rsync")

    def get_file_attr(self, rsync_path):
        """
        Get a list of files and their attributes (equivalent to ls).

        :param rsync_path: Rsync path to file or directory in format rsync://SHARE_NAME/path_relative_to_the_share
        :return: :class:`tuple` with (stdout, stderr) of completed rsync command
        """
        cmd = self._get_cmd(rsync_path, local_path="", args=["--list-only"])
        proc = process.run_process(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE, wait=False)
        return proc.communicate()
