# -*- coding: utf-8 -*-
"""
    Address Sanitizer wrapper for search components
"""

import re
import os
import logging
import traceback

from sandbox.sandboxsdk import svn
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import process
from sandbox.projects.common.search import components as sc


_ADDRESS_SANITIZER_OUTPUT = re.compile(r"AddressSanitizer Init done")
_MEMORY_SANITIZER_OUTPUT = re.compile(r"MemorySanitizer init done")
_THREAD_SANITIZER_OUTPUT = re.compile(r"^\*\*\*\*\* Running under ThreadSanitizer v2 \(pid \d+\) \*\*\*\*\*")


class SanitizerType(parameters.SandboxStringParameter):
    NONE = 'none'
    ADDRESS = 'address'
    MEMORY = 'memory'
    THREAD = 'thread'
    LEAK = 'leak'

    name = 'sanitizer_type'
    description = 'Sanitizer type'
    choices = [('None', NONE), ('Address', ADDRESS), ('Memory', MEMORY), ('Thread', THREAD), ('Leak', LEAK)]
    default_value = NONE


class SanEnvOptions(parameters.SandboxStringParameter):
    """Adds option to environment of corresponding sanitizer type"""
    name = "san_env_options"
    description = "Options for sanitizer environment"


SAN_OPTIONS = {
    SanitizerType.ADDRESS: "ASAN_OPTIONS",
    SanitizerType.MEMORY: "MSAN_OPTIONS",
    SanitizerType.THREAD: "TSAN_OPTIONS",
    SanitizerType.LEAK: "LSAN_OPTIONS",
}

TSAN_SUPPRESSIONS = {
    "middlesearch": "arcadia:/arc/trunk/arcadia/search/suppressions/meta/tsan_suppressions",
    "noapacheupper": "arcadia:/arc/trunk/arcadia/search/suppressions/upper/tsan_suppressions",
}


def get_sanitizer_environment(wrap_symbolizer=0):
    # set options for all sanitizers together
    # it does not matter because only one sanitizer type can be used during build
    env = {opt: "symbolize=1" for opt in SAN_OPTIONS.itervalues()}

    try:
        symbolizer = get_symbolizer(wrap_symbolizer)
        env.update({
            "ASAN_SYMBOLIZER_PATH": symbolizer,
            "MSAN_SYMBOLIZER_PATH": symbolizer,
            "TSAN_SYMBOLIZER_PATH": symbolizer,
            "LSAN_SYMBOLIZER_PATH": symbolizer,
        })
        # see VIDEOPOISK-8275
        external_path = " external_symbolizer_path={}".format(symbolizer)
        env["ASAN_OPTIONS"] += external_path
        env["MSAN_OPTIONS"] += external_path
        env["TSAN_OPTIONS"] += external_path
        env["LSAN_OPTIONS"] += external_path
    except Exception:
        logging.error("Failed to obtain llvm-symbolizer:\n%s", traceback.format_exc())

    return env


def get_symbolizer(wrap_symbolizer=0):
    # this code will download compiler, but it's quite fast (less than a minute)
    dir_name = 'ya_get_symbolizer'
    if not os.path.isdir(dir_name):
        svn.Arcadia.checkout('arcadia:/arc/trunk/arcadia/', dir_name, depth="files")
    symbolizer_retrieve_cmd = [os.path.join(dir_name, 'ya'), 'tool', 'c++', '--print-path']
    p = process.run_process(symbolizer_retrieve_cmd, outs_to_pipe=True)
    output, _ = p.communicate()

    symbolizer_path = os.path.join(os.path.dirname(output.strip()), 'llvm-symbolizer')
    if (wrap_symbolizer):
        new_symbolizer_path = os.path.join(os.path.abspath(dir_name), "llvm-symbolizer")
        with open(new_symbolizer_path, "w") as s:
             s.write("#!/bin/bash\n")
             s.write("{} 2>/dev/null\n".format(symbolizer_path))
        os.chmod(new_symbolizer_path, 0o775)
        return new_symbolizer_path

    return symbolizer_path


def add_tsan_suppressions(env, component_name):
    try:
        if component_name not in TSAN_SUPPRESSIONS:
            return
        path_to_supp = os.path.abspath("tsan_suppressions")
        svn.Arcadia.export(TSAN_SUPPRESSIONS[component_name], path_to_supp)
        env["TSAN_OPTIONS"] += " suppressions={}".format(path_to_supp)
    except Exception:
        logging.error("Failed to add tsan suppressions:\n%s", traceback.format_exc())


def generate_task(
    params_collection,
    base_class,
    start_timeout=sc.DEFAULT_START_TIMEOUT * 2,
    shutdown_timeout=sc.DEFAULT_START_TIMEOUT * 2,
):
    """
        Generates task class based on parent task class
        to run search component built using Address/Memory/Thread Sanitizer insrumentation.
        Base class should have stub method init_search_component(self, component)
    """

    class SanitizerTaskPattern(base_class):
        type = None

        input_parameters = [
            SanitizerType,
            SanEnvOptions,
        ] + sc.tune_search_params(
            params_collection,
            base_class.input_parameters,
            start_timeout=start_timeout,
        )

        def init_search_component(self, component, wrap_symbolizer=0):
            base_class.init_search_component(self, component)

            component.set_cgroup(None)  # Msan uses additional memory, this can lead to OOM
            if component.name == "basesearch":
                component.replace_config_parameter("Collection/UserParams/MemoryMappedArchive", None)
                component.replace_config_parameter("Collection/UserParams/ArchiveOpenMode", None)
                component.replace_config_parameter("Collection/CalculateBinaryMD5", 'no')
                component.replace_config_parameter("Collection/MemSentryOptions", None)

            sanitizer_type = self.ctx.get(SanitizerType.name)
            self._configure_sanitizer_environment(component, sanitizer_type, wrap_symbolizer)

            if sanitizer_type == SanitizerType.ADDRESS:
                self.__verify_sanitizer(component, _ADDRESS_SANITIZER_OUTPUT, "ASAN_OPTIONS")
                self.__verify_initialization(component)
            elif sanitizer_type == SanitizerType.MEMORY:
                self.__verify_sanitizer(component, _MEMORY_SANITIZER_OUTPUT, "MSAN_OPTIONS")
            elif sanitizer_type == SanitizerType.THREAD:
                self.__verify_sanitizer(component, _THREAD_SANITIZER_OUTPUT, "TSAN_OPTIONS")

        def _configure_sanitizer_environment(self, component, sanitizer_type, wrap_symbolizer=0):
            env = component.get_environment()
            env.update(get_sanitizer_environment(wrap_symbolizer))
            add_tsan_suppressions(env, component.name)
            self._add_custom_env_options(env, sanitizer_type)
            component.set_environment(env)
            if sanitizer_type != SanitizerType.NONE:
                component.sanitize_type = sanitizer_type

        def _add_custom_env_options(self, env, san_type):
            custom_env_opt = self.ctx.get(SanEnvOptions.name)
            if custom_env_opt:
                env[SAN_OPTIONS[san_type]] += " " + custom_env_opt.lstrip()

        def __verify_sanitizer(self, component, greeting_pattern, environment_parameter):
            """
                Start basesearch with extra verbosity and search for sanitizer activity in output
            """
            logging.info("Verifying sanitizer type")
            basesearch_binary = self.sync_resource(self.ctx[params_collection.Binary.name])
            basesearch_proc = process.run_process(
                [basesearch_binary, "-v"],
                environment={
                    environment_parameter: "verbosity=1",
                },
                # stderr=subprocess.PIPE,
                wait=False,
                outputs_to_one_file=False,
                log_prefix=component.name,
            )
            basesearch_proc.wait()  # ignore return code at this stage
            with open(basesearch_proc.stderr_path) as err_file:
                for line in err_file:
                    if greeting_pattern.search(line):
                        logging.info("Sanitizer successfully verified")
                        return
                else:
                    raise Exception("Please verify that basesearch was compiled with correct sanitizer")

        def __verify_initialization(self, component):
            """
                Start basesearch to verify against global initialization fiasco
            """

            environment = component.get_environment()
            asan_options = environment["ASAN_OPTIONS"]
            try:
                environment["ASAN_OPTIONS"] = "{}:check_initialization_order=1".format(asan_options)
                with component:
                    pass  # Wait until initialization done
            finally:
                environment["ASAN_OPTIONS"] = asan_options

    return SanitizerTaskPattern
