# -*- coding: utf-8 -*-
"""
    Here are some useful function for release-machine tasks
"""
from sandbox.common import errors as sb_errors
from sandbox.common import rest as sb_rest
from sandbox import sdk2
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import sandboxapi
from sandbox.projects.common import decorators


def is_sdk2(task):
    return isinstance(task, sdk2.Task)


def vault_data(owner, name):
    data = sdk2.Vault.data(owner, name)
    if not data:
        raise sb_errors.TaskError("Failed to get vault data from {} owned by {}".format(name, owner))
    return data


def ctx_field(task, key, default=None):
    if is_sdk2(task):
        if key not in task.Context:
            return default
        return getattr(task.Context, key)
    else:
        return task.ctx.get(key, default)


def ctx_field_set(task, key, value):
    if is_sdk2(task):
        setattr(task.Context, key, value)
    else:
        task.ctx[key] = value


def input_field(task, key, default=None):
    if is_sdk2(task):
        return getattr(task.Parameters, key, default)
    else:
        return task.ctx.get(key, default)


def input_or_ctx_field(task, key, default=None):
    """ RMDEV-139 """
    task_input_field = input_field(task, key, default)
    if task_input_field == default:
        return ctx_field(task, key, default)
    else:
        return task_input_field


def get_task_description(task):
    if is_sdk2(task):
        return task.Parameters.description
    else:
        return task.description


def task_obj(task_id, reload=False):
    """
    Get Sandbox task object.
    There were incidents because of sdk2-task wasn't reloaded: RMINCIDENTS-183, RMINCIDENTS-348
    :param task_id: Sandbox task id
    :param reload: has no effect for sdk1 tasks, need for sdk2 tasks to drop cache
    :return: Sandbox task object
    """
    try:
        return channel.sandbox.get_task(task_id)
    except Exception:
        if reload:
            return sdk2.Task[task_id].reload()
        return sdk2.Task[task_id]


def resource_obj(res_id):
    try:
        return channel.sandbox.get_resource(res_id)
    except Exception:
        return sdk2.Resource[res_id]


def get_resource_attr(resource, attr, default=None):
    if isinstance(resource, sandboxapi.SandboxResource):
        return resource.attributes.get(attr, default)
    else:
        return getattr(resource, attr, default)


def resource_sync_path(res):
    try:
        return channel.task.sync_resource(res.id)
    except Exception:
        return str(sdk2.ResourceData(res).path)


@decorators.retries(3)
def task_status(task_id):
    """
    RMINCIDENTS-183
    We can't use there task_obj because of sandbox status convertation
    (from Sandbox email):
    > task_obj для SDK1-задач она возвращает один объект, для SDK2 — совсем другой.
    > Вот класс, объект которого она возвращает для SDK1:
    > https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/sandboxsdk/sandboxapi.py#L284
    > Статус в нём конвертируется в "старый" (вероятно, когда-то в Sandbox не было статуса RELEASED)
    > In [18]: ctt.Status.old_status('RELEASED')
    > Out[18]: 'FINISHED'
    :param task_id:
    :return: new status for task
    """
    api = sb_rest.Client()
    return api.task[task_id][:]["status"]


def convert_parameter_to_sdk2(param):
    """
    This function converts sdk1 parameter to sdk2
    Not every sdk1 class can be converted, here the list of supported base types:
    -- SandboxStringParameter
    -- SandboxBoolParameter
    -- SandboxIntegerParameter
    -- SandboxFloatParameter
    -- ResourceSelector

    Note that it supports only basic options:
    -- description
    -- long description (text in class docs)
    -- default_value
    -- required
    -- do_not_copy

    And some class related options:
    ResourceSelector:
    -- resource_type
    -- state
    -- multiple

    To make grouping, choices, hiding and other parameter logic use sdk2 statements

    If you need to convert some unmentioned parameter you can add your own handling

    :param param: sdk1 parameter class
    :return: sdk2 parameter
    """
    base_options = {
        "label": param.description,
        "default": param.default_value,
        "required": param.required,
        "do_not_copy": param.do_not_copy,
    }
    if isinstance(param.__doc__, str):
        base_options["description"] = param.__doc__

    if issubclass(param, parameters.SandboxStringParameter):
        return sdk2.parameters.String(**base_options)
    elif issubclass(param, parameters.SandboxBoolParameter):
        return sdk2.parameters.Bool(**base_options)
    elif issubclass(param, parameters.SandboxIntegerParameter):
        return sdk2.parameters.Integer(**base_options)
    elif issubclass(param, parameters.SandboxFloatParameter):
        return sdk2.parameters.Float(**base_options)
    elif issubclass(param, parameters.ResourceSelector):
        base_options.update({
            "state": param.state,
            "resource_type": param.resource_type,
            "multiple": param.multiple,
        })
        sdk2.parameters.Resource
        return sdk2.parameters.Resource(**base_options)
    elif issubclass(param, parameters.LastReleasedResource):
        base_options.update({
            "state": param.state,
            "resource_type": param.resource_type,
            "multiple": param.multiple,
        })
        sdk2.parameters.Resource
        return sdk2.parameters.LastReleasedResource(**base_options)
    raise TypeError("Can't convert {} parameter to sdk2".format(param))
