# -*- coding: utf-8 -*-

import time
import logging
import urllib2
import copy
import os.path
from six.moves import range

from sandbox import common
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

from sandbox.projects import resource_types
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import utils
from sandbox.projects.common.search import config as sconf
from sandbox.projects.common.search import components as sc


class SuperMindModeParameter(parameters.SandboxRadioParameter):
    required = True
    choices = [(x, x) for x in ('mind', 'auto')]
    sub_fields = {
        'auto': ['auto_supermind_mode'],
        'mind': ['mind_supermind_mode'],
    }
    name = 'super_mind_mode'
    description = 'SuperMind mode'
    group = 'SuperMind Parameters'
    default_value = 'mind'


class FailTaskParameter(parameters.SandboxBoolParameter):
    name = 'fail_task_low_improvement'
    description = 'Fail task if improvement is smaller than expected'
    group = 'SuperMind Parameters'
    default_value = False


class MinImprovementParameter(parameters.SandboxFloatParameter):
    name = 'min_improvement'
    description = 'Minimum expected improvement'
    group = 'SuperMind Parameters'
    default_value = 20.0


class MultParameter(parameters.SandboxFloatParameter):
    name = 'super_mind_mult'
    description = 'SuperMind mult (mind mode)'
    group = 'SuperMind Parameters'
    default_value = 0.5


class DegradationThreshold1(parameters.SandboxIntegerParameter):
    name = 'degradation_threshold_1'
    description = 'First degradation threshold (auto mode)'
    group = 'SuperMind Parameters'
    default_value = 30


class DegradationThreshold2(parameters.SandboxIntegerParameter):
    name = 'degradation_threshold_2'
    description = 'Second degradation threshold (auto mode)'
    group = 'SuperMind Parameters'
    default_value = 60


class EnableSuperMindParameter(parameters.SandboxBoolParameter):
    name = 'enable_super_mind'
    description = 'Enable Supermind'
    default_value = False
    group = 'SuperMind Parameters'
    sub_fields = {
        'true': [
            SuperMindModeParameter.name,
            MultParameter.name,
            DegradationThreshold1.name,
            DegradationThreshold2.name
        ]
    }


class DisableSuperMindSharedFile(parameters.SandboxBoolParameter):
    name = "disable_super_mind_shared_file"
    description = "Disable SuperMindSharedFile"
    default_value = False
    group = "SuperMind Parameters"


class BaseTestSuperMindTask(SandboxTask):

    """
        Проверяет, что ручка Supermind управления производительностью базового поиска работает
        и поворот ручки в нужную сторону приводит к увеличению RPS (качество при этом проседает,
        но это в данном случае не так важно).

        В режиме mind задаётся значение множителя и вычисляется разница между RPS в процентах
        между двумя запусками TEST_BASESEARCH_PERFORMANCE_BEST.

        В режиме auto тестируется автоматическая деградация при повышении нагрузки.

        Сделано в рамках SEARCH-628
    """

    AUTO_MODE = 'auto'
    MIND_MODE = 'mind'

    MODES = (
        AUTO_MODE,
        MIND_MODE
    )

    input_parameters = (
        SuperMindModeParameter,
        FailTaskParameter,
        MinImprovementParameter,
        MultParameter,
        DegradationThreshold1,
        DegradationThreshold2,
    )

    def get_last_error_trace(self):
        # TODO: remove it after complete migration to new interface
        pass

    def _create_performance_subtask(self, subtask_num, conf_id):
        subtask_key = 'perf_task_id{}'.format(subtask_num)
        perf_task_id = self.ctx.get(subtask_key)
        if perf_task_id:
            logging.info("Performance subtask #%s already created: %s", subtask_num, perf_task_id)
            return
        sub_ctx = copy.deepcopy(self.ctx)
        sub_ctx['notify_via'] = ''
        sub_ctx['notify_if_failed'] = self.owner
        sub_ctx[sc.DefaultBasesearchParams.Config.name] = conf_id
        self.ctx[subtask_key] = self.create_subtask(
            task_type=self._get_performance_task_type(),
            input_parameters=sub_ctx,
            description="{} #{}".format(self.descr, subtask_num),
        ).id

    def _get_patched_confs(self, mode):
        if mode == self.MIND_MODE:
            return (
                self._patch_conf(),
                self._patch_conf({
                    'Collection/SuperMindMode': 'mind',
                    'Collection/SuperMindMult': utils.get_or_default(self.ctx, MultParameter),
                }),
            )
        elif mode == self.AUTO_MODE:
            return (
                self._patch_conf({
                    'Collection/SuperMindMode': 'auto',
                    'Collection/SuperMindDegradationThreshold': utils.get_or_default(self.ctx, DegradationThreshold1),
                }),
                self._patch_conf({
                    'Collection/SuperMindMode': 'auto',
                    'Collection/SuperMindDegradationThreshold': utils.get_or_default(self.ctx, DegradationThreshold2),
                }),
            )
        else:
            eh.check_failed("Unknown mode = {}".format(mode))

    def _patch_conf(self, patcher=None):
        config_id = self.ctx[sc.DefaultBasesearchParams.Config.name]
        if patcher:
            config_path = self.sync_resource(config_id)
            config = sconf.BasesearchWebConfig.get_config_from_file(config_path)
            config.apply_local_patch(patcher)
            config_patched_path = paths.get_unique_file_name(self.abs_path(), "basesearch_supermind.cfg")
            config.save_to_file(config_patched_path)
            config_patched = self.create_resource(self.descr, config_patched_path, resource_types.SEARCH_CONFIG)
            config_id = config_patched.id
            self.mark_resource_ready(config_id)
        return config_id

    def on_execute(self):
        mode = utils.get_or_default(self.ctx, SuperMindModeParameter)
        for i, conf_id in enumerate(self._get_patched_confs(mode), start=1):
            self._create_performance_subtask(i, conf_id)
        utils.wait_all_subtasks_stop()
        if utils.check_all_subtasks_done():
            utils.check_subtasks_fails()
            self._collect_all_results()
        else:
            utils.restart_broken_subtasks()

    def _collect_all_results(self):
        do_fail = utils.get_or_default(self.ctx, FailTaskParameter)
        min_improvement = utils.get_or_default(self.ctx, MinImprovementParameter)

        requests1 = channel.sandbox.get_task(self.ctx['perf_task_id1']).ctx['requests_per_sec']
        requests2 = channel.sandbox.get_task(self.ctx['perf_task_id2']).ctx['requests_per_sec']
        improvement = (max(requests2) / max(requests1) - 1) * 100
        self.ctx['improvement'] = round(improvement, 2)
        if do_fail and improvement < min_improvement:
            raise SandboxTaskFailureError('Resulting RPS improvement is smaller than expected')

        self.ctx['requests_per_sec'] = sorted([max(requests1), max(requests2)], reverse=True)
        self.ctx['requests_per_sec_2d'] = [requests1, requests2]

    # TODO: этот метод надо выпилить в common
    def get_all_subtasks_data(self):
        """
            Получить список rps'ов  с подсвеченными максимумами
            Вызывается из view таска
            :return: список с результатами в виде кортежа (task_id, requests_per_sec)
                где task_id - идентификатор подзадачи, где был выполнен обстрел
                    requests_per_sec - список результатов с отмеченными максимумами

                Например:
                (2306716, [224.19499999999999, 1008.575, 975.18299999999999, 976.85000000000002,
                           1018.308, 1022.904, 995.58500000000004, 989.32000000000005,
                           "<b style='color: red'>1040.519</b>", 1028.828]),
                 2306714, [229.98400000000001, 986.17899999999997, 1025.5599999999999,
                           973.57899999999995, 922.52800000000002, 965.87699999999995,
                           '<b>1029.954</b>',
                           1026.046, 1017.337, 976.06700000000001],
                 ...)
        """
        res = []
        child_tasks = getattr(self, 'child_tasks', None)
        if child_tasks is not None:
            # оптимизация для случая когда view вызывается из родительского таска PRIEMKA_BASESEARCH_BINARY
            # в этом случае task имеет свойство child_tasks, заполненное в get_child_tree (yasandbox/manager.py)
            child_tasks = [task for _, task in child_tasks]
        else:
            child_tasks = self.list_subtasks(load=True, completed_only=True)

        for st in child_tasks:
            # упавшие таски пропускаем
            if 'requests_per_sec' not in st.ctx:
                continue
            requests_per_sec = st.ctx['requests_per_sec']

            res.append((st.id, requests_per_sec, requests_per_sec.index(max(requests_per_sec))))

        res = sorted(res, key=lambda x: x[1][x[2]], reverse=True)

        for i in range(len(res)):
            res[i][1][res[i][2]] = ("<b style='color: red'>%s</b>" if i == 0 else "<b>%s</b>") % res[i][1][res[i][2]]

        return [(task_id, _requests_per_sec) for task_id, _requests_per_sec, index_of_max_value in res]

    def get_short_task_result(self):
        improvement = self.ctx.get('improvement', 0)
        if self.is_completed():
            sign = "+" if improvement > 0 else ""
            return "{}{}%".format(sign, improvement)

        return None

    def _get_performance_task_type(self):
        """
            Возвращается имя теста проверки производительности
        """

        raise NotImplementedError()

    @property
    def footer(self):
        all_subtasks = self.get_all_subtasks_data()
        improvement = self.ctx.get('improvement')
        if not (all_subtasks and improvement):
            return None

        min_improvement = self.ctx.get('min_improvement')
        if improvement > min_improvement:
            result = '<h4>{}% > {}%</h4>'.format(improvement, min_improvement)
        else:
            result = '<h4>{}% < {}%</h4>'.format(improvement, min_improvement)

        return [
            {
                'helperName': '',
                'content': {result: self._performance_rps_list}
            },
        ]

    @property
    def _performance_rps_list(self):
        client = common.rest.Client()
        info = client.task.read(
            parent=self.id,
            fields='context.requests_per_sec',
            limit=2,
            hidden=True,
        )
        rps_list = []
        round_to = 1
        max_rps = round(max(self.ctx.get("requests_per_sec", [0])), round_to)
        for rps in info.get('items'):
            req_per_sec = rps.get("context.requests_per_sec") or [0] * self.ctx.get("dolbilka_executor_sessions", 1)
            req_per_sec = [round(i, round_to) for i in req_per_sec]
            max_rps_in_row = max(req_per_sec)
            colored = ' style="color:red"' if max_rps in req_per_sec else ""
            req_per_sec[req_per_sec.index(max_rps_in_row)] = "<b{}>{}</b>".format(colored, max_rps_in_row)
            rps_dict = dict(enumerate(req_per_sec))
            rps_dict['Task id'] = lb.task_link(rps.get("id", "-"))
            rps_list.append(rps_dict)
        return rps_list


class BasesearchWebWithSuperMind(sc.BasesearchWeb):
    """
        Вариант basesearch, который дёргает ручку производительности
        после старта (операция wait).
        Ручка дергается через API, а не через конфигурационный файл,
        потому что именно так она и применяется в production
    """

    def __init__(self, supermind_mode, supermind_multiplier, *args, **kwargs):
        super(BasesearchWebWithSuperMind, self).__init__(*args, **kwargs)
        self.supermind_mode = supermind_mode
        self.supermind_multiplier = supermind_multiplier

    def wait(self, timeout=None):
        super(BasesearchWebWithSuperMind, self).wait(timeout)
        self.set_degrade()

    def set_supermind(self, mode=None, mult=None):
        """
            Old variant on production to change supermind value

            Usage:
                self.set_supermind(mode=self.supermind_mode)
                self.set_supermind(mult=self.supermind_multiplier)
        """
        try:
            url = "http://127.0.0.1:{0}/supermind?".format(self.port)
            if mode is not None:
                url += "action=setmode&mode={0}".format(mode)
            elif mult is not None:
                url += "action=setmult&mult={0}".format(mult)
            logging.info("Fetch url %s" % url)
            urllib2.urlopen(url, timeout=20).read()
        except Exception as e:
            raise SandboxTaskFailureError("Failed to execute supermind request: {}".format(str(e)))

    def set_degrade(self):
        """
            Current variant on production to change supermind value
        """

        emergency_file = os.path.join(self.work_dir, self._EMERGENCY_FILENAME)

        emergency_args = []
        if self.supermind_mode is not None:
            emergency_args.append("pron=sm_{}".format(self.supermind_mode))

        if self.supermind_multiplier is not None:
            emergency_args.append("pron=smm_{}".format(self.supermind_multiplier))

        if emergency_args:
            logging.info("Using emergency file %s...", emergency_file)
            with open(emergency_file, "w") as f:
                emergency_content = "&".join(emergency_args) + "\n"
                logging.info("Emergency file content:'{}'".format(emergency_content))
                f.write(emergency_content)

        time.sleep(self._EMERGENCY_CHECK_PERIOD + 1)


def supermind_component_creator(supermind_mode, supermind_multiplier):
    def wrapper(*args, **kwargs):
        return BasesearchWebWithSuperMind(
            supermind_mode,
            supermind_multiplier,
            *args,
            **kwargs
        )
    return wrapper
