import logging
import os.path
import re

import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import task

from sandbox.projects.common.search import BaseTestSuperMindTask as supermind_task
from sandbox.projects.common import utils
from sandbox.projects.common.search import components as search_components


BASESEARCH_PARAMS_GROUP = 'Basesearch parameters (advanced)'


class RamdriveSizeParameter(parameters.SandboxIntegerParameter):
    name = 'ramdrive_size'
    description = 'Ramdrive size (GB)'
    group = BASESEARCH_PARAMS_GROUP
    default_value = 17


class RamdriveFilesParameter(parameters.SandboxStringParameter):
    name = 'ramdrive_files'
    description = 'Index files in Ramdrive (space separated regex, e.g. indexdir indexarc.*)'
    group = BASESEARCH_PARAMS_GROUP
    required = False


class RamdriveEnabledParameter(parameters.SandboxBoolParameter):
    name = 'ramdrive_enabled'
    description = 'Store database in RAM'
    group = BASESEARCH_PARAMS_GROUP
    required = False
    default_value = False
    sub_fields = {
        'true': (RamdriveSizeParameter.name, RamdriveFilesParameter.name)
    }


class DisableMD5Calculations(parameters.SandboxBoolParameter):
    name = 'disable_md5_calculations'
    description = 'Disable md5 calc for binary/models'
    default_value = True


class BasesearchComponentTask(task.SandboxTask):
    """
        Mixin class with common routines for basesearch
    """

    client_tags = ctc.Tag.LINUX_PRECISE

    basesearch_common_parameters = (
        RamdriveEnabledParameter,
        RamdriveSizeParameter,
        RamdriveFilesParameter,
        DisableMD5Calculations,
    )
    basesearch_supermind_parameters = (
        supermind_task.EnableSuperMindParameter,
        supermind_task.SuperMindModeParameter,
        supermind_task.MultParameter,
        supermind_task.DisableSuperMindSharedFile
    )
    basesearch_input_parameters = basesearch_common_parameters + basesearch_supermind_parameters

    def on_enqueue(self):
        task.SandboxTask.on_enqueue(self)
        if self.ctx[RamdriveEnabledParameter.name]:
            self.ramdrive = self.RamDrive(
                ctm.RamDriveType.TMPFS,
                self.ctx[RamdriveSizeParameter.name] * 1024,
                None
            )

    def on_execute(self):
        pass

    def _component(self, params, *args, **kwargs):
        return search_components.get_basesearch(params=params, *args, **kwargs)

    def _basesearch(self, basesearch_params, run_count=None):
        component_creator = None
        if utils.get_or_default(self.ctx, supermind_task.EnableSuperMindParameter):
            component_creator = supermind_task.supermind_component_creator(
                supermind_mode=self.ctx.get(supermind_task.SuperMindModeParameter.name),
                supermind_multiplier=self.ctx.get(supermind_task.MultParameter.name)
            )

        basesearch = self._component(
            params=basesearch_params,
            component_creator=component_creator,
            run_count=run_count,
        )

        if self.ctx[RamdriveEnabledParameter.name]:
            logging.info("Using ramdrive {} of size {}".format(self.ramdrive.path, self.ramdrive.size))
            ram_database_path = os.path.join(self.ramdrive.path, "database")
            ram_database_files = tuple(self.ctx[RamdriveFilesParameter.name].split())
            old_database_path = self.sync_resource(self.ctx[basesearch_params.Database.name])

            paths.make_folder(ram_database_path, delete_content=True)
            for database_file in os.listdir(old_database_path):
                old_file = os.path.join(old_database_path, database_file)
                ram_file = os.path.join(ram_database_path, database_file)

                if filter(lambda ram_database_file: re.match(ram_database_file, database_file), ram_database_files):
                    paths.copy_path(old_file, ram_file)
                else:
                    os.symlink(old_file, ram_file)
            basesearch.replace_config_parameter("Collection/IndexDir", ram_database_path)
            basesearch.replace_config_parameter("Collection/UseArchiveFromMemory", 'True')

        if utils.get_or_default(self.ctx, DisableMD5Calculations):
            # get rid of long calculations that may affect performance
            basesearch.replace_config_parameter("Collection/CalculateBinaryMD5", 'no')
            basesearch.replace_config_parameter("Collection/CalculateModelsMD5", 'no')

        if utils.get_or_default(self.ctx, supermind_task.DisableSuperMindSharedFile):
            basesearch.replace_config_parameter("Collection/SuperMindSharedFile", "None")

        self.init_search_component(basesearch)
        return basesearch

    # standard hook for massiff and sanitizer task generators
    def init_search_component(self, component):
        pass
