# -*- coding: utf-8 -*-

import re
import urllib
import logging

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.channel import channel

from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import string
from sandbox.projects.common import utils2
from sandbox.projects.common.search import bugbanner2
from sandbox.projects import resource_types

_BUG_REPORT_INIT_FIELD = "_showed_bug_banners"
_BUGBANNER_KEY = 'bugbanner'

Banners = bugbanner2.Banners


class BugBanner(object):
    """
        Show ads about bug reporting in task info (draw ST issue creation links)
    """

    @staticmethod
    def show_ad(
        component,
        st_component_ids=None,
        responsibles=None,
        queue='SEARCH',
    ):
        _show_ad(
            task=channel.task,
            component=component,
            st_component_ids=st_component_ids,
            responsibles=responsibles,
            queue=queue,
        )


class BugBannerTask(SandboxTask):
    """
        Task mixin class for showing ads about bug reporting in task info
        (draw ST issue creation links).
    """

    def configure_bugbanner(self, component, st_component_ids=None, responsibles=None, queue='SEARCH'):
        """
            For internal use only. Consider use add_bugbanner instead.
        """
        if _BUGBANNER_KEY not in self.ctx:
            self.ctx[_BUGBANNER_KEY] = {}

        self.ctx[_BUGBANNER_KEY][component] = {
            'component': component,
            'st_component_ids': st_component_ids,
            'responsibles': responsibles,
            'queue': queue,
        }

    def add_bugbanner(self, banner_class, component=None, add_responsibles=list()):
        if component is None:
            component = banner_class.component

        self.configure_bugbanner(
            component=component,
            st_component_ids=banner_class.st_component_ids,
            responsibles=banner_class.responsibles + add_responsibles,
            queue=banner_class.queue,
        )

    def _show_ad(self, update=False):
        bugbanner_cfgs = self.ctx.get(_BUGBANNER_KEY)
        if not bugbanner_cfgs:
            logging.error('BugBanner is not configured.')
            return

        for component, bugbanner_cfg in bugbanner_cfgs.iteritems():
            if "component" not in bugbanner_cfg:
                # old style bugbanner, skip
                continue

            _show_ad(
                task=self,
                component=bugbanner_cfg['component'],
                st_component_ids=bugbanner_cfg['st_component_ids'],
                responsibles=bugbanner_cfg['responsibles'],
                queue=bugbanner_cfg['queue'],
                update=update,
            )

    def on_success(self):
        logging.debug("on_success handler executed")
        self._show_ad(update=True)
        SandboxTask.on_success(self)

    def on_failure(self):
        logging.debug("on_failure handler executed")
        self._show_ad(update=True)
        SandboxTask.on_failure(self)

    def on_break(self):
        logging.debug("on_break handler executed")
        self._show_ad(update=True)
        SandboxTask.on_break(self)


def _get_task_description(task):
    descr = string.all_to_str(task.descr)
    # SEARCH-1934
    descr = re.sub(r'<a href="([^" ]+)">(.+?)</a>', ' \\1 (\\2) ', descr)

    description = (
        'Task: {task_link}\n'
        '**{task.status}** in task {task.type}\n'
        '\n'
        'Task description:\n{description}\n'
    ).format(
        task=task,
        task_link=lb.task_link(task.id, plain=True),
        description=descr,
    )

    task_resources = task.list_resources()
    logging.debug("Task resources: %s", task_resources)
    for resource in task_resources:
        if resource.type == resource_types.CORE_DUMP:
            description += 'Core dump: {}\n'.format(
                utils2.resource_redirect_url(resource.id),
            )
        if resource.type == resource_types.FILTERED_GDB_TRACEBACK:
            description += 'Filtered traceback: {}\n'.format(
                utils2.resource_redirect_url(resource.id),
            )

    return description


def _show_ad(task, component, st_component_ids, responsibles, queue, update=False):
    initialized = task.ctx.get(_BUG_REPORT_INIT_FIELD, [])

    # do not show ads twice
    if component in initialized:
        if not update:
            return
    else:
        initialized.append(component)
        task.ctx[_BUG_REPORT_INIT_FIELD] = initialized

    params = [
        ('type', '1'),
        ('priority', '2'),
        ('queue', queue),
        ('description', _get_task_description(task)),
    ]

    if responsibles:
        params.append(('assignee', responsibles[0]))
        for responsible in responsibles:
            params.append(('followers', responsible))

    for st_component_id in st_component_ids or []:
        params.append(('components', str(st_component_id)))

    bug_report_link = "https://st.yandex-team.ru/createTicket?" + urllib.urlencode(params)
    bug_report_html = (
        '{update}You are using <b>{component}</b>. '
        'On any problems feel free to <a href="{link}">report {component} issue</a>'.format(
            update='<b>(Updated)</b> ' if update else '',
            component=component,
            link=bug_report_link,
        )
    )
    task.set_info(bug_report_html, do_escape=False)


def sandbox_NNNN(task):
    """
    Please do not delete this sample function, it definitely
    will be used many times in the future.
    :param task: sandbox task
    """
    key = "sandbox_NNNN"
    if not task.ctx.get(key):
        task.set_info(
            "Something is wrong with Sandbox, see "
            "<a href=\"http://st.yandex-team.ru/SANDBOX-NNNN\">SANDBOX-NNNN</a>. "
            "Please vote for this bug. ",
            do_escape=False,
        )
        task.ctx[key] = True
