# -*- coding: utf-8 -*-

import re
import logging
from six.moves.urllib import parse

from sandbox import sdk2

from sandbox.projects import resource_types
from sandbox.projects.common import string
from sandbox.projects.common import link_builder
from sandbox.projects.common import utils2


class Banners(object):

    class BuggyComponent(object):

        # default ST component list is empty
        st_component_ids = []
        # default queue
        queue = "SEARCH"
        # TODO(mvel): make `component` compatible
        # with release machine components names
        case_description = 'On any problems'

    class ReleaseMachine(BuggyComponent):
        queue = "RMDEV"
        component = "release_machine"
        # TODO: grab responsibles list from ABC, etc
        responsibles = [
            "mvel",
            "glebov-da",
            "ilyaturuntaev",
        ]

    class ReleaseMachineTest(BuggyComponent):
        queue = "RMDEV"
        component = "release_machine_test"
        responsibles = [
            "mvel",
            "glebov-da",
            "ilyaturuntaev",
        ]

    class SearchTests(BuggyComponent):
        component = "search_tests"
        st_component_ids = [1604]
        responsibles = [
            "mvel",
        ]

    class WebBaseSearch(BuggyComponent):
        component = "basesearch"
        st_component_ids = [4458]
        responsibles = [
            "nikita-uvarov",
            "sankear",
        ]

    class WebMiddleSearch(BuggyComponent):
        queue = "MIDDLE"
        component = "middlesearch"
        responsibles = [
            "ulyanin",
            "sankear",
            "kulikov",
            "vmordovin",
            "grievous",
        ]

    class Begemot(BuggyComponent):
        queue = "BEGEMOT"
        component = "begemot"
        responsibles = [
            "gluk47",
            "ageraab",
            "denis28",
        ]

    class Wizard(BuggyComponent):
        queue = "REQWIZARD"
        component = "wizard"
        responsibles = [
            "gluk47",
            "ageraab",
            "denis28",
        ]

    class NoapacheUpper(BuggyComponent):
        queue = "NOAPACHE"
        component = "upper"
        responsibles = [
            "avitella",
            "elshiko",
        ]

    class Report(BuggyComponent):
        queue = "WEBREPORT"
        component = "report"
        responsibles = [
            "elshiko",
            "kozunov",
            "alex-ersh",
            "avitella",
        ]

    class Balancer(BuggyComponent):
        queue = "BALANCER"
        component = "balancer"
        responsibles = [
            "carzil",
            "smalukav",
            "dmitryno",
            "velavokr",
            "kulikov",
        ]

    class BalancerConfig(BuggyComponent):
        queue = "MINOTAUR"
        component = "balancer_config"
        responsibles = [
            "kulikov",
            "smalukav",
            "mvel",
        ]

    class Formulas(BuggyComponent):
        component = "formulas"
        st_component_ids = [1604]
        responsibles = [
            "ilnurkh",
            "mvel",
            "gaiduk",
        ]

    class EntitySearch(BuggyComponent):
        queue = "OBJECTS"
        component = "entitysearch_tests"
        st_component_ids = [26929, ]  # backend
        responsibles = ["konovodov"]

    class Turbo(BuggyComponent):
        component = "turbo"
        responsibles = ["shubert"]

    class PrsOps(BuggyComponent):
        component = "prs_ops"
        responsibles = ["morozyto"]

    class UPS(BuggyComponent):
        queue = "UPS"
        component = "ups"
        responsibles = [
            "mvel",
            "ilyaturuntaev",
        ]
        case_description = 'When task is in `FAILURE` status'


class BugBanner(object):
    """
        Show ads about bug reporting in task info (draw ST issue creation links)
    """

    @staticmethod
    def show_ad(
        component,
        st_component_ids=None,
        responsibles=None,
        queue='SEARCH',
    ):
        _show_ad(
            task=sdk2.Task.current,
            component=component,
            st_component_ids=st_component_ids,
            responsibles=responsibles,
            queue=queue,
        )


class BugBannerTask(sdk2.Task):
    """
        Task mixin class for showing ads about bug reporting in task info
        (draw ST issue creation links).
    """

    def configure_bugbanner(
        self, component,
        st_component_ids=None,
        responsibles=None,
        queue='SEARCH',
        case_description=None,
    ):
        """
            For internal use only. Consider use add_bugbanner instead.
        """
        self.Context.bugbanner = self.Context.bugbanner or {}

        self.Context.bugbanner[component] = {
            'component': component,
            'st_component_ids': st_component_ids,
            'responsibles': responsibles,
            'queue': queue,
            'case_description': case_description,
        }

    def add_bugbanner(self, banner_class, component=None, add_responsibles=list()):
        if component is None:
            component = banner_class.component

        self.configure_bugbanner(
            component=component,
            st_component_ids=banner_class.st_component_ids,
            responsibles=banner_class.responsibles + add_responsibles,
            queue=banner_class.queue,
            case_description=banner_class.case_description,
        )

    def _show_ad(self, update=False):
        bugbanner_cfgs = self.Context.bugbanner
        if not bugbanner_cfgs:
            logging.error('BugBanner is not configured.')
            return

        for component, bugbanner_cfg in bugbanner_cfgs.items():
            if "component" not in bugbanner_cfg:
                # old style bugbanner, skip
                continue

            _show_ad(
                task=sdk2.Task.current,
                component=bugbanner_cfg['component'],
                st_component_ids=bugbanner_cfg['st_component_ids'],
                responsibles=bugbanner_cfg['responsibles'],
                queue=bugbanner_cfg['queue'],
                update=update,
                case_description=bugbanner_cfg.get('case_description'),
            )

    def on_success(self, prev_status):
        logging.debug("on_success handler executed")
        self._show_ad(update=True)
        sdk2.Task.on_success(self, prev_status)

    def on_failure(self, prev_status):
        logging.debug("on_failure handler executed")
        self._show_ad(update=True)
        sdk2.Task.on_failure(self, prev_status)

    def on_break(self, prev_status, status):
        logging.debug("on_break handler executed")
        self._show_ad(update=True)
        sdk2.Task.on_break(self, prev_status, status)


def _get_task_description(task):
    descr = string.all_to_str(task.Parameters.description)
    # SEARCH-1934
    descr = re.sub(r'<a href="([^" ]+)">(.+?)</a>', ' \\1 (\\2) ', descr)

    description = (
        'Task: {task_link}\n'
        '**{task.status}** in task {task.type}\n'
        '\n'
        'Task description:\n{description}\n'
    ).format(
        task=task,
        task_link=link_builder.task_link(task.id, plain=True),
        description=descr,
    )

    task_resources = sdk2.Resource.find(task_id=task.id).limit(100)
    logging.debug("Task resources: %s", task_resources)
    for resource in task_resources:
        if resource.type == resource_types.CORE_DUMP:
            description += 'Core dump: {}\n'.format(
                utils2.resource_redirect_url(resource.id),
            )
        if resource.type == resource_types.FILTERED_GDB_TRACEBACK:
            description += 'Filtered traceback: {}\n'.format(
                utils2.resource_redirect_url(resource.id),
            )

    return description


def _show_ad(
    task, component, st_component_ids, responsibles, queue,
    update=False,
    case_description=None,
):
    initialized = task.Context.bug_report_init_field or []
    if not case_description:
        case_description = 'On any problems'

    # do not show ads twice
    if component in initialized:
        if not update:
            return
    else:
        initialized.append(component)
        task.Context.bug_report_init_field = initialized

    params = [
        ('type', '1'),
        ('priority', '2'),
        ('queue', queue),
        ('description', _get_task_description(task)),
    ]

    if responsibles:
        params.append(('assignee', responsibles[0]))
        for responsible in responsibles:
            params.append(('followers', responsible))

    for st_component_id in st_component_ids or []:
        params.append(('components', str(st_component_id)))

    bug_report_link = "https://st.yandex-team.ru/createTicket?{}".format(parse.urlencode(params))
    bug_report_html = (
        '{update}You are using <b>{component}</b>. '
        '{case_description} feel free to <a href="{link}">report {component} issue</a>'.format(
            update='<b>(Updated)</b> ' if update else '',
            component=component,
            case_description=case_description,
            link=bug_report_link,
        )
    )
    task.set_info(bug_report_html, do_escape=False)
