# -*- coding: utf-8 -*-

"""
Basic search components builder
Author: alexeykruglov@
Maintainers: mvel@

Please obey PEP8 coding standards and Sandbox StyleGuides here:
https://wiki.yandex-team.ru/sandbox/codestyleguide
https://wiki.yandex-team.ru/Users/mvel/sandbox/extended-sandbox-styleguide
"""
import six
import glob
import httplib
import json
import logging
import os
import re
import shutil
import stat
import time
import traceback
import urllib2
import tempfile

from sandbox import common
import sandbox.common.errors as sandbox_errors
import sandbox.common.types.misc as ctm
from sandbox import sandboxsdk
from sandbox.sandboxsdk import network
from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk.channel import channel

from sandbox.projects import resource_types
from sandbox.projects.balancer import resources as balancer_resources
from sandbox.projects.common import apihelpers
from sandbox.projects.common import cgroup as cgroup_api
from sandbox.projects.common import config_processor as cfgproc
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils
from sandbox.projects.common.fusion.tools import get_avg_doc_size, get_info_server, run_rtyserver_command
from sandbox.projects.common.profiling import gperftools as profiler
from sandbox.projects.common.search import bugbanner as bb
from sandbox.projects.common.search import bugbanner2 as bb2
from sandbox.projects.common.search import config as sconf
from sandbox.projects.common.search import settings as media_settings
import sandbox.projects.common.thumbdaemon.utils as thumbdaemon_utils
import sandbox.projects.images.basesearch.resources as images_basesearch_resources
from sandbox.projects.images.daemons import resources as images_daemons_resources
from sandbox.projects.images.metasearch import resources as images_metasearch_resources
from sandbox.projects.images.models import resources as images_models_resources
from sandbox.projects.images.rq import resources as images_rq_resources
from sandbox.projects.websearch.upper import resources as upper_resources
from sandbox.projects.websearch.middlesearch import resources as middle_resources
from sandbox.projects.VideoSearch import video_resource_types as video_resources

import sandbox.projects.saas.common.resources as saas_resources


_SAVE_COREDUMP_TIME = 60

# Singleton bug banner instance for search components
_BUG_BANNER = bb.BugBanner()
_BUG_BANNER2 = bb2.BugBanner()


class VerifyStderrCommon(sp.SandboxBoolParameter):
    name = 'verify_stderr'
    description = 'Verify stderr (fail task if error on unknown message in stderr detected)'
    group = 'Debug options'
    default_value = True


class SearchComponent(object):
    """
        Базовое представление поискового компонента в Sandbox.
        "Почти поисковый компонент", т.к. большинство методов, работающих с поисковыми компонентами
        ожидают на входе SearchExecutableComponent. Исключение - DolbilkaExecutor.run_session
        В большинстве случаев нужно пробовать отнаследоваться от SearchExecutableComponent
        или одного из его дочерних классов
    """
    name = 'unknown_component'
    port = None

    def __init__(self, task=None):
        self.process_parameters = None
        self.process = None
        self.task = task or None

    def __enter__(self):
        self.start()
        self.wait()
        return self

    def __exit__(self, exc_type, exc_value, traceback):
        self.stop()

    def start(self):
        pass

    def stop(self):
        """
            Остановить выполнение исполняемого файла, если он запущен.
        """
        if self.process:
            self.process.kill()

    def wait(self):
        pass

    def use_component(self, work_func):
        pass

    def get_port(self):
        return self.port

    def is_running(self):
        """
            Проверка, запущен ли поиск.
        """
        if self.process:
            return self.process.poll() is None
        return False

    @staticmethod
    def _wait_coredump(sleep_sec=_SAVE_COREDUMP_TIME):
        """
            Метод вызывают когда есть вероятность того что процесс скоркается
            и нужно подождать пока создастся корка

            TODO: улучшить метод - выходить из метода сразу как корка создалась
        """
        logging.info("wait for coredump. sleep %s sec", sleep_sec)
        time.sleep(sleep_sec)

    def _process_post_mortem(self):
        utils.show_process_tail(self.process)
        params_str = ' '.join(self.process_parameters)
        if False and self.process.returncode == -9 and "valgrind" not in params_str:
            raise sandbox_errors.TemporaryError(
                "Sporadic coredump detected, restarting task. If you're hit with this problem, "
                "please ask sandbox@ or ping mvel@ at SEARCH-2854. See also CV-335. "
            )
        process.throw_subprocess_error(self.process)


def parse_version_info(text):
    """
        Получить информацию об исполняемом файле, проанализировав вывод команды "[binary] -v"
        Возвращает dict с ключами:
          version
          svn_url
          svn_revision
          last_changes_author
          build_by
          top_src_dir
          top_build_dir
          build_date
          hostname
          compiler
          compile_flags
        Значения в словаре равны None, если соответствующий параметр не найден
    """
    infos = [
        ('version', r'version:(.*)\n'),
        ('svn_url', r'URL:(.*)\n'),
        ('svn_revision', r'Last Changed Rev:(.*)\n'),
        ('last_changed_author', r'Last Changed Author:(.*)\n'),
        ('build_by', r'Build by:(.*)\n'),
        ('top_build_dir', r'Top build dir:(.*)\n'),
        ('top_src_dir', r'Top src dir:(.*)\n'),
        ('build_date', r'Build date:(.*)\n'),
        ('hostname', r'Hostname:(.*)\n'),
        ('compiler', r'Compiler:(.*)\n'),
        ('compile_flags', r'Compile flags:(.*)\n'),
    ]

    result = {}

    for name, regular_expression in infos:
        r = re.search(regular_expression, text)
        result[name] = r.group(1).strip() if r else None

    return result


def store_sync_time(sync_time):
    channel.task.ctx['resource_sync_time'] = channel.task.ctx.get('resource_sync_time', 0) + int(sync_time)


DEFAULT_START_TIMEOUT = 600
DEFAULT_SHUTDOWN_TIMEOUT = 360


def _check_file(path, name, check_is_file=False):
    """
        Check attributes only, do not set them.
        Attributes should be set ONLY on resource creation.
        See SEARCH-2423 and https://ml.yandex-team.ru/thread/2370000003085773876/
    """
    logging.info('Checking if %s at %s exists...', name, path)
    eh.ensure(os.path.exists(path), 'Cannot find {} "{}"'.format(name, path))
    if check_is_file:
        eh.ensure(os.path.isfile(path), 'Sorry, {} "{}" is not a file'.format(name, path))
    mode = os.stat(path)[stat.ST_MODE]
    mode_str = oct(mode)
    logging.info('Current file %s access mode is %s', path, mode_str)
    mask = stat.S_IXUSR | stat.S_IXGRP | stat.S_IXOTH
    if mode & mask != mask:
        logging.info('Some executable bits of file %s are missing', path)


class SearchExecutableComponent(SearchComponent):
    """
        Общее представление для исполняемого компонента поиска в Sandbox.
    """
    http_collection = 'yandsearch'
    memory_limit = None
    outputs_to_one_file = True
    use_verify_stderr = False
    sanitize_type = None

    def __init__(
        self, work_dir, binary, port, config_class, config_file, config_params,
        use_profiler=False,
        use_gperftools=False,
        start_timeout=DEFAULT_START_TIMEOUT,
        shutdown_timeout=DEFAULT_SHUTDOWN_TIMEOUT,
        outputs_to_one_file=True,
        use_verify_stderr=False,
        run_cmd_patcher=None,
        cgroup=None,
        run_count=None,
        task=None
    ):
        SearchComponent.__init__(self, task=task)
        self.outputs_to_one_file = outputs_to_one_file
        self.use_verify_stderr = channel.task.ctx.get(VerifyStderrCommon.name, use_verify_stderr)
        self.cgroup = cgroup

        self.work_dir = work_dir

        self.binary = os.path.abspath(binary)
        self.check_binary()

        self.info = self._get_info()

        self.log_prefix = None
        self.port = str(port)

        if config_file:
            self.config = config_class.get_config_from_file(os.path.abspath(config_file))
        else:
            self.config = config_class.get_production_config()

        if len(config_params) > 0:
            self.config.apply_local_patch(config_params)

        self.run_cmd_patcher = run_cmd_patcher
        self.after_start = None

        self.run_count = run_count or 0
        self.use_profiler = use_profiler
        self.profile_res_id = None
        self.raw_profile_res_id = None
        self.use_gperftools = use_gperftools
        self.start_timeout = start_timeout
        self.shutdown_timeout = shutdown_timeout
        self.configs_dir = None

        self.killed_by_mem_watchdog = False

        self.rss_mem_after_start = None
        self.rss_mem_before_stop = None

        self.environment = {}

    def check_binary(self):
        """
            Verifies that search component executable exists
            and has proper executable attributes.
        """
        _check_file(self.binary, 'search component executable', check_is_file=True)

    @staticmethod
    def _check_if_exists(path, name):
        """
            Проверяет наличие файла или каталога. Пытается выставить исполняемый атрибут
            (если не получилось, не падает, просто пишет ошибку в лог)
        """
        _check_file(path, name)

    def replace_config_parameter(self, name, value):
        """
            Пропатчить параметр в конфиге. Вызывать можно только пока компонент не запущен.
        """
        self.config.apply_local_patch({name: value})

    def _get_required_hosts_and_ports(self):
        """
            Список пар (хост, порт), которыми должен владеть компонент для нормальной работы
        """
        return [('localhost', self.port)]

    def _check_port_is_free(self):
        """
            Определяет, свободен ли порт.
            Если порт занят кидает исключение SandboxTaskUnknownError.
        """
        for host, port in self._get_required_hosts_and_ports():
            logging.info("Checking port %s", port)
            network.get_port_info(port)
            if not network.wait_port_is_free(port, host, 60, 5):
                eh.fail('Port {}:{} is not free.'.format(host, port))

    def start(self):
        """
            Проверить порт и запустить поиск.
            Если поиск не готов к запуску, кидается исключение.
        """
        logging.info('Starting search component "%s"', self.name)

        # SEARCH-5287  Prevent self TCP connection emergence chance
        if int(self.port) >= 32768:
            eh.check_failed("Port number {0} is not from local port range ({0} > 32768)".format(self.port))

        self._check_port_is_free()
        logging.info("Port sanity check complete")

        config_path = self._generate_config()

        cmd = [self.binary] + self._get_run_cmd(config_path)
        if self.run_cmd_patcher:
            cmd = self.run_cmd_patcher(cmd)

        environment = dict(os.environ)
        environment.update(self.get_environment())

        self.run_count += 1
        if self.use_profiler:
            env_upd = profiler.get_profiler_environment(
                use_gperftools=self.use_gperftools,
                executable_path=self.binary,
                session_name="{0}-{1}".format(self.port, self.run_count),
                work_dir=self.work_dir
            )
            environment.update(env_upd)

        if 'PATH' in environment:
            logging.info("environment['PATH']=%s", environment['PATH'])

        environment['SB_SEARCH_COMPONENT_LOG_PATH'] = paths.get_logs_folder()
        logging.info(
            "Will run search component %s with the following environment:\n%s\nCgroup:%s",
            self.name,
            json.dumps(environment, indent=4),
            None if self.cgroup is None else self.cgroup.name
        )

        self.log_prefix = 'run_{}'.format(self.name)

        self.process_parameters = cmd
        self.process = process.run_process(
            cmd,
            wait=False,
            log_prefix=self.log_prefix,
            outputs_to_one_file=self.outputs_to_one_file,
            environment=environment,
            preexec_fn=None if self.cgroup is None else self.cgroup.set_current
        )
        logging.info('Search component %s pid: %s', self.name, self.process.pid)

        self._run_mem_profiler()

        return self.process

    def get_configs_dir(self):
        if self.configs_dir is None:
            self.configs_dir = utils.create_misc_resource_and_dir(
                channel.task.ctx,
                'patched_configs_resource_id',
                'patched configs',
                'configs',
                task=self.task
            )
        return self.configs_dir

    def get_collection(self):
        return self.http_collection

    def get_port(self):
        return self.port

    def _generate_config(self):
        configs_dir = self.get_configs_dir()
        config_path = paths.get_unique_file_name(configs_dir, '{}{}.cfg'.format(self.name, self.port))
        self.config.save_to_file(config_path)

        return config_path

    def _get_run_cmd(self, config_path):
        """
            Возвращает список параметров в виде list для запуска поиска.
        """
        raise Exception("not implemented")

    def get_environment(self):
        """
            Возвращает словарь с переменными окружения для запуска поиска.
        """
        return self.environment

    def set_environment(self, new_environment):
        """
            Устанавливает требуемое окружение компонента поиска
            (будет затем использовано в методе start())
        """
        self.environment = new_environment

    def set_cgroup(self, cgroup):
        """
            Setup cgroup for component (will be used in start() method)
        """
        self.cgroup = cgroup

    def _run_mem_profiler(self):
        profiler_dir = utils.create_misc_resource_and_dir(
            channel.task.ctx,
            'profiler_logs_resource_id',
            'profiler logs',
            'profiler',
            task=self.task
        )
        file_name = paths.get_unique_file_name(profiler_dir, '{}{}.txt'.format(self.name, self.port))

        watchdog_func = None
        if self.memory_limit:
            logging.debug('creating watchdog for limit %s', self.memory_limit)

            def mem_watchdog(component, ps_info):
                if int(ps_info['vsz']) >= component.memory_limit:
                    logging.info('mem watchdog is going to kill process %s', component.process.pid)
                    component.killed_by_mem_watchdog = True
                    component.process.kill()

            def watchdog_func(ps_info):
                mem_watchdog(self, ps_info)

        process.start_process_profiler(self.process, ['rss', 'vsz', 'pcpu'], file_name, watchdog_func=watchdog_func)

    def wait(self, timeout=None):
        """
            Подождать запуска компонента поиска
        """
        if timeout is None:
            timeout = self.start_timeout
        logging.info("Wait search component %s [%s] for %s seconds", self.name, self.process.pid, timeout)
        start = time.time()
        finish = start + timeout
        while time.time() < finish:
            if self.is_running():
                if all([not network.is_port_free(port, host) for host, port in self._get_required_hosts_and_ports()]):
                    logging.info("Search component {} [{}] started after {} seconds".format(
                        self.name, self.process.pid, round(time.time() - start, 3)))
                    try:
                        self.rss_mem_after_start = self.get_rss_memory()
                    except Exception:
                        logging.info('Can not get RSS memory size:\n%s', traceback.format_exc())

                    # for SEARCH-1010 and similar
                    if self.after_start is not None:
                        logging.info("Running after start hook for %s [%s]", self.name, self.process.pid)
                        self.after_start()

                    return True
            else:
                self._process_post_mortem()
            time.sleep(1)
        if self.is_running():
            logging.info("Kill process %s", self.process.pid)
            os.kill(self.process.pid, 6)  # generate coredump
            self._wait_coredump()
        eh.check_failed("Cannot connect to search component {}, port {} in {} seconds".format(
            self.name, self.port, timeout
        ))

    def warmup_request(self):
        """
        Некоторые тесты перед стрельбой требуют одного "прогревочного" запроса,
        см. SEARCH-616, по умолчанию не делаем ничего.
        """
        pass

    def use_component(self, work_func):
        """
            Работать с компонентами нужно через этот метод,
            т.к. он проверяет, что компонент не завершил свою работу
            после того, как его подолбили.
        """
        try:
            return work_func()
        except Exception:
            logging.info("Exception at search::components::use_component:\n%s", eh.shifted_traceback())
            self._wait_coredump()
            raise
        finally:
            if self.is_running():
                logging.info("Component is running")
            else:
                logging.info("Component is not running")
                self._process_post_mortem()

    def stop(self):
        """
            Остановить выполнение поиска. Если не запущен или остановлен,
            то кидается исключение SandboxTaskFailureError.
        """
        logging.info('Stopping search component "%s"', self.name)

        if self.process is None:
            eh.check_failed("Search component is not started")
        if not self.is_running():
            logging.info("%s died", self.name)
            self._wait_coredump()
            self._process_post_mortem()

        component_output = self.process.stderr_path if not self.outputs_to_one_file else self.process.stdout_path

        try:
            self.rss_mem_before_stop = self.get_rss_memory()
        except Exception:
            logging.info('Can not get RSS memory size:\n' + traceback.format_exc())
        self._stop_impl()

        if self.use_profiler:
            profiler.read_profile_data(
                executable_path=self.binary,
                pid=self.process.pid,
                session_name="{0}-{1}".format(self.port, self.run_count),
                profile_res_id=self.profile_res_id,
                raw_profile_res_id=self.raw_profile_res_id,
                use_gperftools=self.use_gperftools,
            )
        self._check_port_is_free()
        self.process = None
        logging.info('Stopped search component "%s"', self.name)

        if self.use_verify_stderr:
            if component_output:
                logging.info('Verifying stderr for component "%s" at "%s"', self.name, component_output)
                self.verify_stderr(component_output)
            else:
                logging.error('Empty component output filename, stderr verification disabled for "%s"', self.name)
        else:
            logging.info('Verifying stderr is disabled for component "%s"', self.name)

    def _stop_impl(self):
        """
            Kill the process and wait it to die and release port
        """
        self.process.kill()
        time.sleep(1)

    def verify_stderr(self, file_name):
        """
            Check stderr output stored in file for errors/warnings existance.

            To be redefined in child classes.
            Also you may want separate stderr/stdout streams to different files using
            self.outputs_to_one_file = False
        """
        pass

    def get_pid(self):
        eh.verify(self.is_running(), "search component is not running")
        return self.process.pid

    def get_rss_memory(self):
        """
            Get resident memory size (in Kbytes)
        """
        with open("/proc/{}/status".format(self.get_pid())) as f:
            for line in f:
                if line.startswith('VmRSS:'):
                    size_kb = line[6:].strip('\n').strip(' ')
                    if size_kb.endswith(' kB'):
                        return int(size_kb[:-3])

    def fetch_cgi(self, cgi, efforts=1, timeout=20, expected_code=None):
        """
            Пытается задать запрос вида http://localhost:<component-port>/<collection>/<cgi>
            :param cgi: cgi-параметры запросаадрес без указания хост-порта
            :param efforts: количество попыток
            :param timeout таймаут на каждый запрос
            :param expected_code: ожидаемый код возврата (по умолчанию - не проверяется)
            :return: ответ сервера (в виде строки)
        """
        if cgi and cgi[0] != '?':
            cgi = '?' + cgi
        return self.fetch(
            url="/{}{}".format(self.http_collection, cgi),
            efforts=efforts,
            timeout=timeout,
            expected_code=expected_code,
        )

    def fetch(self, url, efforts=1, timeout=20, expected_code=None):
        """
            Пытается задать запрос вида http://localhost:<component-port><url>
            :param url: адрес без указания хост-порта
            :param efforts: количество попыток
            :param timeout время ожидания на каждый запрос
            :param expected_code: ожидаемый код возврата (по умолчанию не проверяется)
            :return: ответ сервера (в виде строки)
        """
        response = None
        full_url = "http://localhost:{}".format(self.port) + url
        logging.info("Fetching url {} with timeout={}".format(full_url, timeout))
        for effort in range(efforts):
            try:
                response = urllib2.urlopen(full_url, timeout=timeout)
                break
            except Exception:
                logging.info("Error on effort {}:\n{}".format(effort, eh.shifted_traceback()))
        else:
            # если все попытки провалились - упасть
            eh.check_failed(
                "Cannot fetch url after {} attempts and {} timeout (see logs for detailed error): {}".format(
                    efforts, timeout, full_url
                )
            )

        if expected_code is not None:
            if not expected_code == response.getcode():
                eh.check_failed("Wrong return code!\nExpected code: {},\nGot: {}".format(
                    expected_code, response.getcode()
                ))
        return response.read()

    def get_factor_names(self):
        logging.info("Try to get factor names")
        return self.fetch_cgi("?info=factornames")

    def _get_info(self):
        """
            Получить информацию об исполняемом файле. См. метод parse_version_info
        """
        proc = process.run_process(
            [self.binary, '-v'],
            check=False,
            outs_to_pipe=True,
        )
        text_info = proc.stdout.read()

        return parse_version_info(text_info)


class StandardSearch(SearchExecutableComponent):
    """
        Различные базовые поиски, метапоиск
    """
    @property
    def _common_test_urls(self):
        return [
            {
                "url": "/{}?info=getconfig".format(self.http_collection),
                "expected_code": httplib.OK,
                "expected_re": [
                    re.compile('.*<Collection.*id="yandsearch".*'),
                    re.compile('.*<Collection.*autostart="must".*'),
                ],
            },
        ]

    def check_simple_reqs(self, add_urls=None):
        """
            Check binary with simple requests
            :param add_urls: list of additional urls to fetch with expected code and expected data
        """
        if not self.is_running():
            eh.check_failed("Search is not started")
        info = parse_version_info(self.fetch_cgi(
            cgi="?info=getversion",
            expected_code=httplib.OK
        ))
        # check these fields to be nonempty and equal for both ways of getting
        important_info_fields = [
            'svn_url',
            'svn_revision',
            'last_changed_author',
            'top_src_dir',
            'hostname',
            # 'compiler',  # devtools@ broke it
        ]
        for key in important_info_fields:
            if not info[key]:
                eh.check_failed("Empty binary info, field = {}".format(key))
            if info[key] != self.info[key]:
                eh.check_failed("Difference between ways of getting basesearch info:\n{}: {} != {}".format(
                    key, self.info[key], info[key]
                ))
        fetch_urls = self._common_test_urls
        if add_urls:
            fetch_urls.extend(add_urls)
        for i in fetch_urls:
            resp = self.fetch(i["url"], expected_code=i["expected_code"])
            if i.get("expected_re"):
                if any((reg.search(resp) is None) for reg in i["expected_re"]):
                    eh.check_failed("Response:\n{}\nDoes not match regexp".format(resp))

    @property
    def _stop_command(self):
        return "/admin?action=shutdown"

    def _stop_impl(self):
        """
            Остановить программу поиска.
        """
        # see logic in ScheduleExit (e.g. hsgenpages.cpp). we tell search to shut down
        # within 0.95*shutdown_timeout and set a bit more time for cgi request handling.
        shutdown_cgi = "{}&timeout={}".format(self._stop_command, int(int(self.shutdown_timeout) * 0.95))
        logging.info("Send shutdown command with %s sec timeout using url %s", self.shutdown_timeout, shutdown_cgi)
        try:
            self.fetch(shutdown_cgi, timeout=self.shutdown_timeout)
        except Exception:
            self._wait_coredump()
            raise

        logging.info("Will wait %s sec for component termination:", self.shutdown_timeout)
        for seconds_passed in six.moves.xrange(self.shutdown_timeout):
            if self.process.poll() is not None:
                break
            logging.debug(
                "Waiting 1 sec for search component, passed %s of %s [%s]",
                seconds_passed, self.shutdown_timeout, self.process.pid,
            )
            time.sleep(1)

        if self.process.poll() is not None:
            if self.process.returncode:
                self._wait_coredump()
                self._process_post_mortem()
            logging.info("Search component with pid [%s] terminated correctly", self.process.pid)
        else:
            logging.info(
                "Search component process with pid [%s] is still alive, "
                "wait until it coredumps correctly (SEARCH-6726)",
                self.process.pid,
            )
            self._wait_coredump(_SAVE_COREDUMP_TIME * 3)
            try:
                self.process.kill()
            except Exception as exc:
                eh.log_exception("Cannot kill process, maybe it's already died", exc)
            logging.info("Search component with pid [%s] was killed by timeout", self.process.pid)
            time.sleep(1)  # need some time to release port


# порт по умолчанию для запуска базового поиска на клиентах Sandbox
DEFAULT_BASESEARCH_PORT = 17171


class BasesearchBase(StandardSearch):
    """
        Представление базового поиска в Sandbox
    """
    name = 'basesearch'

    _LOAD_LOG_FILENAME = 'basesearch{}_load.log'
    _SERVER_LOG_FILENAME = 'basesearch%s_server.log'
    _PASSAGE_LOG_FILENAME = 'basesearch%s_passage.log'
    _EMERGENCY_FILENAME = 'degrade'
    _EMERGENCY_CHECK_PERIOD = 5  # period can't be less than 5 sec (see TCommonSearch::InitEmergencyCgi)

    ignore_bad_dh_errors = False

    def __init__(
        self, work_dir, config_class, binary, database_dir,
        config_file=None,
        archive_model_path=None,
        polite_mode=True,
        ignore_index_generation=False,
        port=DEFAULT_BASESEARCH_PORT,
        config_params=None,
        use_profiler=False,
        use_gperftools=False,
        start_timeout=DEFAULT_START_TIMEOUT,
        from_sdk2=False,
        **kwargs
    ):
        bbanner = _BUG_BANNER2 if from_sdk2 else _BUG_BANNER
        ws_banner = bb2.Banners.WebBaseSearch
        bbanner.show_ad(
            component=ws_banner.component,
            st_component_ids=ws_banner.st_component_ids,
            responsibles=ws_banner.responsibles,
        )

        self.database_dir = database_dir
        self._check_if_exists(database_dir, 'basesearch database folder')

        if not config_params:
            config_params = {}
        config_params.update({
            'Collection/IndexDir': database_dir,
            'Collection/MXNetFile': (archive_model_path, True),
            'Collection/UserParams/Polite': ('true' if polite_mode else None, True),
            'Collection/UserParams/NoIndexGeneration': ('' if ignore_index_generation else None, True),
            'Collection/EmergencyFile': os.path.join(work_dir, self._EMERGENCY_FILENAME),
            'Collection/EmergencyCheckPeriod': (str(self._EMERGENCY_CHECK_PERIOD), True),
            'Server/Port': str(port),
            'Server/ClientTimeout': 300,
            'Server/LoadLog': os.path.join(work_dir, self._LOAD_LOG_FILENAME.format(port)),
            'Server/ServerLog': os.path.join(work_dir, self._SERVER_LOG_FILENAME % port),
            'Server/PassageLog': os.path.join(work_dir, self._PASSAGE_LOG_FILENAME % port),
            'Server/LogExceptions': 'True',
        })

        super(BasesearchBase, self).__init__(
            work_dir=work_dir,
            binary=binary,
            port=port,
            config_class=config_class,
            config_file=config_file,
            config_params=config_params,
            use_profiler=use_profiler,
            use_gperftools=use_gperftools,
            start_timeout=start_timeout,
            **kwargs
        )

    def _get_run_cmd(self, config_path):
        return ['-d', '-p', str(self.port), config_path]

    def get_loadlog(self):
        return os.path.join(self.work_dir, self._LOAD_LOG_FILENAME.format(self.port))

    def get_emergency_file(self):
        return os.path.join(self.work_dir, self._EMERGENCY_FILENAME)

    def get_emergency_check_period(self):
        return self._EMERGENCY_CHECK_PERIOD

    def verify_stderr(self, file_name):
        if not os.path.exists(file_name):
            return

        # Try to resolve SEARCH-1008 and SEARCH-1022
        counters = {
            'debug': 0,
            'info': 0,
            'warning': 0,
            'errors': 0,
            'broken_pipe': 0,
            'bad_dh_param': 0,
            'oversized_formulas': 0,
            'polite_mode': 0,
            'cgi_parse_error': 0,
            'perf_records': 0
        }

        with open(file_name) as errors:
            for line in errors:
                line = line.strip()
                if not line:
                    # skip empty lines
                    continue

                if '[DEBUG]' in line:
                    counters['debug'] += 1
                    continue

                if '[INFO]' in line:
                    counters['info'] += 1
                    continue

                if '[WARNING]' in line:
                    counters['warning'] += 1
                    continue

                if 'can not write to socket output stream' in line or 'can not writev to socket output stream' in line:
                    counters['broken_pipe'] += 1
                    continue
                    # TODO: ignore only in int tests/sync middlesearch test
                    # if self.ignore_broken_pipe:
                    #    continue

                if 'Bad dh param' in line:
                    # Bad dh param(|eter)
                    counters['bad_dh_param'] += 1
                    if self.ignore_bad_dh_errors:
                        continue

                if 'has more features than librank knows about' in line:
                    counters['oversized_formulas'] += 1
                    continue

                if 'and polite mode is disabled' in line:
                    # erf size mismatch with disabled polite mode
                    counters['polite_mode'] += 1
                    continue

                if 'CGI parsing error' in line:
                    # soft error, but should not be too often
                    counters['cgi_parse_error'] += 1
                    continue

                if ("perf record: Woken up" in line) or ("perf record: Captured and wrote" in line):
                    counters['perf_records'] += 1
                    continue

                if "Invalid factors for doc id" in line:
                    if "IsLawQuery:begemot_query_factors" in line or "IsFinancialQuery:begemot_query_factors" in line:
                        # temporary workaround for SENS-73
                        continue

                counters['errors'] += 1

        channel.task.set_info(
            "Basesearch counts: {}".format(
                ", ".join([
                    "{}: <b>{}</b>".format(k, v)
                    for k, v in counters.items() if v > 0
                ])
            ),
            do_escape=False,
        )

        non_bad_dh_errors = counters['errors'] - counters['bad_dh_param']

        # generic error amount should be relatively small
        eh.ensure(
            non_bad_dh_errors < 50,
            "Verification failed for basesearch: Too many errors ({}) in stderr, "
            "check for problems".format(counters['errors'])
        )

        # On search index switching there can be some old instances that spawn "Bad dh param" errors,
        # but error percent should be miserable.
        eh.ensure(
            counters['bad_dh_param'] < 300,
            "Verification failed for basesearch: Too many 'Bad dh param' errors ({}) in stderr, "
            "check for problems".format(counters['bad_dh_param'])
        )

        # Some random production requests can produce CGI parsing errors, but they should be relatively rare.
        eh.ensure(
            counters['cgi_parse_error'] < 300,
            "Verification failed for basesearch: Too many 'CGI parsing' errors ({}) in stderr, "
            "check for problems".format(counters['cgi_parse_error'])
        )


class BasesearchWeb(BasesearchBase):
    """
        Представление web базового поиска в Sandbox
    """

    def __init__(self, patch_request_threads, **kwargs):
        config_params = get_basesearch_config_overridden_params(patch_request_threads)

        super(BasesearchWeb, self).__init__(
            config_class=sconf.BasesearchWebConfig,
            config_params=config_params,
            **kwargs
        )


class BasesearchWebWithProfilerResources(BasesearchBase):
    """
        Представление web базового поиска в Sandbox
        плюс ресурсы для профайлера
    """

    def __init__(self, patch_request_threads, **kwargs):
        config_params = get_basesearch_config_overridden_params(patch_request_threads)

        super(BasesearchWebWithProfilerResources, self).__init__(
            config_class=sconf.BasesearchWebConfig,
            config_params=config_params,
            **kwargs
        )


class BasesearchWebOverSamohod(BasesearchBase):
    """
        Запуск basesearch поверх индекса Самохода, построенного rtyserver'ом
    """

    stamp_tag_dir = ''

    def __init__(self, database_dir, patch_request_threads, **kwargs):
        patched_index_dir = [x[0] for x in os.walk(database_dir)][1]
        self.stamp_tag_dir = make_platinum_stamp_tag_from_quick(os.path.join(patched_index_dir, 'stamp.TAG'))
        params = get_basesearch_config_overridden_params(patch_request_threads)
        params.update({
            'Collection/CustomStampTag': self.stamp_tag_dir,
        })

        super(BasesearchWebOverSamohod, self).__init__(
            database_dir=patched_index_dir,
            config_class=sconf.BasesearchWebConfig,
            config_params=params,
            **kwargs
        )


def make_platinum_stamp_tag_from_quick(input_path):
    output_dir = tempfile.mkdtemp()
    logging.info("patching stamp.TAG input: %s, output: %s/stamp.TAG", input_path, output_dir)
    replacements = [
        (re.compile(r"^FlatBastardsLanguages=$"), 'FlatBastardsLanguages=ukr,tur'),
        (re.compile(r"^SearchZone=Quick$"), 'SearchZone=PlatinumTier0'),
        (re.compile(r"^BaseType=quick$"), 'BaseType=rus'),
        (re.compile(r"^TierType=quick$"), 'TierType=platinum0'),
        (re.compile(r"^ShardName=primus-Quick-0-0-(\d+)$"), r"ShardName=PlatinumTier0-0-0-\1"),
    ]

    lines = []
    with open(input_path) as inp:
        for line in inp:
            line = line.strip()
            for repl in replacements:
                line = repl[0].sub(repl[1], line)
            lines.append(line)

    with open(os.path.join(output_dir, 'stamp.TAG'), 'w') as out:
        out.write('\n'.join(lines))

    logging.info('\n'.join(lines))
    logging.info('stamp.TAG patched successfully')
    return output_dir


def get_basesearch_config_overridden_params(patch_request_threads):
    params = {
        'Collection/PrefetchSize': '100000000',
    }

    if patch_request_threads:
        params.update({
            'Collection/RequestThreads': '${1 + NCPU}',
            'Collection/RequestQueueSize': '${1 + NCPU}',
            'Collection/SnippetThreads': '${1 + NCPU}',
            'Collection/SnippetQueueSize': '${1 + NCPU}',
            'Collection/FactorsThreads': '${1 + NCPU}',
            'Collection/LongReqsThreads': None,
            'Collection/LongReqsQueueSize': None,
        })

    return params


def get_basesearch_ex(
    binary_id,
    config_id,
    database_id,
    archive_model_id=None,
    polite_mode=True,
    ignore_index_generation=False,
    patch_request_threads=True,
    use_profiler=False,
    use_gperftools=False,
    port=DEFAULT_BASESEARCH_PORT,
    start_timeout=DEFAULT_START_TIMEOUT,
    component_creator=None,
    **kwargs
):
    """
        Синхронизировать ресурсы и получить обёртку web базового поиска
        используется в старых тасках
        :return: объект BasesearchWeb
    """
    start_time = time.time()
    binary_path = channel.task.sync_resource(binary_id)
    config_path = channel.task.sync_resource(config_id)
    database_path = channel.task.sync_resource(database_id)
    archive_model_path = channel.task.sync_resource(archive_model_id) if archive_model_id else None
    store_sync_time(time.time() - start_time)

    if not component_creator:
        component_creator = BasesearchWeb

    return component_creator(
        work_dir=channel.task.abs_path(),
        binary=binary_path,
        database_dir=database_path,
        config_file=config_path,
        archive_model_path=archive_model_path,
        polite_mode=polite_mode,
        ignore_index_generation=ignore_index_generation,
        patch_request_threads=patch_request_threads,
        use_profiler=use_profiler,
        use_gperftools=use_gperftools,
        port=port,
        start_timeout=start_timeout,
        **kwargs
    )


def create_basesearch_params(
    n=None,
    config_required=True,
    database_required=True,
    archive_model_required=True,
    default_patch_request_threads=True,
    default_verify_stderr=None,
    group_name=None,
    component_name=None,
    default_start_timeout=DEFAULT_START_TIMEOUT
):
    if n is None:
        n = ""

    if group_name is None:
        group_name = 'Basesearch{} parameters'.format(n)

    if component_name is None:
        component_name = "basesearch{}".format(n)

    class Params(object):
        class Binary(sp.ResourceSelector):
            name = '{}_executable_resource_id'.format(component_name)
            description = 'Executable'
            resource_type = [
                resource_types.BASESEARCH_EXECUTABLE,
                resource_types.VIDEOSEARCH_EXECUTABLE,
                resource_types.ADDRESSNIP_SEARCH,
                resource_types.SERPAPI_SEARCH,
                images_basesearch_resources.IMGSEARCH_EXECUTABLE,
                images_rq_resources.IMAGES_RQ_BASESEARCH_EXECUTABLE,
                resource_types.IMGSEARCH_RTYSERVER_EXECUTABLE
            ]
            group = group_name
            required = True

        class Config(sp.ResourceSelector):
            name = '{}_config_resource_id'.format(component_name)
            description = 'Config'
            resource_type = [
                resource_types.SEARCH_CONFIG,
                images_basesearch_resources.IMAGES_SEARCH_CONFIG,
                images_basesearch_resources.IMAGES_SEARCH_TIER1_CONFIG,
                images_basesearch_resources.IMAGES_QUICK_SEARCH_CONFIG,
                images_basesearch_resources.IMAGES_CBIR_SEARCH_CONFIG,
                images_basesearch_resources.IMAGES_CBIR_QUICK_SEARCH_CONFIG,
                images_basesearch_resources.IMAGES_CBIR_SEARCH_TIER1_CONFIG,
                resource_types.VIDEO_SEARCH_CONFIG,
                resource_types.ADDRESSNIP_SEARCH_CONFIG,
                resource_types.SERPAPI_SEARCH_CONFIG,
                images_rq_resources.IMAGES_RQ_BASESEARCH_CONFIG,
                images_basesearch_resources.IMGQUICK_SAAS_RTYSERVER_CONFIGS_BUNDLE,
            ]
            group = group_name
            required = config_required

        class Database(sp.ResourceSelector):
            name = '{}_database_resource_id'.format(component_name)
            description = 'Database'
            resource_type = [
                resource_types.SEARCH_DATABASE,
                resource_types.IMAGES_SEARCH_DATABASE,
                resource_types.VIDEO_SEARCH_DATABASE,
                resource_types.VIDEO_QUICK_SEARCH_DATABASE,
                resource_types.VIDEO_ULTRA_SEARCH_DATABASE,
                resource_types.ADDRESSNIP_SEARCH_DATABASE,
                resource_types.SERPAPI_SEARCH_DATABASE,
                resource_types.RTYSERVER_SEARCH_DATABASE,
            ]
            group = group_name
            required = database_required

        class ArchiveModel(sp.ResourceSelector):
            name = '{}_models_archive_resource_id'.format(component_name)
            description = 'Models archive'
            resource_type = [
                resource_types.DYNAMIC_MODELS_ARCHIVE_BASE,
                resource_types.DYNAMIC_MODELS_ARCHIVE,  # deprecated
                images_models_resources.IMAGES_DYNAMIC_MODELS_ARCHIVE,
                images_models_resources.IMAGES_BASESEARCH_PROD_DYNAMIC_MODELS_ARCHIVE,
                resource_types.VIDEO_DYNAMIC_MODELS_ARCHIVE,
            ]
            group = group_name
            required = archive_model_required

        class PoliteMode(sp.SandboxBoolParameter):
            name = '{}_polite_mode'.format(component_name)
            description = 'Polite mode'
            group = group_name
            default_value = True

        class IgnoreIndexGeneration(sp.SandboxBoolParameter):
            name = '{}_ignore_index_generation'.format(component_name)
            description = 'Ignore index generation'
            group = group_name
            default_value = False

        class PatchRequestThreads(sp.SandboxBoolParameter):
            name = '{}_patch_request_threads'.format(component_name)
            description = 'Patch request threads'
            group = group_name
            default_value = default_patch_request_threads

        class StartTimeout(sp.SandboxIntegerParameter):
            name = '{}_indexing_timeout'.format(component_name)
            description = 'Start timeout (sec)'
            group = group_name
            default_value = default_start_timeout

        class VerifyStderr(VerifyStderrCommon):
            name = '{}_verify_stderr'.format(component_name)
            group = group_name
            default_value = VerifyStderrCommon.default_value if default_verify_stderr is None else default_verify_stderr

        class Cgroup(sp.SandboxStringParameter):
            name = '{}_cgroup'.format(component_name)
            description = 'Cgroup'
            group = group_name

        params = (
            Binary,
            Config,
            Database,
            ArchiveModel,
            PoliteMode,
            IgnoreIndexGeneration,
            PatchRequestThreads,
            StartTimeout,
            VerifyStderr,
            Cgroup,
        )

    return Params


def create_noweb_basesearch_params(**kwargs):
    """
        Used for IMAGES and VIDEO
        If database shard is not specified, its name will be taken from requests resource attributes
        Avoid patching request threads to get more production-like results
    """
    return create_basesearch_params(
        database_required=False,
        default_patch_request_threads=False,
        **kwargs
    )


def tune_search_params(params_collection, params, start_timeout=None):
    """
        Utility method to change default values of some basesearch params
        Returns a list with modified params
    """

    def tune_default_value(base_class, value):
        """
            Returns a new parameter class inherited from specified one
            with modified 'default_value' field
        """

        class NewParameter(base_class):
            default_value = value

        return NewParameter

    result = []

    for param in params:
        if start_timeout is not None and param.name == params_collection.StartTimeout.name:
            result.append(tune_default_value(param, start_timeout))
        else:
            result.append(param)

    return result


def cgroup_setup(params, ctx, cgroup=None):
    if common.config.Registry().common.installation == ctm.Installation.LOCAL:
        logging.info("Skipping cgroups creation on local sandbox")
        cgroup = None
    else:
        if cgroup is None:
            try:
                cgroup_props = utils.get_or_default(ctx, params.Cgroup)
                if cgroup_props:
                    cgroup = cgroup_api.create_cgroup(params.Cgroup.name, json.loads(cgroup_props))
            except Exception as e:
                eh.check_failed('Failed to initialize cgroup: {}'.format(str(e)))
    return cgroup


DefaultBasesearchParams = create_basesearch_params()
DefaultVideosearchParams = create_noweb_basesearch_params()


def get_basesearch(
    params=DefaultBasesearchParams,
    use_profiler=False,
    use_gperftools=False,
    port=DEFAULT_BASESEARCH_PORT,
    component_creator=None,
    cgroup=None,
    **kwargs
):
    """
        Получить обёртку web базового поиска - объект класса BasesearchWeb
        Идентификаторы параметров-ресурсов берутся из контекста задачи.
        Ресурсы синхронизируются.
    """
    ctx = channel.task.ctx
    sys_info = sandboxsdk.util.system_info()
    resource = channel.sandbox.get_resource(ctx[params.Binary.name])

    if not sandboxsdk.util.is_arch_compatible(resource.arch, sys_info['arch']):
        eh.check_failed('Cannot execute task.\nResource: {} has incorrect arch {} instead of {}'.format(
            resource.id, resource.arch, sys_info['arch']
        ))

    cgroup = cgroup_setup(params, ctx, cgroup)

    return get_basesearch_ex(
        binary_id=ctx[params.Binary.name],
        config_id=ctx[params.Config.name],
        database_id=ctx[params.Database.name],
        archive_model_id=ctx[params.ArchiveModel.name],
        polite_mode=utils.get_or_default(ctx, params.PoliteMode),
        ignore_index_generation=utils.get_or_default(ctx, params.IgnoreIndexGeneration),
        patch_request_threads=utils.get_or_default(ctx, params.PatchRequestThreads),
        use_profiler=use_profiler,
        use_gperftools=use_gperftools,
        port=port,
        start_timeout=utils.get_or_default(ctx, params.StartTimeout),
        component_creator=component_creator,
        use_verify_stderr=utils.get_or_default(ctx, params.VerifyStderr),
        cgroup=cgroup,
        **kwargs
    )


FUSION_DB_PRE_CREATED = "pre_created"
FUSION_DB_EMPTY = "empty_database"
FUSION_WAIT_INDEX = "wait_index"
FUSION_WAIT_SEARCHER = "wait_searcher"
FUSION_WAIT_MEMORY_FILLED = "wait_memory_filled"

FUSION_DB_SOURCE_TITLE = "Database source params"


class FusionSearch(BasesearchBase):
    """
        Представление Refresh в Sandbox
    """

    name = 'Refresh'

    def __init__(
        self, work_dir, database_dir,
        shard=0,
        shards_count=1,
        sw_config=None,
        frequent_keys=None,
        archive_model_path=None,
        polite_mode=None,
        patch_request_threads=None,
        binary=None,
        configs_folder=None,
        config_file=None,
        fusion_config_path=None,
        max_documents=None,
        default_wait=None,
        oxygen_config=None,
        mxnet_models=None,
        event_log=False,
        static_data_path=None,
        environment_file_path=None,
        extra_params=None,
        patch_params=None,
        start_timeout=DEFAULT_START_TIMEOUT * 3,
        **kwargs
    ):
        config_params = {}

        self.shard = shard
        self.shards_count = shards_count
        self.sw_config = sw_config

        self.configs_folder = configs_folder
        self.basesearch_cfg_path = config_file
        self.fusion_config_path = fusion_config_path
        self.port = kwargs.get("port")
        self.index_dir = self.get_index_dir(database_dir)
        self.max_documents = max_documents
        self.default_wait_function = default_wait
        self.oxygen_config = oxygen_config
        self.mxnet_models = mxnet_models
        self.static_data_path = static_data_path
        self.event_log = event_log
        self.http_collection = str()

        super(FusionSearch, self).__init__(
            config_file=config_file,
            work_dir=work_dir,
            binary=binary,
            config_class=sconf.FusionConfig,
            database_dir=self.index_dir,
            config_params=config_params,
            start_timeout=start_timeout,
            **kwargs
        )

        self.basesearch_port_shift = 1
        self.memory_search_port_shift = 2
        self.controller_port_shift = 3

        self.search_port = int(self.port)
        self.controller_port = self.search_port + self.controller_port_shift
        self.memory_search_port = self.search_port + self.memory_search_port_shift
        self.basesearch_port = self.search_port + self.basesearch_port_shift
        self.start_fail_time = 5

        self.logs_path = channel.task.log_path("refresh_logs_{}".format(self.port))
        if not os.path.exists(self.logs_path):
            paths.make_folder(self.logs_path)

        self.ram_disk_path = channel.task.abs_path("ram_disk_{}".format(self.port))
        if not os.path.exists(self.ram_disk_path):
            paths.make_folder(self.ram_disk_path)

        self.environment_file_path = environment_file_path
        self.extra_params = extra_params or {}
        self.patch_params = patch_params

        if self.extra_params.get('BSCONFIG_INAME') is None:
            self.extra_params['BSCONFIG_INAME'] = os.environ.get('GSID', 'UNKNOWN_ENV')

        self.frequent_keys = frequent_keys

    def create_empty_file(self, path):
        with open(path, 'a'):
            pass

    def get_doc_size(self):
        return get_avg_doc_size(self.controller_port)

    def clone_database(self, source_dir, target_dir):
        # database have to be copied after recent SaaS changes
        ELEMENTS_TO_IGNORE = ["shard.conf", "source_indexes"]

        logging.info("Cloning database: %s -> %s", source_dir, target_dir)

        for element in os.listdir(source_dir):
            element_path = os.path.join(source_dir, element)
            new_path = os.path.join(target_dir, element)

            if element in ELEMENTS_TO_IGNORE:
                continue
            else:
                logging.info("Copying : %s -> %s", element_path, new_path)
                if os.path.isdir(element_path):
                    shutil.copytree(element_path, new_path)
                else:
                    shutil.copy(element_path, new_path)
                paths.chmod(new_path, 0o755)

        self.create_empty_file(os.path.join(target_dir, "source_indexes"))
        return target_dir

    def is_multi_index_dir(self, source_index_dir):
        """
            Detects RTYServer-style index directory, which would have index_* subfolders
        """
        matches = glob.glob(os.path.join(source_index_dir, 'index_*_*'))
        return bool([m for m in matches if os.path.isdir(m)])

    def get_index_dir(self, source_index_dir):
        index_dir = channel.task.abs_path("db_{}".format(self.port))
        disk_index_name = 'index_0000000000_0000000000'

        if os.path.exists(index_dir):
            shutil.rmtree(index_dir)
        logging.info("Making an index dir %s ", index_dir)
        paths.make_folder(index_dir)
        if source_index_dir:
            if not self.is_multi_index_dir(source_index_dir):
                new_index_folder = os.path.join(index_dir, disk_index_name)
                os.mkdir(new_index_folder)
            else:
                new_index_folder = index_dir
            self.clone_database(source_index_dir, new_index_folder)
        return index_dir

    def _generate_config(self):
        return self.fusion_config_path

    def _get_run_cmd(self, config_path):
        max_documents = self.max_documents or 100500

        cmd = [
            self.fusion_config_path,
            "-V", "IndexDir={}".format(self.index_dir),
            "-V", "INDEX_DIRECTORY={}".format(self.index_dir),
            "-V", "LogsDir={}".format(self.logs_path),
            "-V", "LOG_PATH={}".format(self.logs_path),
            "-V", "JournalDir={}".format(self.logs_path),
            "-V", "GlobalLog={}".format(os.path.join(self.logs_path, "global.log")),
            "-V", "FetcherLog={}".format(os.path.join(self.logs_path, "fetcher.log")),

            "-V", "BasePort={}".format(self.search_port),

            "-V", "BackendMemoryPort={}".format(self.memory_search_port),
            "-V", "BackendControllerPort={}".format(self.controller_port),
            "-V", "BackendBasesearchPort={}".format(self.basesearch_port),
            "-V", "BackendSearchPort={}".format(self.search_port),

            "-V", "shardid={}".format(self.shard),
            "-V", "SHARDS_NUMBER={}".format(self.shards_count),

            "-V", "MaxDocuments={}".format(max_documents),
            "-V", "RamdiskDir={}".format(self.ram_disk_path)
        ]

        if self.configs_folder:
            cmd.extend(["-V", "CONFIG_PATH={}".format(self.configs_folder)])
        if self.sw_config:
            cmd.extend(["-V", "SHARDWRITER_CONFIG={}".format(self.sw_config)])
        if self.frequent_keys:

            # hack to make working priemka before new configs release
            shutil.copyfile(self.binary, './rtyserver')
            self.binary = './rtyserver'
            shutil.copyfile(self.frequent_keys, 'frequent_keys.txt')
            self.frequent_keys = 'frequent_keys.txt'

            cmd.extend(["-V", "FREQUENT_KEYS_PATH={}".format(self.frequent_keys)])
        if self.basesearch_cfg_path:
            cmd.extend(["-V", "BaseSearchConfig={}".format(self.basesearch_cfg_path)])
        if self.oxygen_config:
            cmd.extend(["-V", "OxygenOptionsFile={}".format(self.oxygen_config)])
        if self.mxnet_models:
            cmd.extend(["-V", "MXNetFile={}".format(self.mxnet_models)])
            cmd.extend(["-V", "MODELS_PATH={}".format(os.path.dirname(self.mxnet_models))])
        if self.event_log:
            cmd.extend(["-V", "EventLog={}".format(os.path.join(self.logs_path, "event.log"))])
        if self.static_data_path:
            cmd.extend(["-V", "STATIC_DATA_DIRECTORY={}".format(self.static_data_path)])
        if self.environment_file_path:
            cmd.extend(["-E", self.environment_file_path])
        for name, value in self.extra_params.iteritems():
            cmd.extend(["-V", "{}={}".format(name, value)])
        if self.patch_params:
            for name, value in self.patch_params.iteritems():
                cmd.extend(["-P", "{}={}".format(name, value)])

        return cmd

    def start(self):
        instance = super(FusionSearch, self).start()
        time.sleep(self.start_fail_time)
        process.check_process_return_code(instance)

    def kill_softly(self):
        timeout = 160
        address_template = "http://{host}:{port}/?command=shutdown&async=da"
        address = address_template.format(host="localhost", port=self.controller_port)
        response = urllib2.urlopen(address)
        logging.info("Stop command response: %s", response.read())
        self.process.communicate()

        start = time.time()
        while time.time() - start <= timeout:
            if self.process.returncode is not None:
                if self.process.returncode != 0:
                    logging.error("Refresh was stopped with exitcode %s", self.process.returncode)
                    self._wait_coredump()
                    self._process_post_mortem()
                else:
                    logging.info("Refresh was stopped with exitcode %s", self.process.returncode)

                return
            else:
                logging.info("Process still running %s", self.process.__dict__)
                time.sleep(5)
        raise Exception("Failed to stop Refresh: %s seconds timeout" % timeout)

    def _stop_impl(self):
        """
            Kill the process of search
        """
        logging.info("Stopping Refresh")
        if self.process and self.process.poll() is None:
            try:
                self.kill_softly()
            except Exception as e:
                logging.exception("Failed to stop Refresh gracefully: %s", e)
                self.process.kill()

    def fetch_info_request(self, port, request):
        address_template = "http://{host}:{port}/{req}"
        address = address_template.format(host="localhost", port=port, req=request)
        try:
            response = urllib2.urlopen(address, timeout=5)
            return response.read()
        except Exception as e:
            logging.exception("Controller is not available on %s: %s", address, e)
            return None

    def fetch_tass(self):
        return self.fetch_info_request(self.search_port, "tass")

    def fetch_stats(self):
        return self.fetch_info_request(self.controller_port, "status")

    def get_stats(self):
        metric_re = re.compile(r"(?P<name>\S*)\s*:\s*(?P<value>\S*)")
        stats = self.fetch_stats()
        parsed_stats = {}
        if stats:
            for line in stats.split(os.linesep):
                parsed = metric_re.search(line)
                if parsed:
                    name = parsed.group("name").lower()
                    value = parsed.group("value").lower()
                    try:
                        parsed_stats[name] = int(value)
                    except ValueError:
                        parsed_stats[name] = value
        logging.debug(parsed_stats)
        return parsed_stats

    def run_fusion_command(self, command, timeout=10):
        return run_rtyserver_command("localhost", self.controller_port, command, timeout)

    def get_info_server(self):
        info_server = get_info_server(self.controller_port)
        return info_server

    def is_index_ready(self):
        try:
            stats = self.get_stats()
            if int(stats['active']) > 0 and int(stats['active_disk_index_count']) > 0:
                return True
        except KeyError:
            logging.debug("Refresh is not running: No stats")
        except Exception as e:
            logging.debug("Refresh is not yet running (%s)", e)
        return False

    def is_server_running(self):
        try:
            if int(self.get_stats()['search_server_running']) > 0:
                return True
        except KeyError:
            logging.debug("Refresh is not running: No stats")
        except Exception as e:
            logging.debug("Refresh is not yet running (%s)", e)
        return False

    def is_memorysearch_filled(self):
        try:
            if int(self.get_info_server()['result']['docs_in_memory_indexes']) >= self.max_documents:
                return True
        except KeyError:
            logging.debug("Refresh is not running: incorrect info_server")
        except Exception as e:
            logging.debug("Refresh is not yet running (%s)", e)
        return False

    def is_fetching_complete(self, max_fetched):
        return lambda instance=self, max_fetched=max_fetched: instance.is_fetching_complete_cond(max_fetched)

    def is_fetching_complete_cond(self, max_fetched):
        try:
            result = self.get_info_server()['result']
            fetched = (int(result['docs_in_final_indexes']) +
                       max(int(result['docs_in_disk_indexers']),
                           int(result['docs_in_memory_indexes']))
                       )
            if fetched >= max_fetched:
                return True
        except KeyError:
            logging.debug("Refresh is not running: incorrect info_server")
        except Exception as e:
            logging.debug("Refresh is not yet running (%s)", e)
        return False

    def is_indexing_complete(self, max_searchable):
        return lambda instance=self, max_searchable=max_searchable: instance.is_indexing_complete_cond(max_searchable)

    def is_indexing_complete_cond(self, max_searchable):
        try:
            if int(self.get_info_server()['result']['searchable_docs']) >= max_searchable:
                return True
        except KeyError:
            logging.debug("Refresh is not running: incorrect info_server")
        except Exception as e:
            logging.debug("Refresh is not yet running (%s)", e)
        return False

    def wait(self, timeout=None, is_ready=None):
        if not is_ready:
            if self.default_wait_function == FUSION_WAIT_SEARCHER:
                is_ready = self.is_server_running
            elif self.default_wait_function == FUSION_WAIT_MEMORY_FILLED:
                is_ready = self.is_memorysearch_filled
            else:
                is_ready = self.is_index_ready

        if timeout is None:
            timeout = self.start_timeout
        if timeout is None:
            timeout = DEFAULT_START_TIMEOUT * 3
        start = time.time()
        while time.time() - start < timeout:
            if self.process.poll() is not None:
                self._wait_coredump()
                self._process_post_mortem()
            if is_ready():
                return
            else:
                time.sleep(10)
        eh.check_failed("Failed to start Refresh. Timeout: {}".format(timeout))


def get_fusion_db_param(n=""):
    class FusionDatabase(sp.ResourceSelector):
        name = 'basesearch{}_database_resource_id'.format(n)
        description = 'Database'
        resource_type = [
            resource_types.SEARCH_DATABASE,
            resource_types.IMAGES_SEARCH_DATABASE,
            resource_types.RTYSERVER_INDEX_DIR,
        ]
        group = FUSION_DB_SOURCE_TITLE
    return FusionDatabase


def create_fusion_params(n=None, old_task=False):
    if n is None:
        n = ""

    fusion_database_param = get_fusion_db_param(n)

    class ConfigsFolder(sp.ResourceSelector):
        name = 'config_resource_id'
        description = 'Folder with Refresh Configs'
        resource_type = [saas_resources.SaasRtyserverConfigs]
        group = 'Refresh params'
        required = True

    class FusionParams(object):
        ParamsTypeName = 'FusionParams'
        Database = fusion_database_param

        class Binary(sp.ResourceSelector):
            name = 'basesearch{}_executable_resource_id'.format(n)
            description = 'Executable'
            resource_type = [
                resource_types.RTYSERVER_EXECUTABLE,
                resource_types.RTYSERVER,
            ]
            group = 'Refresh params'
            required = True

        class Config(sp.ResourceSelector):
            name = 'fusion{}_config_resource_id'.format(n)
            description = 'Refresh Config'
            resource_type = [resource_types.FUSION_SEARCH_CONFIG]
            group = 'Refresh params'
            required = True

        class StaticData(sp.ResourceSelector):
            name = 'refresh{}_static_data_resource_id'.format(n)
            description = 'Refresh Static Data'
            resource_type = [resource_types.RTYSERVER_MODELS]
            group = 'Refresh params'
            required = False

        class BasesearchConfig(sp.ResourceSelector):
            name = 'basesearch{}_config_resource_id'.format(n)
            description = 'Basesearch Config'
            resource_type = [resource_types.SEARCH_CONFIG]
            group = 'Basesearch parameters'
            required = True

        class OxygenConfig(sp.ResourceSelector):
            name = 'oxygen{}_config_resource_id'.format(n)
            description = 'OxygenOptions Config'
            resource_type = [
                resource_types.FUSION_OXYGEN_CONFIG,
                resource_types.REFRESH_TESTING_CONFIGS
            ]
            group = 'Refresh params'

        class MaxDocs(sp.SandboxIntegerParameter):
            name = "db{}_max_docs".format(n)
            description = 'Maximum number of documents in memory index (surpassing means creating disk index)'
            default_value = 10000
            group = FUSION_DB_SOURCE_TITLE

        class DbSource(sp.SandboxStringParameter):
            name = "database_source{}".format(n)
            description = "Choose the way to set Refresh database"
            group = FUSION_DB_SOURCE_TITLE

            choices = [
                ("Start with empty database", FUSION_DB_EMPTY),
                ("Use pre-created db", FUSION_DB_PRE_CREATED),
            ]
            default_value = FUSION_DB_PRE_CREATED

            sub_fields = {
                FUSION_DB_PRE_CREATED: (fusion_database_param.name,),
                FUSION_DB_EMPTY: (),
            }

        class StartTimeout(sp.SandboxIntegerParameter):
            name = 'basesearch{}_indexing_timeout'.format(n)
            description = 'Start timeout (sec)'
            group = 'Basesearch parameters'
            default_value = DEFAULT_START_TIMEOUT

        class Models(sp.ResourceSelector):
            name = 'mxnet_models{}_id'.format(n)
            description = 'MXNet models{}'.format(n)
            resource_type = [resource_types.DYNAMIC_MODELS_ARCHIVE]
            group = 'Basesearch parameters'

        params = (
            Binary, Config, StaticData, BasesearchConfig,
            DbSource, Database, MaxDocs,
            OxygenConfig, StartTimeout, Models
        )

    if not old_task:
        del FusionParams.Config
        del FusionParams.BasesearchConfig
        del FusionParams.OxygenConfig
        setattr(FusionParams, 'ConfigsFolder', ConfigsFolder)
        FusionParams.Binary.name = 'executable_resource_id'
        FusionParams.ConfigsFolder.name = 'config_resource_id'
        FusionParams.StaticData.name = 'static_data_resource_id'
        FusionParams.Models.name = 'models_archive_resource_id'
        FusionParams.Database.name = 'search_database_resource_id'
        FusionParams.params = (
            FusionParams.Binary, FusionParams.ConfigsFolder, FusionParams.StaticData, FusionParams.Models,
            FusionParams.DbSource, FusionParams.Database, FusionParams.MaxDocs,
            FusionParams.StartTimeout
        )
        if n:
            suffix = '_{}'.format(n)
            for parameter in FusionParams.params:
                parameter.name += suffix

    return FusionParams


DefaultFusionParams = create_fusion_params(old_task=True)


def create_images_fusion_params(n=None, group_name=None, component_name=None):
    group_name = group_name or ''
    component_name = component_name or ''

    if n is None:
        n = ''
    else:
        n = str(n)

    n = '_'.join([s for s in (component_name, n) if s])

    class FusionDatabase(sp.ResourceSelector):
        name = 'search_database_resource_id'
        description = 'Database'
        resource_type = [
            resource_types.SEARCH_DATABASE,
            resource_types.IMAGES_SEARCH_DATABASE,
            resource_types.RTYSERVER_INDEX_DIR,
            resource_types.RTYSERVER_SEARCH_DATABASE,
        ]
        group = group_name

    class FusionParams(object):
        ParamsTypeName = 'FusionParams'
        Database = FusionDatabase

        class ConfigsFolder(sp.ResourceSelector):
            name = 'config_resource_id'
            description = 'Folder with Refresh Configs'
            resource_type = [
                images_basesearch_resources.IMGQUICK_SAAS_RTYSERVER_CONFIGS_BUNDLE,
            ]
            group = group_name
            required = True

        class Config(ConfigsFolder):
            # For basesearch params compatibility.
            pass

        class Binary(sp.ResourceSelector):
            name = 'executable_resource_id'
            description = 'Executable'
            resource_type = [
                resource_types.RTYSERVER_EXECUTABLE,
                resource_types.RTYSERVER,
                resource_types.IMGSEARCH_RTYSERVER_EXECUTABLE,
            ]
            group = group_name
            required = True

        class StaticData(sp.ResourceSelector):
            name = 'static_data_resource_id'
            description = 'Refresh Static Data'
            resource_type = [resource_types.RTYSERVER_MODELS]
            group = group_name
            required = False

        class MaxDocs(sp.SandboxIntegerParameter):
            name = "db_max_docs"
            description = 'Maximum number of documents in memory index (surpassing means creating disk index)'
            default_value = 10000
            group = group_name

        class DbSource(sp.SandboxStringParameter):
            name = "database_source"
            description = "Choose the way to set Refresh database"
            group = group_name

            choices = [
                ("Start with empty database", FUSION_DB_EMPTY),
                ("Use pre-created db", FUSION_DB_PRE_CREATED),
            ]
            default_value = FUSION_DB_PRE_CREATED

            db_param_name = '_'.join(x for x in (FusionDatabase.name, n) if x)
            sub_fields = {
                FUSION_DB_PRE_CREATED: (db_param_name,),
                FUSION_DB_EMPTY: (),
            }

        class StartTimeout(sp.SandboxIntegerParameter):
            name = 'basesearch_indexing_timeout'
            description = 'Start timeout (sec)'
            group = group_name
            default_value = DEFAULT_START_TIMEOUT

        class Models(sp.ResourceSelector):
            name = 'models_archive_resource_id'
            description = 'MXNet models'
            resource_type = [
                resource_types.DYNAMIC_MODELS_ARCHIVE,
                images_models_resources.IMAGES_DYNAMIC_MODELS_ARCHIVE,
                images_models_resources.IMAGES_BASESEARCH_PROD_DYNAMIC_MODELS_ARCHIVE
            ]
            group = group_name

        class ArchiveModel(Models):
            # For basesearch params compatibility.
            pass

        class ShardWriterConfig(sp.ResourceSelector):
            name = 'sw_config'
            description = 'ShardWriter config'
            resource_type = resource_types.IMAGES_SHARDWRITER_CONFIG
            required = True
            group = group_name

        class Cgroup(sp.SandboxStringParameter):
            name = 'cgroup'
            description = 'Cgroup'
            group = group_name

        class FrequentKeys(sp.ResourceSelector):
            name = 'frequent_keys'
            description = 'Resource with frequent_keys.txt'
            resource_type = resource_types.OTHER_RESOURCE
            required = True
            # https://sandbox.yandex-team.ru/resource/1105170671/view
            default_value = 1105170671
            group = group_name

    FusionParams.params = (
        FusionParams.Binary, FusionParams.ConfigsFolder, FusionParams.StaticData,
        FusionParams.Models, FusionParams.DbSource,
        FusionParams.Database, FusionParams.MaxDocs, FusionParams.StartTimeout,
        FusionParams.ShardWriterConfig, FusionParams.Cgroup,
        FusionParams.FrequentKeys,
    )
    if n:
        for parameter in FusionParams.params:
            parameter.description += ' {}'.format(n)
            parameter.name += '_{}'.format(n)

    return FusionParams


def _check_attr_is_sandbox_resource(obj, attr_name):
    return isinstance(getattr(obj, attr_name), type) and issubclass(getattr(obj, attr_name), sp.SandboxParameter)


def get_fusion_diff_from_base_params(default_params, fusion_params, ignored=['Models', 'ConfigsFolder']):
    full_diff = set(dir(fusion_params)) - set(dir(default_params))
    full_diff -= set(ignored)
    subparams = [attr for attr in full_diff if _check_attr_is_sandbox_resource(fusion_params, attr)]

    class ExtraParams(object):
        pass

    final_params = []
    for attr in subparams:
        setattr(ExtraParams, attr, getattr(fusion_params, attr))
        getattr(ExtraParams, attr).required = False
        final_params.append(getattr(ExtraParams, attr))

    ExtraParams.params = tuple(final_params)
    return ExtraParams


def _set_extra_attributes_to_default_params(target_params, extra_params):
    fields_to_copy = [attr for attr in set(dir(extra_params)) if _check_attr_is_sandbox_resource(extra_params, attr)]
    for attr in fields_to_copy:
        setattr(target_params, attr, getattr(extra_params, attr))


def _duplicate_params_with_new_names(params, old2new_names):
    for old, new in old2new_names:
        setattr(params, new, getattr(params, old))


def _check_is_fusion_params(params):
    return hasattr(params, 'ParamsTypeName') and getattr(params, 'ParamsTypeName') == 'FusionParams'


def extend_basesearch_params_to_fusion_params(params, extension):
    _set_extra_attributes_to_default_params(params, extension)
    old2new_extra_names_to_duplicate = (
        ('ArchiveModel', 'Models'),
        ('Config', 'ConfigsFolder'),
    )
    _duplicate_params_with_new_names(params, old2new_extra_names_to_duplicate)


class OldFusionBinary(DefaultFusionParams.Binary):
    name = 'old_fusion_resource_id'
    description = 'Old Executable (for results comparison)'
    resource_type = [
        resource_types.RTYSERVER_EXECUTABLE,
    ]
    group = 'Refresh params'
    default_value = None


class NewFusionBinary(DefaultFusionParams.Binary):
    name = 'new_fusion_resource_id'
    description = 'New Executable (to run acceptance for)'
    resource_type = [
        resource_types.RTYSERVER_EXECUTABLE,
    ]
    group = 'Refresh params'
    default_value = None


def get_fusion_search(
    params=DefaultFusionParams,
    shard=0,
    shards_count=1,
    use_profiler=False,
    port=DEFAULT_BASESEARCH_PORT,
    component_creator=None,
    get_db=True,
    max_documents=None,
    default_wait=FUSION_WAIT_INDEX,
    event_log=False,
    use_frozen_configs=False,
    extra_params=None,
    **kwargs_ignored
):
    ctx = channel.task.ctx

    sys_info = sandboxsdk.util.system_info()
    resource = channel.sandbox.get_resource(ctx[params.Binary.name])

    if not sandboxsdk.util.is_arch_compatible(resource.arch, sys_info['arch']):
        eh.check_failed(
            'Cannot execute task. resource: {0} has incorrect arch {1} instead of {2}'.format(
                resource.id, resource.arch, sys_info['arch']
            )
        )

    binary_path = channel.task.sync_resource(ctx[params.Binary.name])

    cgroup = None
    if hasattr(params, 'Cgroup'):
        cgroup = cgroup_setup(params, ctx)

    if not get_db:
        database_path = None
    else:
        database_path = channel.task.sync_resource(ctx[params.Database.name])

    configs_folder_path = None
    old_fusion_config_path = None
    basesearch_cfg_path = None
    oxygen_config_path = None
    environment_file_path = None

    frozen_suff = ''
    if use_frozen_configs:
        frozen_suff = '-frozen'

    if hasattr(params, 'ConfigsFolder'):
        configs_folder_path = channel.task.sync_resource(ctx[params.ConfigsFolder.name])
        old_fusion_config_path = os.path.join(
            configs_folder_path, 'rtyserver.conf{}'.format(frozen_suff if use_frozen_configs else '-common')
        )
        basesearch_cfg_path = os.path.join(configs_folder_path, 'basesearch-refresh{}'.format(frozen_suff))
        oxygen_config_path = os.path.join(configs_folder_path, 'OxygenOptions.cfg{}'.format(frozen_suff))
        environment_file_path = os.path.join(configs_folder_path, 'environment{}'.format(frozen_suff))
    else:
        old_fusion_config_path = channel.task.sync_resource(ctx[params.Config.name])
        basesearch_cfg_path = channel.task.sync_resource(ctx[params.BasesearchConfig.name])
        if ctx.get(params.OxygenConfig.name, None):
            oxygen_config_path = channel.task.sync_resource(ctx[params.OxygenConfig.name])

    # SEARCH-1339
    fusion_config_path = "{}/patched_configs/instance{}.{}".format(
        paths.get_logs_folder(),
        port,
        os.path.basename(old_fusion_config_path),
    )
    paths.make_folder(os.path.dirname(fusion_config_path))
    patched_fusion_config_lines = []
    for line in fu.read_line_by_line(old_fusion_config_path):
        if 'StateRoot' in line:
            # Separate StateRoot for different instances to prevent race
            line = '    StateRoot : {}/instance{}/state'.format(paths.get_logs_folder(), port)
        patched_fusion_config_lines.append(line)

    fu.write_lines(fusion_config_path, patched_fusion_config_lines)

    static_data_path = None
    if ctx.get(params.StaticData.name, None):
        static_data_path = channel.task.sync_resource(ctx[params.StaticData.name])

    mxnet_models = None
    if ctx.get(params.Models.name, None):
        mxnet_models = channel.task.sync_resource(ctx[params.Models.name])

    sw_config = None
    if hasattr(params, 'ShardWriterConfig') and ctx.get(params.ShardWriterConfig.name, None):
        sw_config = channel.task.sync_resource(ctx[params.ShardWriterConfig.name])

    frequent_keys = None
    if hasattr(params, 'FrequentKeys') and ctx.get(params.FrequentKeys.name, None):
        frequent_keys = channel.task.sync_resource(ctx[params.FrequentKeys.name])

    return FusionSearch(
        work_dir=channel.task.abs_path(),
        binary=binary_path,
        configs_folder=configs_folder_path,
        database_dir=database_path,
        use_profiler=use_profiler,
        port=port,
        shard=shard,
        shards_count=shards_count,
        start_timeout=ctx.get(params.StartTimeout.name) or DEFAULT_START_TIMEOUT * 3,
        sw_config=sw_config,
        frequent_keys=frequent_keys,
        fusion_config_path=fusion_config_path,
        config_file=basesearch_cfg_path,
        max_documents=max_documents,
        default_wait=default_wait,
        oxygen_config=oxygen_config_path,
        mxnet_models=mxnet_models,
        event_log=event_log,
        static_data_path=static_data_path,
        environment_file_path=environment_file_path,
        extra_params=extra_params,
        cgroup=cgroup,
    )


class YMusicSearch(BasesearchBase):
    """
        Представление ymusic базового поиска в Sandbox
    """

    def __init__(self, work_dir, database_dir, **kwargs):
        config_params = {
            'Collection/WizardWorkDir': database_dir,
            'Server/ReqAnsLog': os.path.join(work_dir, 'reqans_log')
        }

        super(YMusicSearch, self).__init__(
            work_dir=work_dir,
            config_class=sconf.SearchConfig,
            database_dir=database_dir,
            config_params=config_params,
            **kwargs
        )


def get_ymusicsearch_ex(
    binary_id, config_id, database_id,
    polite_mode=True,
    port=DEFAULT_BASESEARCH_PORT
):
    """
        Синхронизировать ресурсы и получить обёртку ymusic базового поиска
        используется в старых тасках

        :return: объект YMusicSearch
    """
    start_time = time.time()
    binary_path = channel.task.sync_resource(binary_id)
    config_path = channel.task.sync_resource(config_id)
    database_path = channel.task.sync_resource(database_id)
    store_sync_time(time.time() - start_time)

    return YMusicSearch(
        work_dir=channel.task.abs_path(),
        binary=binary_path,
        database_dir=database_path,
        config_file=config_path,
        polite_mode=polite_mode,
        port=port
    )


DEFAULT_MIDDLESEARCH_PORT = 8033


class QueryCacheDirPatcher(cfgproc.ParamPatcherByGroup):
    def __init__(self, config_params):
        self.config_params = config_params

    def patch_param(self, param_name, old_value, group_values):
        return self.config_params.get(param_name, old_value)


class QueryCacheMemoryLimitPatcher(cfgproc.ParamPatcherByGroup):
    def __init__(self):
        pass

    def patch_param(self, param_name, old_value, group_values):
        if "Dir" in group_values and param_name == "MemoryLimit":
            return None
        else:
            return old_value


class Middlesearch(StandardSearch):
    """
        Представление метапоиска в Sandbox
        (int, middlesearch, базовый класс для upper)
    """
    name = 'middlesearch'

    _LOAD_LOG_FILENAME = 'middlesearch{}_load.log'
    _SERVER_LOG_FILENAME = 'middlesearch%s_server.log'
    _EVENT_LOG_FILENAME = 'middlesearch%s_event.log'

    def __init__(
        self, is_int, work_dir, binary,
        config_class=None,
        config_file=None,
        pure_dir=None,
        rearrange_dir=None,
        rearrange_index_dir=None,
        rearrange_dynamic_dir=None,
        rearrange_data_fast_dir=None,
        archive_model_path=None,
        event_log=None,
        load_log=None,
        basesearches=None,
        patch_queues_size=True,
        disable_timeouts=False,
        sending_timeout=None,
        port=DEFAULT_MIDDLESEARCH_PORT,
        start_timeout=DEFAULT_START_TIMEOUT,
        shutdown_timeout=DEFAULT_SHUTDOWN_TIMEOUT,
        server_input_deadline=None,
        query_cache=None,
        use_profiler=False,
        use_gperftools=False,
        set_n_groups_for_source_multiplier=False,
        set_max_snippets_per_request=0,
        max_allfactors_per_bs=None,
        disable_client_timeout=False,
        neh_cache=None,
        use_async_search=None,
        reset_memory_limit=True,
        config_params=None,
        apphost_mode=False,
        patch_port_in_cfg=True,
        **kwargs
    ):
        if not config_params:
            config_params = {}

        ms_banner = bb2.Banners.WebMiddleSearch
        _BUG_BANNER.show_ad(
            component=ms_banner.component,
            st_component_ids=ms_banner.st_component_ids,
            responsibles=ms_banner.responsibles,
        )

        self._event_log_const_path = event_log

        if load_log is None:
            load_log_dir = os.path.join(work_dir, self._LOAD_LOG_FILENAME.format(port))
        else:
            load_log_dir = load_log

        if query_cache is None:
            self.query_cache_dir = os.path.join(work_dir, 'cache_{}'.format(port))
        else:
            eh.ensure(
                cfgproc.TMP_DIR_PATTERN not in query_cache,
                "TMP_DIR_PATTERN should not be used in cache path"
            )
            self.query_cache_dir = query_cache
        paths.make_folder(self.query_cache_dir)
        self.query_cache_dir = os.path.join(self.query_cache_dir, cfgproc.TMP_DIR_PATTERN)

        config_params.update({
            'Server/LoadLog': load_log_dir,
            'Server/ServerLog': os.path.join(work_dir, self._SERVER_LOG_FILENAME % port),
            'Server/LogExceptions': 'True',
            'Collection/UseRequiredBaseTimestamp': 'False',
            'Collection/IndexDir': pure_dir,
            'Collection/RearrangeDataDir': rearrange_dir,
            'Collection/RearrangeIndexDir': rearrange_index_dir,
            'Collection/RearrangeDynamicDataDir': rearrange_dynamic_dir,
            'Collection/RearrangeDataFastDir': rearrange_data_fast_dir,
            'Collection/MXNetFile': archive_model_path,
            'Collection/NehCache': neh_cache,
            'Collection/RequestTimeoutLimit': "40s",  # SEARCH-5816
            'Collection/AbortTout': '600s',  # prod/hamster 60s is sometimes too low for sanitizers
            'Server/ErrorBoosterUnifiedAgentLog/Uri': None,
        })

        query_cache_dir_patcher_config = {
            'Dir': self.query_cache_dir,
        }

        self.apphost_mode = apphost_mode
        if patch_port_in_cfg:
            config_params["Server/Port"] = str(port)
        if apphost_mode:
            config_params["Server/AppHostOptions"] = "Port={}, Threads=24, GrpcPort=+2, GrpcThreads=24".format(port + 1)

        if server_input_deadline:
            config_params['Server/NehLimits'] = 'Http2ServerInputDeadline={}'.format(server_input_deadline)

        self.environment = {
            'NOSIGHANDLER': 'yes',
        }

        if reset_memory_limit:
            config_params['Collection/QueryCache'] = QueryCacheMemoryLimitPatcher()

        if set_n_groups_for_source_multiplier:
            config_params['Collection/NGroupsForSourceMultiplier'] = 5

        if max_allfactors_per_bs:
            config_params['Collection/MaxAllfactorsPerBasesearch'] = max_allfactors_per_bs

        if set_max_snippets_per_request:
            config_params['Collection/MaxSnippetsPerRequest'] = set_max_snippets_per_request

        if patch_queues_size:
            config_params.update({
                'Server/Threads': 100,
                'Server/QueueSize': 100,
                'Collection/RequestThreads': 100,
                'Collection/RequestQueueSize': 100
            })

        if disable_timeouts:
            config_params.update({
                'Collection/ScatterOptions/TimeoutTable': '300s',
                'Collection/ScatterOptions/MessengerOptions': None,
                'Collection/ScatterOptions/RemoteTimeoutOptions': None,
                'Collection/ConnectTimeout': '300s',
                'Collection/MessengerOptions': None,
                'Collection/UseTimeoutTable': None,
            })

        if sending_timeout is not None:
            config_params.update({
                'Collection/SendingTimeout': sending_timeout,
            })

        if disable_client_timeout:
            config_params.update({
                'Server/ClientTimeout': 0,
            })

        if use_async_search is False:
            # we do not want to spoil default (production) config values
            # so we turn off snippets reask only when async search was explicitly disabled
            config_params.update({
                'Collection/AskSnippetsTwice': False,  # SEARCH-1022
            })

        config_params['Collection/QueryCache'] = QueryCacheDirPatcher(query_cache_dir_patcher_config)

        if config_class is None:
            if is_int:
                config_class = sconf.MidlesearchIntConfig
            else:
                config_class = sconf.MidlesearchMetaConfig

        super(Middlesearch, self).__init__(
            work_dir=work_dir,
            binary=binary,
            port=port,
            start_timeout=start_timeout,
            shutdown_timeout=shutdown_timeout,
            config_class=config_class,
            config_file=config_file,
            config_params=config_params,
            use_profiler=use_profiler,
            use_gperftools=use_gperftools,
            **kwargs
        )

        if basesearches:
            self.config.replace_basesearches(basesearches)

        if use_async_search is not None:
            if use_async_search:
                self.config.enable_async_search()
            else:
                self.config.disable_async_search()

        self.messenger_enabled = False
        # see code enabling listener in metasearch.cpp
        if (
            "MessengerOptions" in self.config.text and
            (
                "Listen" in self.config.text or
                "Downstream" in self.config.text or
                "Upstream" in self.config.text
            )
        ):
            self.messenger_enabled = True

    @property
    def _common_test_urls(self):
        return [
            {
                "url": "/{}?info=getconfig".format(self.http_collection),
                "expected_code": httplib.OK,
                "expected_re": [
                    re.compile('.*<Collection.*id="yandsearch".*'),
                    re.compile('.*<Collection.*autostart="must".*'),
                    re.compile('.*<Collection.*meta="yes".*'),
                ]
            },
            {
                "url": "/admin?action=clearcache&id={}".format(self.http_collection),
                "expected_code": httplib.OK,
                "expected_re": [re.compile('^Cache is cleared with code 0')]
            },
            {
                "url": "/admin?action=clearcache111&id={}".format(self.http_collection),
                "expected_code": httplib.OK,
                "expected_re": [re.compile('<html>.*')]
            },
            {
                "url": "/admin?action=clearcache",
                "expected_code": httplib.OK,
                "expected_re": [re.compile('^Cache is cleared with code 1')]
            },
            {
                "url": "/admin?action=clearcache&id=id_that_does_not_exist",
                "expected_code": httplib.OK,
                "expected_re": [re.compile('^Cache is cleared with code 1')]
            }
        ]

    def set_cache_thread_limit(self, thread_count=32):
        # For SEARCH-1474
        logging.info("Try to patch CacheThreadLimit")
        self.config.apply_local_patch({'Collection/CacheThreadLimit': thread_count})

    def disable_hybrid_cache(self):
        logging.info("Try to disable hybrid cache")
        self.config.apply_local_patch({'Collection/QueryCache/ExclusiveHybridCache': 'no'})

    def invalidate_cache(self):
        logging.info("try to invalidate cache")
        self.fetch("/admin?action=droplock")
        self.fetch("/admin?action=prefetch0")

    def disable_cache(self):
        self.config.disable_cache()

    def clear_cache(self):
        logging.info("try to clear cache")
        cache_dir = self.config.get_parameter("Collection/QueryCache/Dir")
        if self.is_running():
            logging.info("WARNING! Search is running, cannot clear cache")
        else:
            paths.remove_path(cache_dir)

    def warmup_request(self):
        """
            Чтобы победить SEARCH-616 (мигание в тестах атрибутов IndexGeneration
            и SourceTimestamp), задём один запрос, который гарантированно пойдёт на все источники.
        """
        fail_count = 0
        try_count = 10
        failures = ""
        timeout_sec = 20  # 20 seconds
        timeout_usec = timeout_sec * 1000000
        for attempt in six.moves.xrange(0, try_count):
            # В лучших традициях, надо задать запрос несколько раз
            # потому что есть проблемы типа SEARCH-912
            try:
                # запрос подбираем максимально быстро отвечающий
                # такой чтобы в идеале ничего не находилось (SEARCH-1202)
                # UPD пять лет спустя: подокументный индекс таким образом не прогревается,
                # начинаются флапы на сложных запросах. Пробуем, наоборот,
                # поднять побольше кишок и документов.
                url = (
                    "http://127.0.0.1:{}/{}?"

                    "&text=1%3A%3A798+%26/%28-3+3%29+2%3A%3A798+%26/%28-3+3%29+3%3A%3A798+%26/%28-3+3%29"
                    "+4%3A%3A798+%26/%28-3+3%29+5%3A%3A798+%26/%28-3+3%29+6%3A%3A798+%26/%28-3+3%29"
                    "+%287%3A%3A798+^+%D1%81%D0%B5%D0%BC%D0%B5%D1%80%D0%BA%D0%B0%3A%3A53621806%29+%26/%28-3+3%29"
                    "+8%3A%3A798+%26/%28-3+3%29+9%3A%3A798+%26/%28-3+3%29+0%3A%3A798+softness%3A6"

                    "&user_request=1+2+3+4+5+6+7+8+9+0"
                    "&ms=proto"
                    "&qtree=cHicvdS9i9RAFADw9ybZ3DgXlpDj4Ei1xGa0mk32I2slYnGIynGVXCWHwtpeJQvKHiKIgmh7p"
                    "WLpIcct6LF-IFeI1WxvbeVfYOVkslmGEELSmCZv3mTeb3iBx24wl7Y8Zws6yIkAHwII4TJEcMWl4IHK"
                    "AwcBV1vbrR24A3edMb5COEJ4g3CC8BlBPT8QJN4LHrGbLDtF1am0GnYD7Kzqoa6Huh5sQ1pvPHfzctg"
                    "t1BOQ4LWLlHgQYDdcF6uny88tgfo6MMYJTBFUkeAxu1Xko5THTj0_qvAj04-4tOv5cRM_rvBj04_575"
                    "p-r4nfq_B7pt_jp049v9_E71f4fdPv8z81_UETf1DhD0x_wM8v1POHTfxhhT80_SE_YvX8pImfVPiJ6"
                    "Sf8W01_1MQfVfgj0x_xuVvil4wfoaZZXV5U8MLkBf9bMn7CbI6SjsVb4r-v0svufSf0cDrFTYDJ9eA9"
                    "YQ8K7WgvDuVMzuVsMZVn8kOAl0Lky-asGbM-b84v-TrO21M4W9arGaYtDgpfhsBRdestydp18AJZ4Ys"
                    "trSawaz37-nOPvDved1QZ4Bv7VL2JnHNrGX3UEQnI4mkeyZmOcLnLVGTLL_JTnl28XJ15oveJ3j_j1h"
                    "ge4oRQXP4-m7qey221eo4bzFaNI_46pbtrFH3r9s79LGt7TkmWem5Jtu35hayr6zq-2juAbEm9trn01"
                    "Bm9VFf6B6vLQQM%2C"
                    "&relev=wizqbundle%3DehRMWi40AQAAAACAcwQB8jIKUypGehRMWi40AQAAAACANAABdRI0Eh0SCzAB"
                    "APAZMRgAIAAqBQoBMRAAMgMQxAMgACoBMTIAOAJAAEoDEIIEUhUA7wAAADC_xM_DvOzf86EBVQAMFDJV"
                    "ABEyVQAg0ARVABUyVQAykgVSFQAAVQCf4MTcvcuzpOq3VQANFDNVABEzVQAR5lUAFTNVABCsVQAAFQAA"
                    "VQCfosP07cCjxfaeVQANFDRVABE0VQAgtAaqABU0VQAyqgdSFQAAVQDPnMiB5_OfzMehAQpSVAEKFDVV"
                    "ABE1VQAR7FUAFTVVADKiCFIVAABVAJ_7kPiKx57puEZUAAwUNlQAETZUACDECakAFTZUACCEDKgBEglU"
                    "AI-L8I261e_mz1QADRQ3VAARN1QAIJoKVAAVN1QAEPpUAAAVAACoAJ_A6f_TuJOm5AqoAAwUOFQAEThU"
                    "ABHAVAAVOFQAMq4NUhUAAFQAn-Ke4v-mirC9TU8CDBQ5VAAROVQAILwMqAAVOVQAMoQQUhUAAFQAkIXc"
                    "1P7ehqjM8aUBGEX5AkYzAAF2-QIUGFQAETBUABH6-AIVMFQAEMROAgAVAABUAPgBz86spOiK69LMAQqq"
                    "ASqcAVYA8H2KAAH7DhKZAhLuARIO0YHQtdC80LXRgNC60LAYACABKhIKFgBqEAEqFAoQKgCM0LwQACoW"
                    "ChIWAE7QuBAALgBN0YUQAFgAH7UUAAAPPgABP77QuVQAAD6-0Y5AAC3RgxQA-wC-0LoQADIEEIakXSCj"
                    "AirbAPAIMgIIATgBQABKBRCGysQCUgQQhqRdIOADoDDj8KOV7eT567cBAQ_6AxQfAfoDDZ_Wzs6M6Mrf"
                    "_lz-AQwH-QMAVAAP-QMKn5uR0MDmh7vYDv4BDAf4AwBUAADjAw_4AwavjaTK_t-dpuLjAVUADAf4AwBV"
                    "AA_4Awqfg8DsuuDioeXLVQANB_gDAFUAD_gDCp_krum3ns7I7NZVAA0H-QMAVQAP-QMKn4PMwOThvujo"
                    "jlUADQf6AwBVAADlAw_6AwafzsKxoZ_vu8-fUQINB_sDAFUAD_sDCp-vsO2B8ovGgFaoAQwH-wMAVAAP"
                    "-wMKkMPGoouY0O_Q7qkAD_sDEwBUAADmAw_7AwbwB5qpyOertNb56gESpQIIChAAGg8KBSqJAFASAggA"
                    "GgQABxEAIAEaBAAHEQAgAhoEAAcRACADGgQABxEAIAQaBAAHEQAgBRoEAAcRACAGGgQABxEAIAcaBAAH"
                    "EQAgCBoEAAcRACAJGgQAsCEKFwgBEIGAgAEhGgTwAABAj0AqBRDc0JEzEgIIClYAoC0AAIA_MgMaATEF"
                    "ABAyBQAQMwUAEDQFABA1BQAQNgUAEDcFABA4BQAQOQUA8RswQhsIAQgCCAQICAgQCCAIQAiAAQiAAgiA"
                    "BBACGAoSbQgKEDQaCBICCAsgAQAKABEMGQEACgARDRIBAAoAEQ4LAQAKABEPBAEACgAREP0AAAoAEBGg"
                    "AAFGABES7wAAFAARE-gAAAoAoBQaAggJLQAAgD8AAAA%2C"
                    "&hr=da"
                    "&nocache_completely=da"
                    "&pron=tiermask3"
                    "&rearr=qlang%3Dnon_words"
                    "&rearr=all_off"  # SEARCH-1142
                    "&timeout={}"  # SEARCH-1186
                    "&waitall=da"
                ).format(self.port, self.http_collection, timeout_usec)
                logging.info("Fetch IndexGeneration warmup request, try %d, url: %s", attempt, url)

                # Верим, что за 80 секунд уж точно прочухается (30 сек было мало)
                start = time.time()
                urllib2.urlopen(url, timeout=80).read()
                answer_time = round(time.time() - start)
                logging.info(
                    "Warmup request attempt {} for search component {} [{}] took {} seconds".format(
                        attempt, self.name, self.process.pid, answer_time
                    )
                )
                if answer_time < timeout_sec / 10:
                    logging.info(
                        "Warmup request attempt {} answer was short enough, seems it warmed up".format(attempt)
                    )
                    break

            except Exception as e:
                failures += "Try {}, error: {}\n".format(attempt, str(e))
                fail_count += 1
                # wait some time (prefetching)
                time.sleep(20)

        eh.ensure(
            fail_count < try_count,
            "Cannot answer to IndexGeneration warmup request, "
            "tried {} times. Errors: {}".format(try_count, failures)
        )

    def _get_run_cmd(self, config_path):
        return ['-d', '-p', self.port, config_path]

    def get_environment(self):
        return self.environment

    def get_event_log_path(self):
        return self.event_log_path

    def _generate_config(self):
        self._generate_eventlog_path()
        return super(Middlesearch, self)._generate_config()

    def _generate_eventlog_path(self):
        if self._event_log_const_path:
            self.event_log_path = self._event_log_const_path
        else:
            evlog_dir = utils.create_misc_resource_and_dir(
                channel.task.ctx,
                'eventlog_misc_resource_id',
                'eventlogs',
                'eventlogs'
            )
            self.event_log_path = paths.get_unique_file_name(evlog_dir, self._EVENT_LOG_FILENAME % self.port)

        config_params = {'Server/EventLog': self.event_log_path}
        self.config.apply_local_patch(config_params)

    def verify_stderr(self, file_name):
        if not os.path.exists(file_name):
            return

        with open(file_name) as errors:
            for line in errors:
                line = line.strip()

                if not line:
                    continue

                if "[ERROR]" not in line:
                    continue

                if 'requestTimestamp <= LastUpdTimestamp' in line:
                    continue

                if "Cannot lock" in line:
                    continue

                if "frequentTermsPath" in line:
                    continue

                if "Failed initialization of rearrange rule variants: Panther" in line:
                    continue

                if "Wrong factor found:" in line:
                    continue

                if "Block pool exhausted" in line:
                    continue

                eh.check_failed("Unexpected stderr log line: '{}'".format(line))


def get_middlesearch_ex(
    binary_id, config_id,
    data_id=None,
    is_int=False,
    basesearches=None,
    index_id=None,
    archive_model_id=None,
    event_log=None,
    load_log=None,
    patch_queues_size=True,
    disable_timeouts=False,
    sending_timeout=None,
    port=DEFAULT_MIDDLESEARCH_PORT,
    start_timeout=DEFAULT_START_TIMEOUT,
    shutdown_timeout=DEFAULT_SHUTDOWN_TIMEOUT,
    query_cache=None,
    use_profiler=False,
    use_gperftools=False,
    set_n_groups_for_source_multiplier=False,
    set_max_snippets_per_request=0,
    max_allfactors_per_bs=None,
    disable_client_timeout=False,
    use_async_search=None,
    work_dir=None,
    reset_memory_limit=True,
    apphost_mode=False,
    patch_port_in_cfg=True,
    **kwargs
):
    """
        Синхронизировать ресурсы и получить обёртку промежуточного поиска
    """
    start_time = time.time()
    binary_path = channel.task.sync_resource(binary_id)
    config_path = channel.task.sync_resource(config_id)

    if data_id:
        data_res = channel.sandbox.get_resource(data_id)
        logging.debug("Data resource type = %s", data_res.type)
        data_path = channel.task.sync_resource(data_id)
        pure_dir = os.path.join(data_path, "pure")
        if data_res.type == resource_types.QUICK_REARRANGE_RULES_DATA:
            logging.info("Data resource is for quick. Use it as rearrange folder")
            rearrange_dir = data_path
        else:
            rearrange_dir = os.path.join(data_path, "rearrange")
    else:
        pure_dir = None
        rearrange_dir = None

    if index_id:
        rearrange_index_dir = channel.task.sync_resource(index_id)
    else:
        rearrange_index_dir = None

    archive_model_path = channel.task.sync_resource(archive_model_id) if archive_model_id else None
    store_sync_time(time.time() - start_time)

    return Middlesearch(
        is_int=is_int,
        work_dir=work_dir or channel.task.abs_path(),
        binary=binary_path,
        config_file=config_path,
        pure_dir=pure_dir,
        rearrange_dir=rearrange_dir,
        rearrange_index_dir=rearrange_index_dir,
        archive_model_path=archive_model_path,
        basesearches=basesearches,
        port=port,
        start_timeout=start_timeout,
        shutdown_timeout=shutdown_timeout,
        event_log=event_log,
        load_log=load_log,
        query_cache=query_cache,
        patch_queues_size=patch_queues_size,
        disable_timeouts=disable_timeouts,
        sending_timeout=sending_timeout,
        use_profiler=use_profiler,
        use_gperftools=use_gperftools,
        set_n_groups_for_source_multiplier=set_n_groups_for_source_multiplier,
        set_max_snippets_per_request=set_max_snippets_per_request,
        max_allfactors_per_bs=max_allfactors_per_bs,
        disable_client_timeout=disable_client_timeout,
        use_async_search=use_async_search,
        reset_memory_limit=reset_memory_limit,
        apphost_mode=apphost_mode,
        patch_port_in_cfg=patch_port_in_cfg,
        **kwargs
    )


def create_middlesearch_params(
    n=None,
    with_evlogdump=False,
    with_config=True,
    group_name=None,
    component_name=None,
    use_int=True,
    required_hack=True,
):
    n = "" if n is None else "_{}".format(n)

    if group_name is None:
        group_name = 'Middlesearch{} parameters'.format(n)

    if component_name is None:
        component_name = 'middlesearch{}'.format(n)

    class Params(object):
        class Binary(sp.ResourceSelector):
            name = '{}_executable_resource_id'.format(component_name)
            description = 'Executable'
            resource_type = [
                middle_resources.IntsearchExecutable,
                middle_resources.L1IntsearchExecutable,
                middle_resources.RankingMiddlesearchExecutable,
                resource_types.FRESH_RANKING_MIDDLESEARCH_EXECUTABLE,
                resource_types.IMAGES_MIDDLESEARCH_EXECUTABLE,
                resource_types.DIRECTMETASEARCH_EXECUTABLE,
                resource_types.VIDEO_RANKING_MIDDLESEARCH_EXECUTABLE,
                resource_types.MISC_MIDDLESEARCH_EXECUTABLE,
                resource_types.GEOMETASEARCH_EXECUTABLE,
                upper_resources.NoapacheUpper,
            ]
            group = group_name
            required = required_hack

        class Config(sp.ResourceSelector):
            name = '{}_config_resource_id'.format(component_name)
            description = 'Config'
            resource_type = [
                resource_types.MIDDLESEARCH_CONFIG,
                images_metasearch_resources.IMAGES_MIDDLESEARCH_CONFIG,
                resource_types.VIDEO_MIDDLESEARCH_CONFIG,
                resource_types.GEOMETASEARCH_CONFIG
            ]
            group = group_name
            required = required_hack

        class Data(sp.ResourceSelector):
            name = '{}_data_resource_id'.format(component_name)
            description = 'Rearrange data'
            resource_type = [
                resource_types.MIDDLESEARCH_DATA,
                images_metasearch_resources.IMAGES_MIDDLESEARCH_DATA,
                resource_types.VIDEO_MIDDLESEARCH_DATA,
                resource_types.QUICK_REARRANGE_RULES_DATA,
                resource_types.MIDDLESEARCH_GEO_DATA,
            ]
            group = group_name

        class Index(sp.ResourceSelector):
            name = '{}_index_resource_id'.format(component_name)
            description = 'Rearrange index'
            resource_type = [
                resource_types.IMAGES_MIDDLESEARCH_INDEX,
                resource_types.VIDEO_MIDDLESEARCH_INDEX
            ]
            group = group_name

        class ArchiveModel(sp.ResourceSelector):
            name = '{}_models_archive_resource_id'.format(component_name)
            description = 'Models archive'
            resource_type = [
                resource_types.DYNAMIC_MODELS_ARCHIVE,
                images_models_resources.IMAGES_MIDDLESEARCH_DYNAMIC_MODELS_ARCHIVE,
                images_models_resources.IMAGES_MIDDLESEARCH_PROD_DYNAMIC_MODELS_ARCHIVE,
                resource_types.VIDEO_MIDDLE_DYNAMIC_MODELS_ARCHIVE,
                "DYNAMIC_MODELS_ARCHIVE_L1",
                images_models_resources.IMAGES_L1_DYNAMIC_MODELS_ARCHIVE,
                video_resources.VIDEO_L1_DYNAMIC_MODELS_ARCHIVE,
            ]
            group = group_name

        class Evlogdump(sp.ResourceSelector):
            name = '{}_evlogdump_resource_id'.format(component_name)
            description = 'Evlogdump'
            resource_type = resource_types.EVLOGDUMP_EXECUTABLE
            group = group_name
            required = False

        class UseInt(sp.SandboxBoolParameter):
            name = '{}_is_int'.format(component_name)
            description = 'Is int'
            group = group_name
            default_value = False

        class ApphostMode(sp.SandboxBoolParameter):
            name = '{}_apphost_mode'.format(component_name)
            description = 'Apphost mode'
            group = group_name
            default_value = False

        class MaxSnippetsPerRequest(sp.SandboxIntegerParameter):
            name = '{}_set_max_snippets_per_request'.format(component_name)
            description = 'Max snippets per request'
            group = group_name
            default_value = 0

        class StartTimeout(sp.SandboxIntegerParameter):
            name = '{}_start_timeout'.format(component_name)
            description = 'Start timeout (sec)'
            group = group_name
            default_value = DEFAULT_START_TIMEOUT

        class SendingTimeout(sp.SandboxStringParameter):
            name = '{}_sending_timeout'.format(component_name)
            description = 'Set SendingTimeout (e.g 100ms) (see SEARCH-988)'
            group = group_name

        class VerifyStderr(VerifyStderrCommon):
            name = '{}_verify_stderr'.format(component_name)
            group = group_name

        class Cgroup(sp.SandboxStringParameter):
            name = '{}_cgroup'.format(component_name)
            description = 'Cgroup'
            group = group_name

        params = (
            Binary,
            Data,
            Index,
            ArchiveModel,
            MaxSnippetsPerRequest,
            StartTimeout,
            SendingTimeout,
            Cgroup,
            ApphostMode,
        )

        if use_int:
            params = (UseInt,) + params
        if with_config:
            params = params + (Config, )
        if with_evlogdump:
            params = params + (Evlogdump,)
        params = params + (VerifyStderr,)  # this parameter should be the last because of different group names

    return Params


DefaultMiddlesearchParams = create_middlesearch_params()


def get_middlesearch(
    basesearches=None,
    params=DefaultMiddlesearchParams,
    patch_queues_size=True,
    disable_timeouts=False,
    port=DEFAULT_MIDDLESEARCH_PORT,
    event_log=None,
    load_log=None,
    query_cache=None,
    use_profiler=False,
    use_gperftools=False,
    set_n_groups_for_source_multiplier=False,
    max_allfactors_per_bs=None,
    disable_client_timeout=False,
    use_async_search=None,
    work_dir=None,
    reset_memory_limit=True,
    shutdown_timeout=DEFAULT_SHUTDOWN_TIMEOUT,
    patch_port_in_cfg=True,  # SEARCH-5727
):
    """
        Получить обёртку промежуточного поиска - объект класса Middlesearch
        Идентификаторы параметров-ресурсов берутся из контекста таска
        Ресурсы синхронизируются
    """

    ctx = channel.task.ctx
    return get_middlesearch_ex(
        is_int=utils.get_or_default(ctx, params.UseInt),
        binary_id=ctx[params.Binary.name],
        config_id=ctx[params.Config.name],
        data_id=ctx[params.Data.name],
        index_id=ctx.get(params.Index.name),
        archive_model_id=ctx[params.ArchiveModel.name],
        basesearches=basesearches,
        patch_queues_size=patch_queues_size,
        disable_timeouts=disable_timeouts,
        sending_timeout=utils.get_or_default(ctx, params.SendingTimeout),
        port=port,
        start_timeout=utils.get_or_default(ctx, params.StartTimeout),
        event_log=event_log,
        load_log=load_log,
        query_cache=query_cache,
        use_profiler=use_profiler,
        use_gperftools=use_gperftools,
        set_n_groups_for_source_multiplier=set_n_groups_for_source_multiplier,
        set_max_snippets_per_request=utils.get_or_default(ctx, params.MaxSnippetsPerRequest),
        max_allfactors_per_bs=max_allfactors_per_bs,
        disable_client_timeout=disable_client_timeout,
        use_async_search=use_async_search,
        work_dir=work_dir,
        use_verify_stderr=utils.get_or_default(ctx, params.VerifyStderr),
        reset_memory_limit=reset_memory_limit,
        shutdown_timeout=shutdown_timeout,
        apphost_mode=utils.get_or_default(ctx, params.ApphostMode),
        patch_port_in_cfg=patch_port_in_cfg,
    )


DEFAULT_THUMB_DAEMON_PORT = 15579


class ThumbDaemon(StandardSearch):
    """
        Представление thumb daemon в Sandbox
    """

    name = 'thumb_daemon'

    def __init__(
        self,
        work_dir,
        binary,
        database_dir,
        config_file,
        its_config_file=None,
        port=DEFAULT_THUMB_DAEMON_PORT,
        version=None,
        use_profiler=False,
        use_gperftools=False,
        database_writeable=False,
        **kwargs
    ):
        self._check_if_exists(database_dir, 'thumb daemon database')
        self.database_dir = database_dir
        self.database_writeable = database_writeable
        self.version = version
        self.its_config_file = its_config_file

        super(ThumbDaemon, self).__init__(
            work_dir=work_dir,
            binary=binary,
            port=port,
            config_class=sconf.SearchConfig,
            config_file=config_file,
            config_params={},
            use_profiler=use_profiler,
            use_gperftools=use_gperftools,
            **kwargs
        )

    # used in some initalizations (daemon have incompatible config format)
    def replace_config_parameter(self, *args, **kwargs):
        pass

    def _get_run_cmd(self, config_path):
        db_path = thumbdaemon_utils.thdb_path(self.database_dir)
        db_arg = '-d' if self.database_writeable else '-m'
        cmd = ['-p', str(self.port), '-C', config_path, db_arg, db_path]
        if self.version is not None:
            cmd.extend(['-v', self.version])
        if self.its_config_file is not None:
            cmd.extend(['--dynamic-config', self.its_config_file])
        return cmd


def create_thumbdaemon_params(n="", database_required=True, group_name=None, component_name=None):
    if group_name is None:
        group_name = "Thumbdaemon{} parameters".format(n)

    if component_name is None:
        component_name = "thumb_daemon{}".format(n)

    class Parameters:
        class Binary(sp.ResourceSelector):
            name = "{}_executable_resource_id".format(component_name)
            description = 'Executable'
            resource_type = [images_daemons_resources.NAIL_DAEMON_EXECUTABLE]
            group = group_name
            required = True

        class Config(sp.ResourceSelector):
            name = "{}_config_resource_id".format(component_name)
            description = 'Config'
            resource_type = [images_daemons_resources.NAIL_DAEMON_CONFIG]
            group = group_name
            required = True

        class Database(sp.ResourceSelector):
            name = "{}_database_resource_id".format(component_name)
            description = 'Database (index name should be "thdb")'
            resource_type = [
                resource_types.THUMB_DAEMON_DATABASE,
                resource_types.VIDEO_ULTRA_THUMB_DATABASE,
            ]
            group = group_name
            required = database_required

        class StartTimeout(sp.SandboxIntegerParameter):
            name = "{}_start_timeout".format(component_name)
            description = 'Start timeout (sec)'
            group = group_name
            default_value = DEFAULT_START_TIMEOUT

        params = (Binary, Config, Database, StartTimeout)

    return Parameters


DefaultThumbDaemonParameters = create_thumbdaemon_params()


def get_thumb_daemon(
    params=DefaultThumbDaemonParameters,
    port=DEFAULT_THUMB_DAEMON_PORT,
    version=None,
    use_profiler=False,
    use_gperftools=False,
    database_dir=None,
    its_config_file=None,
    database_writeable=False,
    **kwargs
):
    """
        Получить обёртку thumb daemon - объект класса ThumbDaemon
        Идентификаторы параметров-ресурсов берутся из контекста таска
        Ресурсы синхронизируются
    """

    ctx = channel.task.ctx

    binary_id = ctx[params.Binary.name]
    binary_path = channel.task.sync_resource(binary_id)

    config_id = ctx[params.Config.name]
    config_path = channel.task.sync_resource(config_id)

    if database_dir is None:
        database_id = ctx[params.Database.name]
        database_dir = channel.task.sync_resource(database_id)

    return ThumbDaemon(
        work_dir=channel.task.abs_path(),
        binary=binary_path,
        database_dir=database_dir,
        config_file=config_path,
        port=port,
        its_config_file=its_config_file,
        version=version,
        use_profiler=use_profiler,
        use_gperftools=use_gperftools,
        database_writeable=database_writeable,
        **kwargs
    )


class DefaultUpperSearchParams:
    class ReportCoreParameter(sp.LastReleasedResource):
        name = 'report'
        description = 'Report core:'
        resource_type = [
            resource_types.REPORT_CORE_PACKAGE,
            resource_types.REPORT_NEWS_CORE_PACKAGE,
            resource_types.REPORT_IMAGES_CORE_PACKAGE,
            resource_types.REPORT_VIDEO_CORE_PACKAGE,
            resource_types.REPORT_YACA_CORE_PACKAGE,
        ]

    class ReportTemplatesParameter(sp.LastReleasedResource):
        name = 'report_templates'
        description = 'Report templates:'
        resource_type = resource_types.REPORT_TEMPLATES_PACKAGE

    class RearrangeBundleParameter(sp.LastReleasedResource):
        name = 'rearrange_bundle'
        description = 'rearrange bundle:'
        resource_type = resource_types.REARRANGE_BUNDLE

    class UseCproxyParameter(sp.SandboxBoolParameter):
        name = 'use_cproxy'
        description = 'Use cproxy'

    class UseKtraceParameter(sp.SandboxBoolParameter):
        name = 'use_ktrace'
        description = 'Use ktrace'

    class PerlConfParameter(sp.SandboxStringParameter):
        name = 'perl_conf'
        description = 'Add options to .perl.conf (e.g.: IS_LAS=1, IS_AMS=1})'

    class GeneratorConfParameter(sp.SandboxStringParameter):
        name = 'generator_conf'
        description = 'Add options to config generator (e.g.: IS_LAS:1, IS_AMS=1)'

    class SaveLogsParameter(sp.SandboxBoolParameter):
        name = 'save_logs'
        description = 'Save logs as resource'

    class StartTimeoutParameter(sp.SandboxIntegerParameter):
        name = 'start_timeout'
        description = 'Wait upper to start'
        default_value = 5 * 60

    params = (
        ReportCoreParameter,
        ReportTemplatesParameter,
        RearrangeBundleParameter,
        UseCproxyParameter,
        UseKtraceParameter,
        SaveLogsParameter,
        PerlConfParameter,
        GeneratorConfParameter,
        StartTimeoutParameter
    )


_BALANCER_GROUP_NAME = 'Balancer parameters'


class DefaultBalancerParams:
    class ExecutableResourceParameter(sp.LastReleasedResource):
        name = 'balancer_executable_resource_id'
        description = 'Executable'
        resource_type = balancer_resources.BALANCER_EXECUTABLE
        group = _BALANCER_GROUP_NAME

    class ConfigResourceParameter(sp.ResourceSelector):
        name = 'balancer_config_resource_id'
        description = 'Config'
        resource_type = [
            balancer_resources.BALANCER_ADDRS_CONFIG,
            balancer_resources.BALANCER_WEB_CONFIG,
            balancer_resources.BALANCER_BARNAVIG_CONFIG,
            balancer_resources.BALANCER_BLOG_SEARCH_CONFIG,
            balancer_resources.BALANCER_CLICKS_CONFIG,
            balancer_resources.BALANCER_CONQUISTA_CONFIG,
            balancer_resources.BALANCER_LYRICSLOVER_CONFIG,
            balancer_resources.BALANCER_MUSIC_MIC_SEARCH_CONFIG,
            balancer_resources.BALANCER_NEWS_CONFIG,
            balancer_resources.BALANCER_OPENSMETANKA_CONFIG,
            balancer_resources.BALANCER_REQWIZARD_CONFIG,
            balancer_resources.BALANCER_RESINFOD_CONFIG,
            balancer_resources.BALANCER_SMETANKA_CONFIG,
            balancer_resources.BALANCER_SUGGEST_CONFIG
        ]
        required = True
        group = _BALANCER_GROUP_NAME

    params = (ExecutableResourceParameter, ConfigResourceParameter)


class Balancer(StandardSearch):
    """
        Balancer
    """
    name = 'balancer'
    START_TIMEOUT = 5
    DEFAULT_HTTP_PORT = 10203

    def __init__(self, http_port, opts, **kwargs):
        self.http_port = http_port
        self.opts = opts or []
        kwargs['port'] = http_port
        super(Balancer, self).__init__(
            config_class=sconf.SearchConfig,
            config_params=[],
            **kwargs)

    def _get_run_cmd(self, config_path):
        cmd = [config_path]
        for (option_name, option_value) in self.opts:
            cmd.extend(['-V', '{}={}'.format(option_name, option_value)])
        return cmd


def get_balancer(params=DefaultBalancerParams, http_port=Balancer.DEFAULT_HTTP_PORT, opts=None):
    ctx = channel.task.ctx
    binary_path = channel.task.sync_resource(ctx[params.ExecutableResourceParameter.name])
    config_path = channel.task.sync_resource(ctx[params.ConfigResourceParameter.name])

    work_dir = channel.task.abs_path()
    return Balancer(
        http_port=http_port,
        opts=opts,
        work_dir=work_dir,
        binary=binary_path,
        config_file=config_path
    )


class MediaBalancer(StandardSearch):
    """
        Media balancer
    """
    name = 'media-balancer'
    START_TIMEOUT = 5
    DEFAULT_HTTP_PORT = 12121
    LOG_DIR = 'balancer-logs/'

    def __init__(self, http_port, opts, **kwargs):
        self.http_port = http_port
        self.opts = opts or {}
        kwargs['port'] = http_port
        super(MediaBalancer, self).__init__(
            config_class=sconf.SearchConfig,
            config_params=[],
            **kwargs)

    def _get_run_cmd(self, config_path):
        cmd = [config_path]
        for (option_name, option_value) in self.opts.iteritems():
            cmd.extend(['-V', '{}={}'.format(option_name, option_value)])
        return cmd

    def kill_softly(self):
        timeout = 160
        address_template = "http://{host}:{port}/admin?action=shutdown"
        address = address_template.format(host="localhost", port=self.http_port)
        try:
            response = urllib2.urlopen(address)
            logging.info("Stop command response: %s", response.read())
            self.process.communicate()
        except urllib2.URLError:
            return

        start = time.time()
        while time.time() - start <= timeout:
            if self.process.returncode is not None:
                logging.info("Instance was stopped with exitcode %s", self.process.returncode)
                return
            else:
                logging.info("Process still running %s", self.process.__dict__)
                time.sleep(5)
        else:
            raise Exception("Failed to stop instance: %s seconds timeout" % timeout)

    def save_logs_resource(self):
        channel.task.create_resource(
            "Logs for {}".format(channel.task.descr),
            self.opts['LogDir'],
            balancer_resources.BALANCER_MEDIA_LOGS
        )


def get_media_balancer(http_port, binary_path, config_path, opts=None):
    if not opts:
        opts = {}
    work_dir = channel.task.abs_path()
    if 'LogDir' not in opts:
        log_dir = '{}/{}/'.format(work_dir.rstrip('/'), MediaBalancer.LOG_DIR.rstrip('/'))
        paths.make_folder(log_dir, False)
        opts['LogDir'] = log_dir
    return MediaBalancer(
        http_port=http_port,
        opts=opts,
        work_dir=work_dir,
        binary=binary_path,
        config_file=config_path
    )


IMPROXY_PARAMETERS_GROUP_NAME = 'Improxy parameters'


class ImproxyExecutableResourceParameter(sp.LastReleasedResource):
    name = 'balancer_executable_resource_id'
    description = 'Executable'
    resource_type = balancer_resources.BALANCER_IMPROXY_EXECUTABLE
    group = IMPROXY_PARAMETERS_GROUP_NAME


class ImproxyConfigResourceParameterBase(sp.LastReleasedResource):
    name = 'balancer_config_resource_id'
    description = 'Balancer config'
    group = IMPROXY_PARAMETERS_GROUP_NAME


class ImproxyUseCgroup(sp.SandboxBoolParameter):
    name = 'improxy_use_cgroup'
    description = 'Use cgroup limits'
    group = IMPROXY_PARAMETERS_GROUP_NAME
    default_value = True


THUMBS_CACHER_PARAMETERS_GROUP_NAME = 'Cacher parameters'


class ThumbsCacherConfigParameter(sp.SandboxBoolParameter):
    name = 'thumbs_cacher_enable'
    description = 'Enable cacher'
    group = THUMBS_CACHER_PARAMETERS_GROUP_NAME
    default_value = True


def create_improxy_params(config_name_list):
    """
        Создает список параметров для тумбнейлерной прокси.
        :param config_name_list: список допустимых конфигов
    """
    class ImproxyConfigResourceParameter(ImproxyConfigResourceParameterBase):
        resource_type = media_settings.config_name_list2resource_type_list(config_name_list)

    return (
        ImproxyExecutableResourceParameter,
        ImproxyConfigResourceParameter,
        ImproxyUseCgroup,
    )


class ImproxyComponent(StandardSearch):
    """
        Представление тумбнейлерной прокси в Sandbox
    """
    name = 'improxy'
    CONFIG_PARAMS = []
    START_TIMEOUT = 5
    DEFAULT_HTTP_PORT = 11111
    DEFAULT_HTTPS_PORT = 11112
    DEFAULT_BAN_SERVER_PORT = 11115
    DEFAULT_CACHE_WEIGHTS_FILE = 'cacher.weight'
    SEARCH_CONFIG = sconf.SearchConfig

    def __init__(
        self, http_port, log_dir,
        https_port=None,
        ssl_certs_dir=None,
        ban_server_port=None,
        cache_weights_file=None,
        cgroup=None,
        **kwargs
    ):
        self.http_port = http_port
        self.https_port = https_port
        self.log_dir = '{}/'.format(log_dir)
        self.ssl_certs_dir = ssl_certs_dir
        self.ban_server_port = ban_server_port
        self.cache_weights_file = cache_weights_file

        # HTTPS-порт нужен только для указания в команде запуска
        # мониторинг живости/готовности происходит через http-port
        kwargs['port'] = http_port
        super(ImproxyComponent, self).__init__(
            config_class=self.SEARCH_CONFIG,
            config_params=self.CONFIG_PARAMS,
            cgroup=cgroup,
            **kwargs
        )

    def _get_run_cmd(self, config_path):
        cmd = [config_path]
        option_list = [
            ('ImproxyPort', self.http_port),
            ('InstancePort', self.http_port),
            ('SkipBindToServiceAddrs', "True"),
            ('SkipBindToWildcard', "False"),
            ('LogDir', self.log_dir),
            ('cacherWeightFile', self.cache_weights_file),
            ('BackendPortShift', 0),
            ('connect_timeout', '1s'),
            ('backend_timeout', '2.5s')
        ]
        if self.https_port is not None:
            assert self.ssl_certs_dir
            option_list.append(('ImproxySSLPort', self.https_port))
            option_list.append(('InstancePortSSL', self.https_port))
            option_list.append(('SSLCert', os.path.join(
                self.ssl_certs_dir,
                resource_types.TEST_SSL_CERTS.public_certificate
            )))
            option_list.append(('SSLCertKey', os.path.join(
                self.ssl_certs_dir,
                resource_types.TEST_SSL_CERTS.private_certificate
            )))
            # sha1 support
            option_list.append(('SSLCertSHA1', os.path.join(
                self.ssl_certs_dir,
                resource_types.TEST_SSL_CERTS.public_certificate
            )))
            option_list.append(('SSLCertKeySHA1', os.path.join(
                self.ssl_certs_dir,
                resource_types.TEST_SSL_CERTS.private_certificate
            )))
        if self.ban_server_port is not None:
            option_list.append(('QuerySearchPort', self.ban_server_port))

        for (option_name, option_value) in option_list:
            cmd.extend(['-V', '{}={}'.format(option_name, option_value)])
        return cmd

    def save_logs_resource(self):
        """
            Сохраняет логи improxy в ресурс
        """
        channel.task.create_resource(
            "Logs for {}".format(channel.task.descr),
            self.log_dir,
            resource_types.IMPROXY_LOGS
        )

    def access_log_path(self):
        """
            Путь к access.log-файлу или None, если такового не нашлось.
            Конкретное имя файла может меняться в зависимости от конфига.
        """
        for path in os.listdir(self.log_dir):
            if path.endswith("access.log") or path.startswith("current-access_log-balancer"):
                return os.path.join(self.log_dir, path)
        return None

    def _get_required_hosts_and_ports(self):
        hostname = os.uname()[1]
        result = [('localhost', self.http_port), (hostname, self.http_port)]
        if self.https_port is not None:
            result.append((hostname, self.https_port))
        return result


class MultiComponent(object):
    """
        Сложный компонент, состоящий из нескольких инстансов, один из которых является главным,
        а другие нужны для функционирования первого, например,
        балансер с отдельно стоящим кешером и кликовым сервером для бана.
        Реализует интерфейс SearchComponent, не наследуясь от него,
        чтобы не тянуть дефолтные поля/методы
    """

    def __init__(self, main, secondary):
        self.main = main
        self.instances = [main] + secondary

    def start(self):
        for instance in self.instances:
            instance.start()

    def stop(self):
        for instance in self.instances:
            instance.stop()

    def wait(self):
        for instance in self.instances:
            instance.wait()

    @property
    def name(self):
        return "-".join((x.name for x in self.instances))

    def __getattr__(self, item):
        """
            Все неизвестные атрибуты ищутся в основном компоненте
        """
        return getattr(self.main, item)


def get_improxy(
        http_port=ImproxyComponent.DEFAULT_HTTP_PORT,
        https_port=ImproxyComponent.DEFAULT_HTTPS_PORT,
        cgroup=None):
    work_dir = channel.task.abs_path()

    ctx = channel.task.ctx
    ban_server_port = ImproxyComponent.DEFAULT_BAN_SERVER_PORT
    cache_weights_file = '{}/{}'.format(work_dir, ImproxyComponent.DEFAULT_CACHE_WEIGHTS_FILE)

    certs_resource_id = apihelpers.get_last_resource(resource_types.TEST_SSL_CERTS)
    certs_path = channel.task.sync_resource(certs_resource_id)

    binary_path = channel.task.sync_resource(ctx[ImproxyExecutableResourceParameter.name])
    balancer_config_path = channel.task.sync_resource(ctx[ImproxyConfigResourceParameterBase.name])
    logs_dir = paths.make_folder(os.path.join(work_dir, 'improxy_logs'))

    # disable cache module for tests when it needed
    if not ctx[ThumbsCacherConfigParameter.name]:
        with open(cache_weights_file, 'w') as out:
            out.write('{}\n{}\n{}\n{}'.format(
                'disable_cacher,1',
                'vla_cacher,0',
                'sas_cacher,0',
                'man_cacher,0',
            ))

    improxy = ImproxyComponent(
        work_dir=work_dir,
        binary=binary_path,
        config_file=balancer_config_path,
        http_port=http_port,
        https_port=https_port,
        log_dir=logs_dir,
        ssl_certs_dir=certs_path,
        ban_server_port=ban_server_port,
        cache_weights_file=cache_weights_file,
        cgroup=cgroup,
    )

    return improxy


class PeopleSsSearch(SearchExecutableComponent):
    def __init__(
        self, work_dir, binary, config_file, port,
        config_params="",
        config_class=sconf.SimpleConfigBase
    ):

        super(PeopleSsSearch, self).__init__(
            work_dir=work_dir,
            config_file=config_file,
            config_class=config_class,
            config_params=config_params,
            port=port,
            binary=binary
        )

    def _get_run_cmd(self, config_path):
        return [config_path]


def get_peopless_ex(binary_id, config_id, port):
    """
        Синхронизировать ресурсы и получить обёртку peopless поиска
        используется в старых тасках
        :return: объект PeopleSsSearch
        :rtype: PeopleSsSearch
    """
    binary_path = channel.task.sync_resource(binary_id)
    config_path = channel.task.sync_resource(config_id)

    return PeopleSsSearch(
        work_dir=channel.task.abs_path(),
        binary=binary_path,
        config_file=config_path,
        port=port
    )


class DefaultPeopleSsParams:
    class Binary(sp.ResourceSelector):
        name = 'peopless_executable_resource_id'
        description = 'Executable'
        resource_type = [resource_types.PEOPLESS_EXECUTABLE]
        group = 'Peopless parameters'
        required = True

    class Config(sp.ResourceSelector):
        name = 'peopless_config_resource_id'
        description = 'Config'
        resource_type = resource_types.PEOPLESS_CONFIG
        group = 'Peopless parameters'
        required = True

    params = (Binary, Config)


DEFAULT_PEOPLESS_PORT = 12125


def get_peopless(port=DEFAULT_PEOPLESS_PORT, params=DefaultPeopleSsParams):
    """
        Получить обёртку peopless поиска - объект класса PeopleSsSearch
        Идентификаторы параметров-ресурсов берутся из контекста таска
        Ресурсы синхронизируются

        :return: объект PeopleSsSearch
        :rtype: PeopleSsSearch
    """

    ctx = channel.task.ctx
    return get_peopless_ex(
        binary_id=ctx[params.Binary.name],
        config_id=ctx[params.Config.name],
        port=port
    )


class HttpCache(SearchExecutableComponent):
    """
        Фейковый http сервер, который отвечает сохраненныеми ответами на запросы от peopless.
        Принимает на вход файл с запросами от peopless и готовыми ответами на них.
    """

    def __init__(self, work_dir, binary, config_file, reqans_file, port):

        config_params = {}
        config_params.update({
            'Server/Backend/MetasearchReqansFile': reqans_file
        })

        super(HttpCache, self).__init__(
            work_dir=work_dir,
            config_file=config_file,
            config_class=sconf.SearchConfig,
            config_params=config_params,
            port=port,
            binary=binary
        )

    def _get_run_cmd(self, config_path):
        return [config_path]


def get_httpcache_ex(binary_id, config_id, reqans_id, port):
    """
        Синхронизировать ресурсы и получить обёртку для фейкового http сервера
        :return: объект HttpCache
        :rtype: HttpCache
    """
    binary_path = channel.task.sync_resource(binary_id)
    config_path = channel.task.sync_resource(config_id)
    reqans_path = channel.task.sync_resource(reqans_id)

    return HttpCache(
        work_dir=channel.task.abs_path(),
        binary=binary_path,
        config_file=config_path,
        reqans_file=reqans_path,
        port=port
    )


class DefaultHttpCacheParams:
    class Binary(sp.ResourceSelector):
        name = 'httpcache_executable_resource_id'
        description = 'Executable'
        resource_type = [resource_types.HTTPCACHE_EXECUTABLE]
        group = 'Http cache parameters'
        required = True

    class Config(sp.ResourceSelector):
        name = 'httpcache_config_resource_id'
        description = 'Config'
        resource_type = resource_types.HTTPCACHE_CONFIG
        group = 'Http cache parameters'
        required = True

    class ReqansFile(sp.ResourceSelector):
        name = 'httpcache_reqansfile_resource_id'
        description = 'Reqansfile'
        resource_type = resource_types.HTTPCACHE_REQANSFILE
        group = 'Http cache parameters'
        required = True

    params = (Binary, Config, ReqansFile)


DEFAULT_HTTP_CACHE_PORT = 22126


def get_httpcache(port=DEFAULT_HTTP_CACHE_PORT, params=DefaultHttpCacheParams):
    """
        Получить объект класса HttpCache - фейковый веб сервер для моделирования ответов
        от rty поиска по людям.

        :return: объект HttpCache
        :rtype: HttpCache
    """

    ctx = channel.task.ctx
    return get_httpcache_ex(
        binary_id=ctx[params.Binary.name],
        config_id=ctx[params.Config.name],
        reqans_id=ctx[params.ReqansFile.name],
        port=port
    )


DEFAULT_WIZARD_PORT = 31333


class Wizard(StandardSearch):
    name = 'wizard'

    def __init__(
        self,
        work_dir,
        binary,
        port,
        config_class,
        config_file,
        data_dir,
        runtime_dir,
        cache,
        eventlog_path=None,
        config_params='',
        use_profiler=False,
        apphost_port=None,  # port + 1 by default
        task=None,
        **kwargs
    ):
        super(Wizard, self).__init__(
            work_dir,
            binary,
            port,
            config_class,
            config_file,
            config_params,
            use_profiler=use_profiler,
            use_gperftools=use_profiler,  # if using profiler, use gperf tools' one
            task=task,
            **kwargs
        )
        self.data_dir = data_dir
        self.runtime_dir = runtime_dir
        self.prepared_data_dir = self.prepare_data(self.data_dir, self.runtime_dir)
        self.cache = cache
        self.eventlog_path = eventlog_path
        self.config_file = config_file
        self.apphost_port = apphost_port or (int(self.port or "8891") + 1)

    def _get_run_cmd(self, config_path):
        cmd = [
            '-s', config_path,
            '-a', self.prepared_data_dir,
            '-p', str(self.port),
            '--apphost-port', str(self.apphost_port),
            '--mmap',
        ]
        if self.eventlog_path:
            cmd.extend(['-e', self.eventlog_path])
        if self.runtime_dir is not None and os.path.isdir(self.runtime_dir):
            cmd.extend(['-R', self.runtime_dir])
        else:
            logging.debug("_get_run_cmd: self.runtime_dir: {}, isdir? {}".format(
                self.runtime_dir, os.path.isdir(self.runtime_dir) if self.runtime_dir is not None else '<None>'
            ))
        if not self.cache:
            cmd.append('--nocache')
        return cmd

    @property
    def _stop_command(self):
        return "/wizard?action=shutdown"

    def prepare_data(self, data_dir, runtime_dir):
        if runtime_dir is not None and not os.path.isdir(runtime_dir):
            mod = (
                stat.S_IMODE(os.lstat(os.path.dirname(data_dir)).st_mode) |
                stat.S_IWOTH |
                stat.S_IWGRP |
                stat.S_IWUSR
            )
            os.chmod(os.path.dirname(self.data_dir), mod)
            cmd = 'tar -C {} -xf {}'.format(self.work_dir, runtime_dir)
            process.run_process(
                cmd,
                work_dir=self.work_dir,
                shell=True,
                log_prefix='extract_files'
            )
            if runtime_dir.endswith('.tar'):
                self.runtime_dir = os.path.basename(runtime_dir)[0:-4]
        os.mkdir(os.path.join(self.work_dir, 'wizard'))
        os.symlink(data_dir, os.path.join(self.work_dir, 'wizard/WIZARD_SHARD'))
        return self.work_dir

    def get_stat(self):
        as_json = self.fetch('/wizard?action=rulestat')
        channel.task.ctx['rule_stats'] = rules = {}
        rules = json.loads(as_json)
        if not isinstance(rules, dict):
            rules = {}

        # let's make a text table in ctx human-readable
        rule_len = 0
        for name, details in rules.iteritems():
            duration = details.get('Duration', 0.0)
            requests = details.get('Requests', 0)
            successes = details.get('Success', 0)
            time_per_req = duration / requests if requests else 'N/A'
            rule_len = max(rule_len, len(name))
            rules[name] = {
                'duration': duration,
                'count': requests,
                'ok': successes,
                'us_per_req': time_per_req,
            }

        lines = [(name, d['duration'], d['ok'], d['count'], d['us_per_req']) for name, d in rules.iteritems()]
        line_fmt = u'%%%ds\t%%13s us\t%%8s\t%%8s\t%%13s us' % rule_len
        as_text = [line_fmt % ('Rule name', 'Total', 'Ok\'s', 'Requests', 'Time/request')] + \
            [line_fmt % data for data in sorted(lines, key=lambda x: x[4])]  # sort by time per request
        return '\n'.join(as_text)

    def get_yasm_stat(self):
        return dict(json.loads(self.fetch('/wizard?action=stat')))

    def get_exceptions(self):
        exceptions_url = '/wizard?action=exceptions'
        return self.fetch(exceptions_url)


def get_wizard(
        binary_path,
        config_path,
        data_path,
        runtime_path=None,
        cache=True,
        eventlog_path=None,
        use_profiler=False,
        task=None
):
    """
    Синхронизировать ресурсы и получить обёртку для wizard
    :param binary_path: путь до бинарника визарда в sandbox
    :param config_path: путь до конфига визарда в sandbox
    :param data_path: путь до данных визарда в sandbox
    :param runtime_path: путь до данных быстрых визарда в sandbox
    :param cache: включает кеширование в визарде
    :return: объект Wizard
    """
    # TODO: https://rb.yandex-team.ru/arc/r/134649/
    return Wizard(
        work_dir=channel.task.abs_path(),
        binary=binary_path,
        port=DEFAULT_WIZARD_PORT,
        config_class=sconf.SearchConfig,
        config_file=config_path,
        data_dir=data_path,
        runtime_dir=runtime_path,
        cache=cache,
        eventlog_path=eventlog_path,
        use_profiler=use_profiler,
        task=task
    )


DEFAULT_BEGEMOT_PORT = 31400


class Begemot(StandardSearch):
    name = 'begemot'

    def __init__(
        self,
        work_dir,
        binary,
        port,
        config_class,
        config,
        config_params='',
        task=None,
        worker_dir=None,
        fresh_dir=None,
        eventlog_path=None,
        cache_size=0,
        jobs_count=None,
        max_stderr_size_kb=0,
        additional_cmd_args=None,
        **kwargs
    ):
        super(Begemot, self).__init__(
            work_dir,
            binary,
            port,
            config_class,
            config,
            config_params,
            task=task,
            **kwargs
        )
        self.port = port
        self.worker_dir = worker_dir
        self.eventlog_path = eventlog_path
        self._config = config
        self.fresh_dir = fresh_dir
        self.cache_size = cache_size
        self.jobs_count = jobs_count
        self.additional_cmd_args = additional_cmd_args
        self.max_stderr_size_kb = max_stderr_size_kb

    def _get_run_cmd(self, config_path):
        assert self.worker_dir, 'starting begemot without data is no longer supported'
        cmd = ['--cfg', config_path, '--data', self.worker_dir, '--port', str(self.port)]
        if self.eventlog_path:
            cmd.extend(['--log', self.eventlog_path])
        if self.fresh_dir:
            cmd.extend(['--fresh', self.fresh_dir])
        if self.cache_size:
            cmd.extend(['--cache_size', str(self.cache_size)])
        if self.jobs_count:
            cmd.extend(('--jobs', str(self.jobs_count)))
        if self.additional_cmd_args:
            cmd.extend(self.additional_cmd_args)
        return cmd

    def _generate_config(self):
        return self._config

    def verify_stderr(self, file_name):
        if os.path.getsize(file_name) > self.max_stderr_size_kb * 1024:
            raise sandbox_errors.TaskFailure('Begemot stderr size is larger than {} KB, check for unwanted debug info in stderr'.format(self.max_stderr_size_kb))

def get_begemot(
        binary_path,
        config_path,
        worker_dir=None,
        fresh_dir=None,
        port=DEFAULT_BEGEMOT_PORT,
        task=None,
        eventlog_path=None,
        cache_size=0,
        jobs_count=None,
        additional_cmd_args=None,
        start_timeout=DEFAULT_START_TIMEOUT,
        max_stderr_size_kb=0,
):
    """
    Синхронизировать ресурсы и получить обёртку для begemot
    :param binary_path: путь до бинарника бегемота
    :param config_path: путь до конфига бегемота
    :param worker_dir: путь до данных бегемота
    :param fresh_dir: путь до данных фреша бегемота
    :param port: порт, на котором будет работать демон бегемота
    :param jobs_count: количество потоков
    :return: объект Begemot
    """
    return Begemot(
        work_dir=channel.task.abs_path(),
        binary=binary_path,
        port=port,
        config_class=sconf.BegemotConfig,
        config=config_path,
        worker_dir=worker_dir,
        fresh_dir=fresh_dir,
        task=task,
        eventlog_path=eventlog_path,
        cache_size=cache_size,
        jobs_count=jobs_count,
        additional_cmd_args=additional_cmd_args,
        start_timeout=start_timeout,
        max_stderr_size_kb=max_stderr_size_kb,
        use_verify_stderr=max_stderr_size_kb > 0,
    )
