import errno
import httplib
import logging
import os
import random
import time
import urllib2

from sandbox import sdk2
from sandbox import sandboxsdk
from sandbox.common import errors
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils


_logger = logging.getLogger(__name__)


class Component(object):

    """Skeleton class for all components"""

    name = "component"  # Fake field used by sanitizer.py library

    def start(self):
        """Start component"""
        raise NotImplementedError()

    def wait(self, timeout=600):
        """Wait until component ready to use"""
        raise NotImplementedError()

    def warmup(self):
        """Some components need to be warmed up with special query"""
        pass

    def stop(self, timeout=5):
        """Stop component"""
        raise NotImplementedError()

    def __enter__(self):
        _logger.debug("WITH START %s", self.name)
        self.start()
        self.wait()
        self.warmup()
        return self

    def __exit__(self, exc_type, exc_value, traceback):
        _logger.debug("WITH STOP %s", self.name)
        self.stop()


class ProcessComponentMixin(object):
    """Generic mixin for components with executable."""

    def __init__(self, args, log_prefix=None, shutdown_timeout=30, **exec_args):
        self._args = args
        self._log_prefix = log_prefix or os.path.basename(args[0])
        self._environment = {}  # Used by profiling and sanitizer libraries
        self._exec_args = exec_args
        self.process = None
        self.run_cmd_patcher = None  # Old hack used by profiler library, TODO: replace it with get_args/set_args
        self._shutdown_timeout = shutdown_timeout

    def _run_process(self):
        _logger.info("Start running process %s", self._log_prefix)
        self.process = sandboxsdk.process.run_process(
            self._args,
            outputs_to_one_file=False,
            log_prefix=self._log_prefix,
            environment=self._environment,
            wait=False,
            **self._exec_args
        )
        _logger.info("Process id: %s", self.process.pid)

    def start(self):
        if self.run_cmd_patcher:
            self._args = self.run_cmd_patcher(self._args)
        self._run_process()
        _logger.info("Process %s was successfully started", self._log_prefix)

    def stop(self):
        if self.process is None:
            _logger.info("Process in None, nothing to stop")
            return

        self._ensure_process()

        _logger.info("Waiting for process %s termination in %s seconds", self._log_prefix, self._shutdown_timeout)
        wait_for = time.time() + self._shutdown_timeout
        while time.time() < wait_for:
            if self.process.poll() is not None:
                _logger.info("Process %s was successfully terminated", self._log_prefix)
                return

            try:
                self.process.terminate()
            except EnvironmentError as e:
                if e.errno != errno.ESRCH:
                    _logger.warning("Failed to terminate process %s: %s", self._log_prefix, e)
                _logger.info("Process %s was successfully terminated", self._log_prefix)
                return
            time.sleep(0.1)

        try:
            _logger.info("Process %s is still alive. Killing it.", self._log_prefix)
            self.process.kill()
        except EnvironmentError as e:
            if e.errno != errno.ESRCH:
                _logger.warning("Failed to kill process %s: %s", self._log_prefix, e)

    # TODO: do something with environment methods:
    def get_environment(self):
        return self._environment

    def set_environment(self, env):
        self._environment = env

    def _ensure_process(self):
        rc = self.process.poll()
        if rc is not None:
            utils.show_process_tail(self.process)
            eh.check_failed(
                "Process {} was unexpectedly exited with code {}".format(self._log_prefix, rc)
            )

    def verify_stderr(self, custom_stderr_path=None):
        """Check stderr output stored in file for errors/warnings existence."""
        pass


class WaitUrlComponentMixin:
    """Generic mixin for components that need to wait for some url availability"""

    def __init__(self, url, wait_timeout=60, ensure_process=None):
        """Generic mixin for components that need to wait for some url availability
        Arguments
            url -- checked url
            wait_timeout -- checking time limit (seconds)
            ensure_process -- function for checking process status
                (SHOULD raise exception if process already finished
                see ProcessComponentMixin._ensure_process() as example)
        """
        self.__url = url
        self.__wait_timeout = wait_timeout
        self.__ensure_process = ensure_process

    def wait(self):
        _logger.info("Waiting for component via probing url '%s'", self.__url)
        wait_for = time.time() + self.__wait_timeout
        while time.time() < wait_for:
            try:
                urllib2.urlopen(self.__url)
                _logger.info("Component was successfully answered at '%s' and ready to use", self.__url)
                return True
            except (urllib2.URLError, httplib.BadStatusLine):
                if self.__ensure_process:
                    self.__ensure_process()
                time.sleep(2)
        eh.check_failed(
            "Failed to connect to url {} in {} seconds".format(self.__url, self.__wait_timeout)
        )


class WaitPortComponentMixin:
    """Generic mixin for components that need to wait until port opened"""

    def __init__(self, endpoints, wait_timeout=60, ensure_process=None):
        """Generic mixin for components that need to wait until port opened
        Arguments
            endpoint -- (host, port) tuple or list
            wait_timeout -- checking time limit (seconds)
             ensure_process -- function for checking process status
                (SHOULD raise exception if process already finished
                see ProcessComponentMixin._ensure_process() as example)
        """
        self.__endpoints = endpoints
        self.__wait_timeout = wait_timeout
        self.__ensure_process = ensure_process

    def wait(self):
        _logger.info("Waiting for component on endpoints: %s", self.__endpoints)
        wait_for = time.time() + self.__wait_timeout
        while time.time() < wait_for:
            if all(not sandboxsdk.network.is_port_free(port, host) for host, port in self.__endpoints):
                return
            if self.__ensure_process:
                self.__ensure_process()
            time.sleep(2)
        eh.check_failed(
            "Failed to connect to endpoints {} in {} seconds".format(self.__endpoints, self.__wait_timeout)
        )


class ProcessComponentMixinWithShutdown(ProcessComponentMixin):
    """Generic mixin for components with ability to shutdown via API request"""

    def __init__(self, args, shutdown_url, **kwargs):
        self.__shutdown_url = shutdown_url
        ProcessComponentMixin.__init__(self, args, **kwargs)

    def stop(self):
        try:
            self._ensure_process()
            _logger.info("Sending shutdown request to '%s'", self.__shutdown_url)
            urllib2.urlopen(self.__shutdown_url)

            _logger.info("Waiting for component shutdown in %s seconds", self._shutdown_timeout)
            waitfor = time.time() + self._shutdown_timeout
            while time.time() < waitfor:
                rc = self.process.poll()
                if rc is None:
                    time.sleep(2)
                elif rc != 0:
                    eh.check_failed("Process exited with code {}".format(rc))
                else:
                    _logger.info("Component exited successfully")
                    break
        except errors.TaskFailure:
            raise
        except Exception as e:
            eh.log_exception("Failed to shutdown process", e)
            eh.check_failed("Failed to shutdown process: {}".format(e))
        finally:
            if self.process.poll() is None:
                ProcessComponentMixin.stop(self)


class ProcessComponentMixinWithShutdownSDK2(ProcessComponentMixinWithShutdown):
    """
        Replaces old sdk methods by new ones.
        Place it on the very left of inheritance sequence!
    """

    def __init__(self, args, shutdown_url, **kwargs):
        ProcessComponentMixinWithShutdown.__init__(self, args, shutdown_url, **kwargs)
        self.process_log = sdk2.helpers.ProcessLog(self.task, logger=self._log_prefix)
        self.process_registry = sdk2.helpers.ProcessRegistry

    def start(self):
        self.process_log.__enter__()
        self.process_registry.__enter__()
        super(ProcessComponentMixinWithShutdownSDK2, self).start()

    def stop(self):
        super(ProcessComponentMixinWithShutdownSDK2, self).stop()
        self.process_registry.__exit__()
        self.process_log.__exit__(None, None, None)

    def _run_process(self):
        self.process = sdk2.helpers.subprocess.Popen(
            self._args,
            stdout=self.process_log.stdout, stderr=self.process_log.stderr,
            env=self._environment or None,
            **self._exec_args
        )
        self.process.stderr_path = str(self.process_log.stderr.path)
        self.process.stdout_path = str(self.process_log.stdout.path)


class SearchDaemonComponent(
    ProcessComponentMixin,
    WaitUrlComponentMixin,
    Component,
):

    """Wrapper over daemon"""

    def __init__(self, args, url, log_prefix="daemon", start_timeout=600, shutdown_timeout=120):
        """
            :param args: command list. For example, ['./suggest-web-daemon', '-p', '24535', 'suggest-daemon.conf']
            :param log_prefix: daemon log prefix for logging]
            :param start_timeout: daemon start timeout (in seconds)
            :param shutdown_timeout: daemon shutdown timeout (in seconds)
            :param url: wait for url to start. For example, 'http://localhost:24545/ping'
        """

        ProcessComponentMixin.__init__(
            self,
            args=args,
            log_prefix=log_prefix,
            shutdown_timeout=shutdown_timeout,
        )

        WaitUrlComponentMixin.__init__(
            self,
            url=url,
            wait_timeout=start_timeout,
        )


# Solution from get-port
# See https://github.com/sindresorhus/get-port/blob/a9b445ea0afbd75b81b7b9011c896e111e2d4aee/index.js
class _LockedPorts(object):
    #  This is minimal duration, actual could be double that
    lock_port_for_s = 30

    def __init__(self):
        self.old = set()
        self.young = set()
        self.ts = time.time()

    def lock(self, port):
        return self.young.add(port)

    def is_locked(self, port):
        return port in self.old or port in self.young

    def tick(self):
        now = time.time()
        if now >= self.ts + self.lock_port_for_s:
            self.old = self.young
            if now >= self.ts + 2 * self.lock_port_for_s:
                # It's been too long, we can assume all ports unlocked
                self.old = set()
            self.young = set()
            self.ts = now


_locked_ports = _LockedPorts()


def try_get_free_port(tries=25):
    """
        Tries to find free port.

        :param tries: Number of tries.
        :return: port or raise FailureError.
    """
    for _ in range(tries):
        _locked_ports.tick()

        # https://wiki.yandex-team.ru/sandbox/cookbook/#kakpoluchitdiapazongarantirovannosvobodnyxportov
        port = random.randint(15000, 25000)
        if _locked_ports.is_locked(port):
            _logger.debug('Port %d is locked by previous try_get_free_port calls', port)
            continue

        _logger.debug('Checking if %d port is free', port)
        if sandboxsdk.network.is_port_free(port):
            _logger.debug('Port %d is free', port)
            _locked_ports.lock(port)
            return port
        else:
            _logger.debug('Port %d is busy', port)

    eh.check_failed('Failed to find free port after {} retries'.format(tries))
