# -*- coding: utf-8 -*-
import six
import logging
import os
import requests
import time

# Sandbox stuff
import sandbox.common.types.client as ctc
from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk import task
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.parameters import ResourceSelector
from sandbox.sandboxsdk.paths import add_write_permissions_for_path
from sandbox.sandboxsdk.paths import copy_path
from sandbox.sandboxsdk.process import run_process

from sandbox.projects import resource_types
from sandbox.projects.common import utils
from sandbox.projects.common.dolbilka import DolbilkaFixedRps
from sandbox.projects.common.search import config as sconf
from sandbox.projects.common.search.components import StandardSearch, parse_version_info
from sandbox.projects.common.market_report import generator as market_report_generator


DEFAULT_MARKET_REPORT_PORT = 17051


class MarketReport(StandardSearch):
    name = 'market_report'

    def __init__(
            self,
            binary,
            port,
            config_class,
            config_generator,
            index_dir,
            config_params='',
            use_profiler=False,
            **kwargs
    ):
        work_dir = channel.task.abs_path()
        config_path = config_generator.generate(index_dir, port)
        self.config_path = config_path
        super(MarketReport, self).__init__(
            work_dir,
            binary,
            port,
            config_class,
            config_path,
            config_params,
            use_profiler=use_profiler,
            use_gperftools=use_profiler,
            **kwargs
        )
        self.index_dir = index_dir
        self.id = time.time()

    def _stop_impl(self):
        self.process.terminate()
        time.sleep(3)

    def _get_run_cmd(self, config_path):
        cmd = [
            '-d', config_path,
        ]
        return cmd

    def _get_info(self):
        """
            Получить информацию об исполняемом файле. См. метод parse_version_info
        """
        proc = run_process(
            [self.binary, '-v', '-d', self.config_path],
            check=False,
            outs_to_pipe=True,
        )
        text_info = proc.stdout.read()

        return parse_version_info(text_info)

    def wait_report_start(self):
        for i in six.moves.xrange(0, 200):
            time.sleep(1)
            try:
                response = requests.get(
                    "http://localhost:{0}/yandsearch?place=prime"
                    "&text=nokia&sandbox-report-start-check=1".format(self.port)
                )
                if response.status_code == 200:
                    logging.debug("Response text: %s", response.text)
                    return
            except requests.ConnectionError:
                pass
        raise SandboxTaskFailureError('Failed to start report')

    def start(self):
        result = super(MarketReport, self).start()
        self.wait_report_start()
        return result


class Binary(ResourceSelector):
    name = 'report_binary'
    description = 'Market Report binary'
    resource_type = resource_types.MARKET_REPORT_BINARY
    required = True


class ConfigGenerator(ResourceSelector):
    name = 'report_config_generator'
    description = 'Market Report config generator'
    resource_type = resource_types.MARKET_REPORT_CONFIG_GENERATOR
    required = True


class ConfigTemplate(ResourceSelector):
    name = 'report_config_template'
    description = 'Market Report config template'
    resource_type = resource_types.MARKET_REPORT_CONFIG_TEMPLATE
    required = True


class Index(ResourceSelector):
    name = 'report_index'
    description = 'Market Report index'
    resource_type = resource_types.MARKET_REPORT_INDEX
    required = True


class Plan(ResourceSelector):
    name = "report_plan"
    description = "Market Report plan"
    resource_type = [resource_types.MARKET_REPORT_PLAN]
    required = True


class RpsParameter(DolbilkaFixedRps):
    default_value = 160


class IndexGeneration(sp.SandboxStringParameter):
    name = 'index_generation'
    description = 'Market Index generation'
    required = True


class SearchIndexLink(sp.SandboxStringParameter):
    name = 'search_index_link'
    description = 'Market Search index rsync path (comma-separated)'
    required = True


class RamdriveEnabled(sp.SandboxBoolParameter):
    name = 'ramdrive_enabled'
    description = 'Store index in RAM'
    required = False
    default_value = True


class RamdriveSizeParameter(sp.SandboxIntegerParameter):
    name = 'ramdrive_size'
    description = 'Ramdrive size (GB)'
    default_value = 90


class SaveAnswers(sp.SandboxBoolParameter):
    name = 'save_answers'
    description = 'Save report answers for further analysis'
    required = False
    default_value = False


class ShortRunMode(sp.SandboxBoolParameter):
    name = 'short_run_mode'
    description = 'Do only 100 requests per shooting (for testing purposes)'
    required = False
    default_value = False


class MarkWithTag(sp.SandboxStringParameter):
    name = 'mark_with_tag'
    description = 'Add tag to resource (tag is used to pick eligible for shooting resource in TestEnv)'
    required = False


class EnvironmentType(sp.SandboxStringParameter):
    name = 'environment_type'
    description = (
        'Report environment type (if you have a local SB instance, '
        'set this to "prestable" to use non-production external services)'
    )
    required = False
    default_value = 'production'


REPORT_PERF_PARAMS = [
    Binary,
    Index,
    ConfigGenerator,
    Plan,
    RamdriveEnabled,
    RamdriveSizeParameter,
    RpsParameter,
    ShortRunMode,
    EnvironmentType,
]


REPORT_MONEY_PARAMS = [
    Binary,
    Index,
    ConfigGenerator,
    Plan,
    RamdriveEnabled,
    RamdriveSizeParameter,
    ShortRunMode,
    EnvironmentType,
]


REPORT_INDEX_PARAMS = [
    IndexGeneration,
    SearchIndexLink,
    MarkWithTag,
]


def get_market_report(
        binary_id,
        index_path,
        config_generator,
        use_profiler=False,
):
    """
        Синхронизировать ресурсы и получить обёртку market report

        params:
                бинарный файл
                файл конфига
                файл с индексом
        return: возвращается объект  MarketReport
    """
    binary_path = channel.task.sync_resource(binary_id)

    return MarketReport(
        binary=binary_path,
        port=DEFAULT_MARKET_REPORT_PORT,
        config_class=sconf.SearchConfig,
        config_generator=config_generator,
        index_dir=index_path,
        use_profiler=use_profiler
    )


class MarketReportFiringTask(task.SandboxTask):

    client_tags = ctc.Tag.LINUX_TRUSTY

    def _create_report_logs_dir(self):
        return utils.create_misc_resource_and_dir(
            ctx=self.ctx,
            ctx_field_name="market_report_logs_resource_id",
            descr="Market report logs (analog of /var/log/search)",
            dir_name="market_report_logs")

    def _get_prepare_report(self):
        """
        Download and prepare index.
        Prepare report config
        :return: report object
        """
        config_generator_path = channel.task.sync_resource(self.ctx[ConfigGenerator.name])
        self.report_logs_folder = self._create_report_logs_dir()
        index_path = self._get_index_path()
        generator = market_report_generator.ConfigGenerator(
            config_generator_path=config_generator_path,
            env_type=self.ctx[EnvironmentType.name],
            report_logs_folder=self.report_logs_folder)
        report = get_market_report(self.ctx[Binary.name], index_path, generator)
        return report

    def _get_index_path(self):
        """
        Скачивает индекс из ресурса.
        Опционально располагает его в RAM-drive
        """
        index_path = channel.task.sync_resource(self.ctx[Index.name])
        if self.ctx[RamdriveEnabled.name]:
            self_dir = os.path.abspath("")
            tmp_index_data = os.path.join(self_dir, 'tmp_index_data')
            copy_path(index_path, tmp_index_data)
            add_write_permissions_for_path(tmp_index_data)
            logging.info("Enabling ramdrive")
            inner_index_path = os.path.join(tmp_index_data, 'index')
            ramd_index_path = os.path.join(self.ramdrive.path, 'index')
            run_process(['mv %s  %s' % (inner_index_path, self.ramdrive.path)],
                        shell=True, check=True)
            os.symlink(ramd_index_path, inner_index_path)
            return tmp_index_data
        else:
            return index_path

    def _get_requests_limit(self, desired_request_limits):
        if self.ctx[ShortRunMode.name]:
            return 100
        else:
            return desired_request_limits
