# -*- coding: utf-8 -*-

import os
import re
import logging
import xml.dom.minidom
import httplib
import urllib2
import random

from sandbox.sandboxsdk import network

from sandbox.projects.common import cms
from sandbox.projects.common import decorators
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import config_processor as cfgproc
from sandbox.projects.common import utils
from sandbox.projects.common.base_search_quality import settings as bss


class SearchConfigsException(Exception):
    """
        Исключение при работе с конфигами компонентов поиска
    """
    pass


class SearchConfigsNotImplemented(Exception):
    """
        Исключение при вызове метода, который не определён в реализации класса
    """
    pass


class SearchConfig(object):
    """
        Общий класс для конфигов поиска
    """

    def __init__(self, config_text=None):
        self.text = config_text

    @classmethod
    def _get_production_config_text_by_cms_tag(cls, cms_tags):
        """
            Получить текст конфиг из продакшена по тегу CMS: получается рандомный инстанс
            по данному тэгу, и из него инфо-запросом достается конфиг.

            :param cms_tags: тег CMS в виде строки или списка (если их несколько)
            :return: текст конфига в виде строки
        """

        logging.info('Get production config by CMS tag %s', cms_tags)
        instances = cms.get_cms_instances(cms_tags)

        @decorators.retries(3, 1)
        def get_cfg(instances):
            # due to weird network topology, some instances can be unreacheable
            # see SEARCH-1673
            rand_inst = random.choice(instances)
            return utils.get_config_by_info_request(rand_inst['host'], rand_inst['port'])

        return get_cfg(instances)

    @classmethod
    def _get_production_config_text(cls):
        """
            Получение текста конфига из продакшена
        """
        error_message = 'get_production_config_text method is not implemented for base SearchConfig class'
        raise SearchConfigsNotImplemented(error_message)

    @classmethod
    def _check_file_path(cls, file_path):
        """
            Проверяет путь до конфига

            :param file_path: путь до файла
            :return: проверенный путь, если все проверки прошли успешно;
                в противном случае вызвает исключение SearchConfigsException
        """
        file_path = os.path.abspath(file_path)
        if not os.path.exists(file_path):
            raise SearchConfigsException('Path %s does not exist' % file_path)
        if not os.path.isfile(file_path):
            raise SearchConfigsException('Path %s is not file' % file_path)
        return file_path

    @classmethod
    def get_production_config(cls):
        """
            Получить объект конфига базового из продакшена

            :return: объект конфига SearchConfig
        """
        config_text = cls._get_production_config_text()
        return cls._get_config_from_text(config_text)

    @classmethod
    def get_config_from_file(cls, file_path):
        """
            Получить объект конфига из переданного файла

            :return: объект конфига SearchConfig
        """
        file_path = cls._check_file_path(file_path)
        return cls._get_config_from_text(open(file_path).read())

    @classmethod
    def _get_config_from_text(cls, config_text):
        """
            Получить объект конфига из текста

            :return: объект конфига SearchConfig
        """
        result_config = cls(config_text)
        return result_config

    def get_parameter(self, parameter):
        """
            Получает параметр из конфига по полному пути
            То есть для "Search/Port 8041"
            Будет возвращено 8041
            Возвращает первое включение парметра, если он встречается более, чем один раз

            :parameter: название параметра
            :return: значение параметра в виде строки, если он найден; None, если параметр не найден
        """
        values = cfgproc.get_parameter_values(self.text, parameter)
        if len(values) == 0:
            return None
        return values[0]

    def get_db_rel_path_from(self, parameter):
        """
            Получаем параметр в виде пути относительно папки '/db/'
            (для последующего rsync RearrangeDataDir, RearrangeIndexDir, etc)
        """
        path = self.get_parameter(parameter)
        if not path:
            raise SearchConfigsException('Can not get from config parameter {}'.format(parameter))
        path = path.strip()
        if not path.startswith('/db/'):
            raise SearchConfigsException(
                'Can not get /db rel path from config parameter: {} = {}'.format(parameter, path)
            )
        return path[4:]

    def apply_local_patch(self, params):
        self.text = cfgproc.patch_config(self.text, params.iteritems())

    def save_to_file(self, file_path):
        """
            Сохранить по указанному пути.
            Если не удалось сохранить, вызвается исключение.
            :param file_path: путь, по которому сохранять
        """
        file_path = os.path.abspath(file_path)
        fu.write_file(file_path, self.text)


class BasesearchWebConfig(SearchConfig):
    """
        Класс для работы с конфигом базового web поиска
    """
    INDEX_DIR_RE = re.compile(r"^IndexDir[\s]+(.*)")

    @classmethod
    def get_index_dir(cls, config_text):
        for line in config_text.split("\n"):
            maybe_index_dir = cls.INDEX_DIR_RE.findall(line.strip())
            if maybe_index_dir:
                logging.debug("IndexDir = %s", maybe_index_dir[0])
                return maybe_index_dir[0]
        eh.check_failed("Can't find IndexDir in config:\n{}".format(config_text))

    @classmethod
    def _get_production_config_text(cls):
        """
            Получить текст конфига базового из продакшена

            :return: текст конфига в виде строки
        """
        logging.info('Get baseserch production config')
        return cls._get_production_config_text_by_cms_tag(
            ['production_base', 'a_tier_{}'.format(bss.DEFAULT_TIER)]
        )


class BegemotConfig(SearchConfig):

    @classmethod
    def _check_file_path(cls, file_path):
        """
            Проверяет путь до конфига

            :param file_path: путь до файла
            :return: проверенный путь, если все проверки прошли успешно;
                в противном случае вызвает исключение SearchConfigsException
        """
        file_path = os.path.abspath(file_path)
        if not os.path.exists(file_path):
            raise SearchConfigsException('Path %s does not exist' % file_path)
        if not os.path.isdir(file_path):
            raise SearchConfigsException('Path %s is not file' % file_path)
        return file_path

    @classmethod
    def get_production_config(cls):
        return 'config is a directory'

    @classmethod
    def get_config_from_file(cls, file_path):
        """
            Заглушка
        """
        return cls._get_config_from_text("Not implemented")


def _replace_basesearches(text, basesearches_by_group):
    """
        Заменяет данные базовых источников

        :param basesearches_by_group: список базовых поисков вида
        [
            {
                'basesearch_type': 'WEB',
                'searchers': [('host_1', port_1), ('host_2', port_2), ...]
                'snippetizers': [('host_1', port_1), ('host_2', port_2), ...]
            },
            {
                'basesearch_type': 'FOREIGN',
                'searchers': [('host_1', port_1), ('host_2', port_2), ...]
                'snippetizers': [('host_1', port_1), ('host_2', port_2), ...]
            },
            ...
            {
                'auxsearch_type': 'BOOSTING_WIZARD',
                'searchers': [('host_1', port_1), ('host_2', port_2), ...]
                'collection': 'wizard',
                'options': 'ConnectTimeout=1s, ...'
            },
        ]
        basesearch_type - необязательный ServerDescr для каждой группы. Default = WEB
        Элементы с ключом auxsearch_type вместо basesearch_type попадают в <AuxSource>
        вместо <SearchSource>, в остальном полностью идентичны.
        Опциональный ключ 'collection' задаёт URL относительно хоста источника,
        по умолчанию "yandsearch".
        Опциональный ключ 'options' переопределяет значение Options в конфиге;
        по умолчанию подставляется значение Options от первой секции с ServerDescr,
        равным basesearch_type.

        Если параметр SearchSource не был найден, кидаем исключение.

        Вместо 'searchers' можно также использовать 'hosts_and_ports'
    """

    sources = cfgproc.get_metasearch_sources(text)

    new_parameter_and_value = ''
    for basesearches in basesearches_by_group:
        logging.debug('basesearches: %s', basesearches)

        source_type = 'SearchSource'
        if 'auxsearch_type' in basesearches:
            basesearch_type = basesearches['auxsearch_type']
            source_type = 'AuxSource'
        else:
            basesearch_type = basesearches.get('basesearch_type', 'WEB')

        basesearch_tier = basesearches.get('basesearch_tier')

        template_source = None

        for source in sources:
            if source_type != source.source_type:
                continue

            if basesearch_type and basesearch_type != source.config.get('ServerDescr'):
                continue

            if basesearch_tier and basesearch_tier != source.config.get('Tier'):
                continue

            template_source = source
            break

        if not template_source:
            for source in sources:
                if source_type == source.source_type:
                    template_source = source
                    break

        if not template_source:
            raise SearchConfigsException('no suitable search sources found')

        source_to_insert = cfgproc.MetasearchSource(source_type)

        copied_fields = ['ServerDescr', 'Options']
        if basesearch_tier:
            copied_fields.append('Tier')

        for f in copied_fields:
            if f in template_source.config:
                source_to_insert.config[f] = template_source.config[f]

        options = basesearches.get('options')
        if options:
            source_to_insert.config['Options'] = options

        collection = basesearches.get('collection', 'yandsearch')

        search_backends = basesearches.get('searchers') or basesearches['hosts_and_ports']
        if search_backends:
            source_to_insert.config['CgiSearchPrefix'] = ' '.join([
                'http://{}:{}/{}'.format(host, port, collection) for host, port in search_backends
            ])

        snippet_backends = basesearches.get('snippetizers', [])
        if snippet_backends:
            source_to_insert.config['CgiSnippetPrefix'] = ' '.join([
                'http://{}:{}/{}'.format(host, port, collection) for host, port in snippet_backends
            ])

        new_source = '<{}>\n'.format(source_to_insert.source_type)
        for k, v in sorted(source_to_insert.config.items()):
            new_source += '    {} {}\n'.format(k, v)

        new_source += '</{}>\n'.format(source_to_insert.source_type)

        new_parameter_and_value += new_source

    logging.info('Set new search sources:\n"    %s"', new_parameter_and_value)
    text = cfgproc.remove_section(text, 'Collection/SearchSource')
    text = cfgproc.remove_section(text, 'Collection/AuxSource')
    text = cfgproc.append_section(text, 'Collection', new_parameter_and_value)
    return text


class MidlesearchConfig(SearchConfig):
    """
        Класс для работы с конфигом среднего поиска
    """

    def replace_basesearches(self, basesearches):
        logging.info('Patch middlesearch config with basesearches %s', basesearches)
        self.text = _replace_basesearches(self.text, basesearches)

    def disable_cache(self):
        logging.info('disabling cache in middlesearch config')
        self.text = cfgproc.remove_section(self.text, 'Collection/QueryCache')
        self.text = cfgproc.remove_section(self.text, 'Collection/Squid')

    def enable_async_search(self, searcher="http2:+0"):
        meta_search_options = cfgproc.get_parameter_values(self.text, 'Collection/MetaSearchOptions')
        logging.debug('enable_async_search: meta_search_options: %r', meta_search_options)
        # after SEARCH-659 we need to split an option's value by spaces and add only if it does not exist,
        # but now we will add it unconditionally, as options can be duplicated
        meta_search_options.append('AsyncSearch')
        self.apply_local_patch({'Collection/MetaSearchOptions': ' '.join(meta_search_options)})
        common_source_options = cfgproc.get_parameter_values(self.text, 'Collection/CommonSourceOptions')
        common_source_options.append('Searcher=' + searcher)
        self.apply_local_patch({'Collection/CommonSourceOptions': ','.join(common_source_options)})

    def disable_async_search(self):
        meta_search_options = cfgproc.get_parameter_values(self.text, 'Collection/MetaSearchOptions')
        logging.debug('disable_async_search: meta_search_options: %r', meta_search_options)
        new_opts = []
        for opt in meta_search_options:
            opt = opt.replace('AsyncSearch', '')
            if opt:
                new_opts.append(opt)
        self.apply_local_patch({'Collection/MetaSearchOptions': ' '.join(new_opts)})

    def enable_ipv6(self):
        common_source_options = cfgproc.get_parameter_values(self.text, 'Collection/CommonSourceOptions')
        common_source_options.append('EnableIpV6=yes')
        self.apply_local_patch({'Collection/CommonSourceOptions': ','.join(common_source_options)})

    def add_dns_cache_hosts(self, recs):
        """
            Добавить указанные адреса в dns кеш (если их там ещё нет)
        """
        current_options = cfgproc.get_parameter_values(self.text, 'DNSCache/DNSCache')
        current_options = current_options[0]
        cache = {}
        for kv in current_options.split(' '):
            k, v = kv.split('=')
            cache[k] = v
        new_options = current_options
        for rec in recs:
            host, alias = rec
            if host not in cache:
                cache[host] = alias
                new_options += ' {}={}'.format(host, alias)
        self.apply_local_patch({'DNSCache/DNSCache': new_options})

    def set_src_subsources_one_name(self):
        """
            Делаем одинаковыми все адреса подисточников внутри одного источника
            (для предотвращения флуктуаций в выдаче при использовании в ней имени источника)
        """
        self.text = cfgproc.set_src_subsources_one_name(self.text)

    def remove_samohod(self):
        self.text = cfgproc.remove_section(self.text, 'Collection/SearchSource', 'Options', 'CacheId=samohod')
        self.text = cfgproc.remove_section(self.text, 'Collection/QueryCache', 'Id', 'samohod')


class MidlesearchIntConfig(MidlesearchConfig):
    """
        Класс для работы с конфигом промежуточного (int) поиска
    """
    @classmethod
    def _get_production_config_text(cls):
        """
            Получить текст конфига среднего инта из продакшена

            :return: текст конфига в виде строки
        """
        logging.info('Get middlesearch meta production config')
        return cls._get_production_config_text_by_cms_tag('int-production')


class MidlesearchMetaConfig(MidlesearchConfig):
    """
        Класс для работы с конфигом среднего(mmeta) поиска
    """
    @classmethod
    def _get_production_config_text(cls):
        """
            Получить текст конфига среднего инта из продакшена

            :return: текст конфига в виде строки
        """
        logging.info('Get middlesearch meta production config')
        return cls._get_production_config_text_by_cms_tag('meta-production')


class NoapacheupperConfig(MidlesearchConfig):
    """
        Класс для работы с конфигом верхнего поиска (отделённого от apache)
    """

    def __init__(self, config_text=None):
        super(NoapacheupperConfig, self).__init__(config_text=config_text)
        self.text = cfgproc.remove_section(self.text, 'Server/ReqansUnifiedAgentLog')
        self.text = cfgproc.remove_section(self.text, 'Server/ReqansUnifiedAgentMiscLog')
        self.text = cfgproc.remove_section(self.text, 'Server/AliceReqansUnifiedAgentLog')
        self.text = cfgproc.remove_section(self.text, 'Server/ErrorBoosterUnifiedAgentLog')
        self.text = cfgproc.remove_section(self.text, 'Server/XmlReqansUnifiedAgentLog')
        self.text = cfgproc.remove_section(self.text, 'Server/RtReqansUnifiedAgentLog')
        self.text = cfgproc.remove_section(self.text, 'Server/FatReqansUnifiedAgentLog')

    @classmethod
    def _get_production_config_text(cls):
        """
            Получить текст конфига из продакшена

            :return: текст конфига в виде строки
        """
        logging.info('Get noapacheupper production config')
        return cls._get_production_config_text_by_cms_tag('noapacheupper-production')  # TODO:validate cms tag

    def __eq__(self, other):
        def remove_comments(s):
            return '\n'.join(line for line in s.split('\n') if not line.startswith('#'))

        def remove_unstable_params(s):
            not_stable_params = [
                "Server/ConfigVersion",
                "Collection/RearrangeDataDir:",
                "Collection/RearrangeDynamicDataDir:",
                "Collection/RearrangeDataFastDir:",
            ]
            for param in not_stable_params:
                s = cfgproc.parameter_map(lambda x: "", s, param)
            return s

        logging.debug("Comparing:")
        patched_self_text = remove_unstable_params(remove_comments(self.text))
        patched_other_text = remove_unstable_params(remove_comments(other.text))
        logging.debug(patched_self_text)
        logging.debug(patched_other_text)
        return patched_self_text == patched_other_text


class SimpleConfigBase(object):
    """
        Общий класс для простых конфигов
    """

    def __init__(self, config_text=None):
        """
        """
        self.text = config_text

    @classmethod
    def _check_file_path(cls, file_path):
        """
            Проверяет путь до конфига

            :return: проверенный путь, если все проверки прошли успешно; в противном случае возвращает исключение
        """
        file_path = os.path.abspath(file_path)
        if not os.path.exists(file_path):
            raise SearchConfigsException('Path %s does not exist' % file_path)
        if not os.path.isfile(file_path):
            raise SearchConfigsException('Path %s is not file' % file_path)
        return file_path

    @classmethod
    def get_config_from_file(cls, file_path):
        """
            Получить объект конфига из переданного файла

            :return: объект конфига SearchConfig
        """
        file_path = cls._check_file_path(file_path)
        return cls(open(file_path).read())

    def apply_local_patch(self, params):
        eh.fail("apply_local_patch not implemented")

    def save_to_file(self, file_path):
        """
            Сохранить по указанному пути file_path

            :param file_path: путь, по которому сохранять
            :return:
                путь, по которому был сохранён конфиг;
                если не удалось сохранить, вызвается исключение
        """
        file_path = os.path.abspath(file_path)
        fu.write_file(file_path, self.text)


class XmlConfig(SimpleConfigBase):
    """
        Общий класс для конфигов поиска, основанных на xml
        (в основном подключаемые библиотеки из карт)
    """

    def __init__(self, config_text=None):
        """
        """
        super(XmlConfig, self).__init__(config_text=config_text)
        if self.text is None:
            self.domTree = None
        else:
            self.domTree = xml.dom.minidom.parseString(self.text)

    def apply_local_patch(self, params):
        self._apply_patch_to_node(
            node=self.domTree.documentElement,
            path=self.domTree.documentElement.tagName,
            params=params
        )
        self.text = self.domTree.documentElement.toxml()

    def _apply_patch_to_node(self, node, path, params):
        for child in node.childNodes:
            if child.nodeType != child.ELEMENT_NODE:
                continue
            child_path = '/'.join([path, child.tagName])
            if child_path in params:
                self._set_node_text(child, params[child_path])
            else:
                self._apply_patch_to_node(child, child_path, params)

    def _set_node_text(self, node, text):
        is_replaced = False
        for child in node.childNodes:
            if child.nodeType == child.TEXT_NODE:
                if is_replaced:
                    node.removeChild(child)
                else:
                    node.replaceChild(self.domTree.createTextNode(text), child)
                    is_replaced = True


class FusionConfig(SimpleConfigBase):
    def apply_local_patch(self, config_params):
        logging.info("stub patching config: %s ", config_params)
        config_params = {}


class IniConfig(SimpleConfigBase):
    """
        Общий класс для конфигов поиска, основанных на ini (например, spellchecker)
    """

    def __init__(self, config_text=None):
        """
        """
        super(IniConfig, self).__init__(config_text=config_text)

    def apply_local_patch(self, params):
        current_section = ''
        new_lines = ''
        for line in self.text.splitlines():
            line = line.strip()
            if line:
                pass
            if line.startswith('[') and line.endswith(']'):
                current_section = line[1:-1]
            else:
                name = line.split('=')[0].strip()
                new_value = params.get(current_section + '/' + name)
                if new_value:
                    line = '{0} = {1}'.format(name, new_value)
            new_lines += line + '\n'
        self.text = new_lines


class UpperConfig(object):
    """
        Обёртка для конфига верхнего
    """
    cproxy_start_port = 12000

    def __init__(self, file_path=None):
        """
            file_path - путь до конфига
            Если file_path не указан, конфиг считается пустым
        """
        self.__config = {}
        self.error = None
        self.__original_text = None
        file_path = os.path.abspath(file_path)
        self.file_path = file_path
        self.update()

    def __analyze_config(self, text=None):
        """
            Проанализировать конфиг и заполнить по нему внутреннее состояние UpperConfig
        """
        if text:
            self.__original_text = str(text)
        text = str(self.__original_text)
        result = {}

        # получаем collection ID
        r = re.search(r'Collection\W*id=\"(.*)\"', text)
        if r:
            result['collection_id'] = r.group(1).strip()
        # получаем параметры
        result['parameters'] = {}
        upper_parameters = (
            'IndexDir', 'MetaSearchOptions', 'TimeoutTable', 'KeepAlive', 'Compression',
            'NGroupsForSourceMultiplier', 'SimReqDaemon', 'ResInfoDaemon', 'EventLog',
            'FileNameWithPid', 'WizardPort', 'WizardTimeout', 'RemoteWizards',
            'WizardTasksTimeouts', 'LanguageDataPath', 'PureTrieConfigFile',
            'GazetteerFile', 'BigramsFile', 'SerializationProtocolVersion',
            'ReArrangeOptions', 'ConnectTimeout', r'CgiSearchPrefix:', r'RearrangeDataDir:',
            r'ReqWizardDir:', r'ReqWizardRuntimeDir:', r'ReqWizardRules:',
            'RequestThreads', 'RequestQueueSize', 'IsUpperSearch', 'WithPerlReport'
        )
        for parameter in upper_parameters:
            r = re.search(r'.*%s(.*)\n' % parameter, text)
            if r:
                result['parameters'][parameter] = r.group(1).strip()

        # получаем SearchPageTemplate
        search_page_template_pattern = re.compile(
            r'<SearchPageTemplate>(.*?)</SearchPageTemplate>',
            re.MULTILINE | re.DOTALL
        )
        search_page_template = search_page_template_pattern.search(text)
        if search_page_template:
            result['search_page_template'] = search_page_template.group(1).strip()

        # получаем секцию Server
        server_section_pattern = re.compile(r'<Server>(.*?)</Server>', re.MULTILINE | re.DOTALL)
        server_section = server_section_pattern.search(text)
        if server_section:
            result['server_section'] = server_section.group(1).strip()

        # получаем SearchSource-s
        result['search_sources'] = []
        search_source_parameters = ('ServerDescr', 'CgiSearchPrefix', 'Options', )
        text = re.sub(r'[\(\)]|@\d+', '', text)  # убираем круглые скобки и @ для источников
        search_source_pattern = re.compile(r'<SearchSource>(.*?)</SearchSource>', re.MULTILINE | re.DOTALL)
        for source in search_source_pattern.findall(text):
            new_source = {}
            for parameter in search_source_parameters:
                new_source[parameter] = ''
                r = re.search(r'.*%s(.*)' % parameter, source)
                if r:
                    new_source[parameter] = r.group(1).strip()
            result['search_sources'].append(new_source)
        if not result:
            return False
        self.__config = result
        return True

    @staticmethod
    def __check_host_url(url):
        try:
            urllib2.urlopen(url)
            return True
        except (urllib2.URLError, httplib.BadStatusLine):
            return False

    def left_only_one_host_in_sources(self, check_hosts=False):
        """
            Оставляет по одному хосту в источниках
            Если check_host равно True, будет оставлен только источник, хост которого доступен
            Если все хосты в источнике недоступны, а check_host задано как True, вызывается исключение
        """
        search_sources_in_config = self.__config.get('search_sources', [])
        if search_sources_in_config:
            new_search_sources = []
            for search_source in search_sources_in_config:
                cgi_prefix = search_source.get('CgiSearchPrefix', '')
                # удалим спецсимволы повторов
                re_filter = re.compile(r'(@\d)|\(|\)')
                cgi_prefix = re_filter.sub('', cgi_prefix)
                if not cgi_prefix:
                    raise SearchConfigsException('CgiSearchPrefix is empty for search source: {}'.format(search_source))

                for host_url in cgi_prefix.split(' '):
                    if check_hosts:
                        pass  # тут нужно реализовать проверку хостов и выбор первого отвечающего
                        # если таковых не нашлось, кидать исключение
                    else:
                        search_source['CgiSearchPrefix'] = host_url
                new_search_sources.append(search_source)
            self.__config['search_sources'] = new_search_sources

            # оставляем один источник для RemoteWizards
            remote_wizards = self.__config['parameters']['RemoteWizards']
            remote_wizards = re.sub(r'[()]|@\d+', '', remote_wizards)
            if not remote_wizards:
                raise SearchConfigsException('RemoteWizards list is empty')
            remote_wizards = remote_wizards.split()[0]
            self.__config['parameters']['RemoteWizards'] = remote_wizards
        return True

    def update(self, file_path=None):
        """
            Обновить конфиг из файла, если указан file_path, произойдёт обновление из него
            Если file_path не указан, будет использовано поле self.file_path
        """
        if file_path is None:
            file_path = self.file_path
        if os.path.exists(file_path) and os.path.isfile(file_path):
            config_text = open(self.file_path).read()
            self.__analyze_config(config_text)

    def save(self, file_path=None):
        """
            Сохранить по указанному пути file_path
        """
        if file_path is None:
            file_path = self.file_path
        fu.write_file(file_path, self.to_text())

    def to_text(self):
        """
            Возвращает конфиг в виде текста
        """
        first_line = '<Collection id="{}">\n'.format(self.__config.get('collection_id', 'yandsearch'))

        params = '\n'.join(['{} {}'.format(k, v) for k, v in self.__config.get('parameters', {}).iteritems()])

        search_page_templates = self.__config.get('search_page_template', '')
        if search_page_templates:
            search_page_templates = (
                "<SearchPageTemplate>"
                "    {}"
                "</SearchPageTemplate>"
            ).format(search_page_templates)

        server_section = self.__config.get('server_section', '')
        if server_section:
            server_section = (
                "<Server>"
                "    {}"
                "</Server>"
            ).format(server_section)

        search_sources_in_config = self.__config.get('search_sources', [])
        if search_sources_in_config:
            all_sources = []
            for search_source in search_sources_in_config:
                one_source = (
                    "<SearchSource>"
                    "    ServerDescr {descr}"
                    "    CgiSearchPrefix {prefix}"
                    "    Options {options}"
                    "</SearchSource>"
                ).format(
                    descr=search_source.get('ServerDescr', ''),
                    prefix=search_source.get('CgiSearchPrefix', ''),
                    options=search_source.get('Options', '')
                )
                all_sources.append(one_source)
            search_sources = '\n'.join(all_sources)
        else:
            search_sources = ''

        last_line = '</Collection>\n'
        return '\n'.join([
            server_section,
            first_line,
            params,
            search_page_templates,
            search_sources,
            last_line
        ])

    def to_dict(self):
        """
            Возвращает конфиг в виде словаря
        """
        return self.__config

    def replace_all_hosts_for_cproxy(self):
        """
            Заменяет в конфиге адреса вида hostname:port на 127.0.0.1:new_port
            Возвращает словарь вида {'new_port': 'hostname:port'}
        """
        upper_config_text = self.to_text()
        upper_config_text = upper_config_text.replace(
            'http://hghltd.yandex.net', 'http://hghltd.yandex.net:80'
        )
        match_iter = re.finditer(r'\s(\S*?\.yandex\.(ru|net):\d{1,5})', upper_config_text)
        upper_config_hosts = []
        for match_item in match_iter:
            founded_host = match_item.group(1)
            founded_host = founded_host.replace('http://', '')
            if founded_host not in upper_config_hosts:
                upper_config_hosts.append(founded_host)
        result = {}
        cproxy_port = self.cproxy_start_port
        for proxy_host in upper_config_hosts:
            # подбираем свободный порт
            while not network.is_port_free(cproxy_port):
                cproxy_port += 1
            result[str(cproxy_port)] = str(proxy_host)
            cproxy_port += 1
        # заменяем в тексте конфига хосты
        for key, value in result.items():
            upper_config_text = upper_config_text.replace(value, '127.0.0.1:{}'.format(key))
        self.__analyze_config(upper_config_text)
        return result
