import logging
import os.path

from sandbox.sandboxsdk import copy
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import task

from sandbox.projects.common.search import settings as media_settings
from sandbox.projects.common import decorators
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import sb_decorators

from sandbox.projects.common.search.database import cajuper_tracker
from sandbox.projects.common.search.database import iss_shards

import xmlrpclib


_DOWNLOAD_TIMEOUT = 7200
_CMS_REGISTRY = xmlrpclib.ServerProxy('http://cmsearch.yandex.ru/xmlrpc/bs')
OUT_RESOURCE_KEY = 'out_resource_id'


class ShardNameParameter(parameters.SandboxStringParameter):
    """
        Download database by given shard name (e.g. primus015-050-<timestamp>)
    """
    name = 'shard_name'
    description = 'Shard name'
    required = True


class ShardLinkParameter(parameters.SandboxStringParameter):
    """
        Database remote path
    """
    name = 'shard_link'
    description = 'Shard download link (rbtorrent):'
    required = False


class BaseLoadDatabaseTask(task.ForcedBackupTask):
    """
        Base class for all database downloading tasks

        Replacement of projects.common.BaseGetDatabaseTask
    """

    input_parameters = (
        ShardNameParameter,
        ShardLinkParameter,
    )

    def on_enqueue(self):
        task.SandboxTask.on_enqueue(self)

        attributes = {
            media_settings.SHARD_INSTANCE_ATTRIBUTE_NAME: self.ctx[ShardNameParameter.name],
            'backup_task': True,
        }

        self.ctx[OUT_RESOURCE_KEY] = self.create_resource(
            self.descr,
            self._get_database_local_path(),
            self._get_database_resource_type(),
            attributes=attributes
        ).id

    def _get_database_remote_path(self):
        logging.debug('BaseLoadDatabaseTask._get_database_remote_path()...')
        database_link = self.ctx[ShardLinkParameter.name]
        if not database_link:
            logging.debug('ShardLinkParameter option is empty. Looking in iss/cms...')
            shard_name = self.ctx[ShardNameParameter.name]
            database_link = (
                cajuper_tracker.get_shard_name(shard_name)
                or iss_shards.get_shard_name(shard_name)
                or self._get_cms_shard_name(shard_name)
            )
            logging.debug('The result iss/cms database link: {}'.format(database_link))
            eh.ensure(
                database_link,
                "Failed to find shard {} in registry".format(shard_name)
            )

        eh.ensure(
            database_link.startswith("rbtorrent:"),
            "Only rbtorrent protocol supported supported"
        )
        return database_link

    def _get_cms_shard_name(self, shard_name):
        try:
            return self._try_cms_shard_name(shard_name)
        except Exception as e:
            logging.info("Problem during search in CMS: {}".format(e))
            return None

    @sb_decorators.retries_on_different_clients()
    @decorators.retries(3, 1)
    def _try_cms_shard_name(self, shard_name):
        logging.info("Trying CMS...")
        try:
            return _CMS_REGISTRY.getShard(shard_name)['resource_url']
        except xmlrpclib.Fault:
            return None

    def _get_database_local_path(self):
        return self.abs_path('basesearch_database')

    def _get_database_resource_type(self):
        raise NotImplementedError()

    def _get_expected_files(self):
        return []

    def _get_unexpected_files(self):
        return []

    @sb_decorators.retries_on_different_clients()
    @decorators.retries(max_tries=3, delay=1)
    def _download_database(self, remote_path, local_path):
        copy.RemoteCopy(remote_path, local_path)(timeout=_DOWNLOAD_TIMEOUT)

    def _check_database(self):
        logging.debug('BaseLoadDatabaseTask._check_database()...')
        database_path = self._get_database_local_path()
        logging.debug('local db path: {}'.format(database_path))

        logging.debug('checking for expected files...')
        for file_name in self._get_expected_files():
            eh.ensure(
                os.path.exists(os.path.join(database_path, file_name)),
                'Expected {} file not found, probably wrong database type'.format(file_name)
            )

        logging.debug('checking for unexpected files...')
        for file_name in self._get_unexpected_files():
            eh.ensure(
                not os.path.exists(os.path.join(database_path, file_name)),
                'Unexpected {} file found, probably wrong database type'.format(file_name)
            )

    def on_execute(self):
        logging.debug('BaseLoadDatabaseTask.on_execute()...')
        self._download_database(self._get_database_remote_path(), self._get_database_local_path())
        self._check_database()
        self.mark_resource_ready(self.ctx[OUT_RESOURCE_KEY], force_backup=True)
