# -*- coding: utf-8 -*-
#
import logging

from sandbox.projects import resource_types
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.parameters import ResourceSelector
from sandbox.sandboxsdk import process

from sandbox.projects.common import link_builder as lb
from sandbox.projects.common.telegram_api import notify
import sandbox.projects.release_machine.core.const as rm_const


class Params:
    class EvlogdumpExecutable(ResourceSelector):
        name = 'evlogdump_executable_resource_id'
        description = 'evlogdump executable (optional, - for search errors in eventlog)'
        resource_type = resource_types.EVLOGDUMP_EXECUTABLE


class CheckEventlog(object):
    """
        Базовый класс для задач, в которых требуется проверять eventlog на наличие ошибок,
        если какие-то ошибки нужно игнорировать, эти фильтры реализуются переопределением
        методов on_event_*
        Проверка включается наличием в параметрах задачи-наследника непустого параметра
        Params.EvlogdumpExecutable aka ctx['evlogdump_executable_resource_id']
        (т.е. рекомендуется добавить Params.EvlogdumpExecutable во входные параметры задачи)
    """

    def __init__(self):
        self.checked_events = ['ErrorMessage', 'RearrangeError']

    def check_eventlog_errors(self, eventlog):
        if Params.EvlogdumpExecutable.name in self.ctx and self.ctx[Params.EvlogdumpExecutable.name]:
            logging.info('check eventlog: %s', eventlog)
            self.errors = []  # сюда собираем фатальные ошибки
            self.log_errors = []  # сюда собираем ошибки, которые желательно отобразить, но без FAILURE теста
            with open(eventlog, 'rb') as inp:
                proc = process.run_process(
                    [
                        self.sync_resource(self.ctx[Params.EvlogdumpExecutable.name]),
                        '-i',
                        ','.join(self.checked_events),
                    ],
                    stdin=inp,
                    log_prefix="parse_evlog",
                    wait=True,
                    check=True,
                    outputs_to_one_file=False,
                )
            with open(proc.stdout_path) as f:
                for line in f:
                    line = line.rstrip('\n')
                    tk = line.split('\t')
                    self.on_eventlog_line(tk)
                    if len(self.errors) == 100:
                        self.errors.append('reach errors limit (see log for details)')
                        break
            if self.log_errors:
                logging.warning('\n'.join(self.log_errors))
            if self.errors:
                raise SandboxTaskFailureError('\n'.join(self.errors))
            if self.ctx.get('invalid_reqids'):
                if self.id < 10000:
                    return
                bot = notify.create_bot(self, rm_const.TELEGRAM_TOKEN_NAME, rm_const.COMMON_TOKEN_OWNER)
                bot.send_message(
                    235176614,  # notify elshiko@
                    "Found invalid reqids in eventlog in task {}\n"
                    "See full list in task's log\nHere are some of them:\n{}".format(
                        lb.task_link(self.id, plain=True),
                        '\n'.join(self.ctx['invalid_reqids'][:5]),
                    )
                )
            if self.ctx.get('personalization_errors'):
                if self.id < 10000:
                    return
                bot = notify.create_bot(self, rm_const.TELEGRAM_TOKEN_NAME, rm_const.COMMON_TOKEN_OWNER)
                bot.send_message(
                    235176614,  # notify elshiko@
                    "Found personalization errors in eventlog in task {}\n"
                    "See full list in task's log\nHere are some of them:\n{}".format(
                        lb.task_link(self.id, plain=True),
                        '\n'.join(self.ctx['personalization_errors'][:5]),
                    )
                )
        else:
            logging.info('skip eventlog checking')

    def on_eventlog_line(self, tk):
        """
            Перегружается в случае добавления своих event-ов в self.checked_events
        """
        frame = int(tk[1])
        event = tk[2]
        if event == 'ErrorMessage':
            self.on_event_ErrorMessage(frame, tk[3])
        elif event == 'RearrangeError':
            self.on_event_RearrangeError(frame, tk)
        else:
            self.on_unknown_event(frame, tk)

    def on_event_ErrorMessage(self, frame, text):
        self.errors.append('ErrorMessage at frame {}: {}'.format(frame, text))

    def on_event_RearrangeError(self, frame, tk):
        self.errors.append('RearrangeError at frame {}: {}'.format(frame, ' '.join(tk[3:])))

    def on_unknown_event(self, frame, tk):
        self.errors.append('unknown event at frame {}: {}'.format(frame, ' '.join(tk[2:])))
