# -*- coding: utf-8 -*-

import collections
import os
import logging

import sandbox.projects.common.sdk_compat.task_helper as rm_task
from sandbox import common
from sandbox import sdk2
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk import util as sdkutil
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import errors as se

from sandbox.projects import resource_types
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import apihelpers
from sandbox.projects.common.search import components as sc
from sandbox.projects.websearch.middlesearch import resources as ms_resources
from . import evlogdump

NOT_RESPONDED_REQIDS_PERCENT = "not_responded_reqids_percent"
REARRANGE_ERROR_PERCENT = "rearrange_error_percent"


def get_eventlogs(
    log_names, hosts, target,
    port=8030,
    compress=True,
    logpath="/usr/local/www/logs/",
    event_ids=None,
    order_events=False,
    events_filter=None,
    start_tstamp=None,
    collection_filter='yandsearch',
    user=None,
):
    """
        :param log_names: eventlog file names to analyze
        :param hosts: hosts list
        :param target: file or directory to store results in
        :param port: instance port
        :param compress: enable compression (True by default)
        :param logpath: path for logs on target machines
        :param event_ids: event ids (or names) for evlogdump's -i option
        :param order_events: enable events proper ordering (False by default, slower)
        :param events_filter: dict with various filter criteria (max, min time, collection, frame, etc)
        :param start_tstamp: can be either timestamp or dict specifying average 'rps' and 'count' fields
            In that case starting timestamp will be calculated as (now - count / rps).
        :param collection_filter: filter requests by collection (yandsearch by default)
        :param user: skynet user name to use
    """
    if events_filter is None:
        events_filter = {}

    # if we are not allowed to get skynet_key, we'll got thousands
    # of 'Authentication error' from Skynet call
    # and will waste lots of time on sequential attempts.
    # So it is better to fail earlier
    sdk2.Vault.data(channel.task.owner, common.config.Registry().client.skynet_key.name)

    logging.info("Retrieving eventlogs started using user %s", user)
    alternate_evlogdump_sky_id = get_evlogdump_resource().skynet_id
    logging.info("evlogdump rbtorrent: %s", alternate_evlogdump_sky_id)
    results = evlogdump.run_evlogdump(
        log_names,
        hosts,
        port=port,
        compress=compress,
        logpath=logpath,
        event_ids=event_ids,
        order_events=order_events,
        events_filter=events_filter,
        collection_filter=collection_filter,
        start_tstamp=start_tstamp,
        alternate_evlogdump_sky_id=alternate_evlogdump_sky_id,
    )

    res = {}
    host_error_count = 0
    soft_error_count = 0
    unknown_error_count = 0

    for host, result, failure in results:
        if failure is not None:
            logging.info("Error at %s: %s", host, failure)
            host_error_count += 1
            continue

        if result["error"] is not None:
            err = result["error"]
            str_trace = '\n'.join(err["traceback"])
            logging.info(
                "Error received from host '%s':\nMessage: '%s'\nTraceback:\n%s",
                host,
                err["exception"],
                str_trace,
            )
            soft_error_count += 1
            continue

        remote_file_path = result["log_name"]
        events_count = result["count"]

        logging.info("File %s at %s (%s events)", remote_file_path, host, events_count)
        if events_count == 0:
            soft_error_count += 1
            continue

        if not os.path.isdir(target):
            dst_path = target
        else:
            dst_path = os.path.join(target, host)
            if compress:
                dst_path += '.gz'

        try:
            logging.info("Try to share dumped result to %s", dst_path)
            common.share.skynet_copy(host, remote_file_path, dst_path, user=user)
        except Exception:
            logging.info('Ignore unknown error: %s', eh.shifted_traceback())
            unknown_error_count += 1
            continue

        res[host] = events_count

    evlogdump.del_evlogdump_temp(results)

    logging.info(
        "Dumping finished with %s host errors, %s soft errors, %s unknown errors. Obtained %s results",
        host_error_count,
        soft_error_count,
        unknown_error_count,
        len(res),
    )
    return res


def locate_instability_problems(task, search_resource_id=None):
    """
        Seeks eventlog for unanswer-like problems.
        Typically used in on_failure() method to detect instability problems.
    """
    if task.ctx.get(NOT_RESPONDED_REQIDS_PERCENT) is None:
        return

    if search_resource_id is None:
        search_resource_id = task.ctx[sc.DefaultMiddlesearchParams.Binary.name]

    channel.task = task
    locate_problems(
        task,
        get_evlogdump(search_resource_id),
        channel.sandbox.get_resource(task.ctx['eventlog_out_resource_id']).path
    )


def _make_resource_with_problem(task, resource_name, content):
    fu.json_dump(resource_name, content, indent=2)
    resource = task.create_resource(
        resource_name, resource_name, resource_types.OTHER_RESOURCE
    )
    return resource


def _calc_percent_errors(task, message, errors, num_errors, resource):
    error_percent = round(100.0 * len(errors) / num_errors, 2)
    task.set_info(
        "There are {} % responses {}: {}".format(
            message, error_percent, lb.resource_link(resource.id)
        ),
        do_escape=False
    )
    return error_percent


def _parsed_event_log(evlogdump_exe_path, eventlog_path):
    return process.run_process(
        [
            evlogdump_exe_path, "-o", "-i",
            ",".join([
                "ContextCreated",
                "SubSourceError",
                "CacheUpdateJobStarted",
                "CacheUpdateJobFinished",
                "RearrangeError",
                "0"
            ]),
            eventlog_path
        ],
        log_prefix="dumped_eventlog"
    ).stdout_path


def locate_problems(task, evlogdump_exe_path, eventlog_path):
    """
    This function parses eventlog, creates resource with rearrange_errors, creates resource with unanswered queries,
    puts percent of them into context
    :param task: task with eventlog
    :param evlogdump_exe_path: path to eventlog dumper
    :param eventlog_path: path to eventlog
    :return: percent of unanswered queries
    """
    logging.info("Try to locate not responded reqids and rearrange errors")
    try:
        parsed_event_log_path = _parsed_event_log(evlogdump_exe_path, eventlog_path)
    except se.SandboxSubprocessError as err:
        eh.log_exception("Failed to check for not responded reqids: eventlog died", err)
        task.set_info("Failed to check for not responded reqids: evlogdump died")
        task.ctx[NOT_RESPONDED_REQIDS_PERCENT] = -1  # error marker
        task.ctx[REARRANGE_ERROR_PERCENT] = -1
        return task.ctx[NOT_RESPONDED_REQIDS_PERCENT]

    task.ctx[NOT_RESPONDED_REQIDS_PERCENT] = 0  # means, no unanswers
    task.ctx[REARRANGE_ERROR_PERCENT] = 0

    reqid = None
    errors = []
    all_reqs_count = 0
    result = {}
    rearrange_errors = False
    rearrange_result = {}
    for line in open(parsed_event_log_path):
        info = line.split()
        frame, event_type = info[1:3]
        if event_type == "ContextCreated":
            reqid = info[4]
            all_reqs_count += 1
        elif event_type == "SubSourceError":
            errors.append(info[7])
        elif event_type == "RearrangeError":
            rearrange_errors = True
        elif event_type in ["CacheUpdateJobStarted", "CacheUpdateJobFinished"]:
            if errors:
                logging.debug("Errors with cache in frame %s: %s\t%s", frame, reqid, dict(collections.Counter(errors)))
            errors = []
        elif event_type == "EndOfFrame":
            if errors:
                errs_dict = dict(collections.Counter(errors))
                result.setdefault(reqid, {})["frame: {}".format(frame)] = errs_dict
                errors = []
            if rearrange_errors:
                rearrange_result.setdefault(reqid, {})["frame: {}".format(frame)] = True
                rearrange_errors = False
            reqid = None
        else:
            eh.check_failed("Incorrect event_type found: '{}'".format(event_type))
    if result:
        resource = _make_resource_with_problem(task, "not_responded_reqids", result)
        task.ctx[NOT_RESPONDED_REQIDS_PERCENT] = _calc_percent_errors(
            task, "without answers", result, all_reqs_count, resource)
    if rearrange_result:
        resource = _make_resource_with_problem(task, "rearrange_errors", rearrange_result)
        task.ctx[REARRANGE_ERROR_PERCENT] = _calc_percent_errors(
            task, "with rearrange error", rearrange_result, all_reqs_count, resource)

    return task.ctx[NOT_RESPONDED_REQIDS_PERCENT]


def filter_eventlog(
    evlogdump_exe_path, eventlog_path,
    filter_command,
    output_path,
    evlogdump_options='-o'
):
    """
        Filters eventlog using filter_command (grep)
        and writes results into output_path.
    """
    full_command = "{} {} {} | {} > {}".format(
        evlogdump_exe_path, evlogdump_options, eventlog_path,
        filter_command,
        output_path
    )
    logging.info("filter eventlog using command: %s", full_command)
    evlogdump_return_code = os.system(full_command)
    if evlogdump_return_code:
        eh.check_failed('evlogdump process died with exit code {}'.format(evlogdump_return_code))

    out_size = os.path.getsize(output_path)
    eh.verify(out_size > 0, "Nothing remains after filtering, probably used wrong filter command")
    logging.info("size of filtered log: %s", out_size)
    logging.info("output lines: %s", sum(1 for _ in open(output_path)))


def get_evlogdump_resource(resource_id=None):
    """
        Получить объект ресурса evlogdump.
        Если resource_id не задан, возвращается evlogdump из последнего релиза среднего.

        Если resource_id указан, смотрим на задачу-источник этого ресурса.
        Проверяем, есть ли там EVLOGDUMP_EXECUTABLE.

        Если таковой не найден, возвращается evlogdump из последнего релиза среднего.
        Если найден, возвращается объект бинарника.
        Если найдено несколько таких ресурсов, возвращается первый из них.

        :resource_id: идентификатор ресурса, к которому пробуем найти evlogdump
        :return: объект ресурса подобранного evlogdump-а
    """
    evlogdump_resource = None
    if resource_id:
        logging.info('Try to find evlogdump for resource %s', resource_id)
        source_resource_object = channel.sandbox.get_resource(resource_id)
        evlogdump_resources = apihelpers.list_task_resources(
            source_resource_object.task_id, 'EVLOGDUMP_EXECUTABLE'
        )
        for resource in evlogdump_resources:
            if sdkutil.is_arch_compatible(resource.arch):
                evlogdump_resource = resource

    if not evlogdump_resource:
        logging.info('Try to get evlogdump from the last released middlesearch task')
        middle_exe_resource = apihelpers.get_last_released_resource(
            ms_resources.RankingMiddlesearchExecutable
        )
        evlogdump_resource_id = apihelpers.get_task_resource_id(
            rm_task.task_obj(middle_exe_resource.task_id).id,
            resource_types.EVLOGDUMP_EXECUTABLE
        )
        logging.info("Got evlogdump resource #%s", evlogdump_resource_id)
        evlogdump_resource = channel.sandbox.get_resource(evlogdump_resource_id)
    return evlogdump_resource


def get_evlogdump(resource_id=None):
    """
        Получить путь до evlogdump.

        :resource_id: идентификатор ресурса, к которому пробуем найти evlogdump
        :return: путь до подобранного evlogdump-а
    """
    evlogdump_resource = get_evlogdump_resource(resource_id)
    if evlogdump_resource:
        return channel.task.sync_resource(evlogdump_resource)
    else:
        eh.fail('Cannot find correct EVLOGDUMP_EXECUTABLE in the system.')
