import itertools
import logging
import re

from sandbox.sandboxsdk import paths

from sandbox.projects import resource_types
from sandbox.projects.common import file_utils as fu


def save_as_plain_text_queries_resource(
    task,
    description,
    queries_file_name=None,
    queries=None,
    attributes=None,
    mark_ready=False,
):
    """
        Converts queries to PLAIN_TEXT_QUERIES and saves it as resource.

        :param task: Sandbox task.
        :param description: Resource description.
        :param queries_file_name: Source file with queries.
        :param queries: List of queries (optional). If specified - use `queries_file_name` only for naming.
        :param attributes: Attributes to set to created resource.
        :param mark_ready: Mark created resource ready or not.
        :return: SandboxResource object.
    """
    queries_iterable = queries
    if queries is None:
        logging.info("Reading text queries from %s", queries_file_name)
        queries_iterable = open(queries_file_name)

    try:
        regex = re.compile(r'.*?\?')
        fixed_queries = itertools.imap(lambda query: regex.sub('?', query, count=1), queries_iterable)

        fixed_queries_file_name = paths.get_unique_file_name(task.abs_path(), queries_file_name + '.ptq.txt')
        logging.info("Output PLAIN_TEXT_QUERIES file name is %s", fixed_queries_file_name)
        queries_count = fu.write_lines(fixed_queries_file_name, fixed_queries)
        queries_resource = task.create_resource(
            "{} ({} queries)".format(description, queries_count),
            fixed_queries_file_name,
            resource_types.PLAIN_TEXT_QUERIES,
            attributes=attributes,
        )
        if mark_ready:
            task.mark_resource_ready(queries_resource)
        return queries_resource
    finally:
        if queries is None:
            # so queries_iterable is a file object
            if queries_iterable is not None:
                queries_iterable.close()
