# -*- coding: utf-8 -*-
"""
    This file should NOT have any dependence on custom libraries (including sandboxsdk)
"""

import os
import sys
import stat
import logging
import subprocess
import tempfile
import traceback
import gzip
import socket
import time
from array import array

from . import bsconfig
from . import iss


class Event:
    """
        Eventlog event representation
    """

    def __init__(self, line):
        self.line = line.rstrip()

        line = self.line.split('\t')
        self.time = int(line[0])
        self.frame = int(line[1])
        self.name = line[2]
        self.args = line[3:]
        self.collection = ''

    def __str__(self):
        return self.line


class _RunEvlogdump:
    """
        Skynet evlogdump wrapper
        :param logname:        log filename
        :param port:           config install port
        :param compress:       compress temporary file
        :param logpath:        path were logs are stored
        :param event_ids:      ids of events as list
        :param order_events:   order events by time if True and by frame if False
        :param events_filter:  dict(
            min    = 0    or minimum number of events to get
            max    = 0    or maximum number of events to get
            suffix = None or suffix appended to each line in log
            prefix = None or prefix prepended each line in log
            time   = True or False(None) log time of the event
            frame  = True or False(None) log frame id of the event
            name   = True or False(None) log name of the event
            fields = list() log this fields (numbers starting from 0 = next field after time, frame and name)
        )
    """

    # SEPE-8022
    osUser = 'prod-resource-getter'

    marshaledModules = __name__.split('.', 1)[:1]

    def __init__(
        self, log_names,
        port=8030,
        compress=True,
        logpath="/usr/local/www/logs/",
        event_ids=None,
        order_events=False,
        events_filter={},
        start_tstamp=None,
        collection_filter=None,
        alternate_evlogdump_sky_id=None,
    ):
        self.log_names = log_names
        self.port = port
        self.compress = compress
        self.log_path = logpath
        self.event_ids = event_ids
        self.order_events = order_events
        self.events_filter = events_filter
        self.collection_filter = collection_filter
        self.start_tstamp = start_tstamp
        self.alternate_evlogdump_sky_id = alternate_evlogdump_sky_id
        self.count = 0
        self.evlogdump_cmd_common = None

        self.events_filter.setdefault('min', 0)
        self.events_filter.setdefault('max', 0)
        self.events_filter.setdefault('suffix', '')
        self.events_filter.setdefault('prefix', '')

    def run(self):
        """
            Remote method
        """
        temp_name = None
        try:
            host = socket.gethostname()

            formatter = []
            if self.events_filter.get('time'):
                formatter.append(_EvlogdumpFormatter.append_time)
            if self.events_filter.get('frame'):
                formatter.append(_EvlogdumpFormatter.append_frame)
            if self.events_filter.get('name'):
                formatter.append(_EvlogdumpFormatter.append_name)
            if self.events_filter.get('collection'):
                formatter.append(_EvlogdumpFormatter.append_collection)
            if self.events_filter.get('fields'):
                formatter.append(_EvlogdumpFormatter.create_fields_appender(self.events_filter.get('fields')))

            if len(formatter) > 0:
                self.formatter = _EvlogdumpFormatter(self, formatter)
            else:
                self.formatter = _EvlogdumpFormatter(self, [_EvlogdumpFormatter.append_all])

            temp = tempfile.NamedTemporaryFile(
                delete=False,
                suffix="_evlog" if not self.compress else "_evlog.gz",
                dir="/var/tmp"
            )
            os.fchmod(temp.fileno(), stat.S_IRUSR | stat.S_IWUSR | stat.S_IRGRP | stat.S_IROTH)
            temp_name = temp.name
            if self.compress:
                temp.close()
                temp = gzip.open(temp_name, 'wb')

            self._dump(host, temp, temp_name)

            if self.count < self.events_filter['min']:
                if os.path.exists(temp_name):
                    os.unlink(temp_name)
                raise Exception("host {}: processed only {} events, cmd: '{}', logs {}".format(
                    host, self.count, self.evlogdump_cmd_common, self.log_names
                ))

            return {
                "log_name": temp_name,
                "count": self.count,
                "error": None,
            }
        except Exception:
            exc_type, exc, trace = sys.exc_info()
            return {
                "log_name": temp_name,
                "count": self.count,
                "error": {
                    "type": str(exc_type),
                    "exception": exc,
                    "traceback": traceback.format_exception(exc_type, exc, trace),
                }
            }

    def _evlogdump_path(self, port):
        try:
            return bsconfig.BSConfig()[port].evlogdump()
        except Exception:
            pass
        try:
            return iss.IssConfig()[port].evlogdump()
        except Exception as iss_err:
            all_errors = str(iss_err) + "\n"
            tmp_dir = "{}/tmp_evlogdump".format(tempfile.gettempdir())
            if not os.path.exists(tmp_dir):
                os.mkdir(tmp_dir)
            get_cmd = ["sky", "get", "-wu", "-d", tmp_dir, self.alternate_evlogdump_sky_id]
            get_proc = subprocess.Popen(
                " ".join(get_cmd), shell=True,
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE,
            )
            if get_proc.returncode:
                raise Exception("Stdout: {}\nStderr: {}".format(*get_proc.communicate()))
            evlogdump_path = os.path.join(tmp_dir, "evlogdump")
            all_errors += "path=" + evlogdump_path + "\n"
            list_tmp_dir = os.listdir(tmp_dir)
            if "evlogdump" not in list_tmp_dir:
                raise Exception("List dir = {}, all errors: {}".format(list_tmp_dir, all_errors))
            os.chmod(evlogdump_path, 0o755)
            return evlogdump_path

    def _dump(self, host, temp, temp_name):
        """
            Perform event log dump itself and temp files cleanup on failures
        """
        devnull = open(os.devnull, 'wb')
        try:
            port = self.port
            if isinstance(port, dict):
                port = port[host[0:host.find('.')]]

            evlogdump_cmd_common = [self._evlogdump_path(port)]
            if not self.order_events:
                evlogdump_cmd_common += ["-o"]
            if self.event_ids:
                event_ids = self.event_ids + [284]  # EnqueueYSRequest
                evlogdump_cmd_common += [
                    "-i",
                    ",".join([str(x) for x in event_ids])
                ]

            if self.start_tstamp:
                if isinstance(self.start_tstamp, dict):
                    start_timestamp_seconds = (
                        int(time.time()) - self.start_tstamp['count'] / self.start_tstamp['rps'] - 10 * 60
                    )
                    evlogdump_cmd_common += [
                        '-s',
                        str(start_timestamp_seconds * 1000000)
                    ]
                else:
                    evlogdump_cmd_common += [
                        '-s',
                        str(self.start_tstamp)
                    ]

            for log_name in self.log_names:
                if self._should_stop():
                    break

                abs_log_path = os.path.join(self.log_path, log_name)
                if not os.path.exists(abs_log_path):
                    continue
                self.evlogdump_cmd_common = evlogdump_cmd_common
                evlogdump_cmd = evlogdump_cmd_common + [abs_log_path]

                evlogdump = subprocess.Popen(
                    evlogdump_cmd,
                    stdout=subprocess.PIPE,
                    stderr=devnull,
                    bufsize=4096,
                )

                collection_name = ''
                collection_frame = None
                collection_always_match = self.collection_filter is None
                collection_match = True

                for line in evlogdump.stdout:
                    entry = Event(line)

                    if entry.name == "EnqueueYSRequest":
                        collection_frame = entry.frame
                        collection_name = entry.args[3][1:]
                        collection_match = collection_always_match or self.collection_filter == collection_name
                        continue

                    if not collection_always_match and (not collection_match or entry.frame != collection_frame):
                        continue

                    entry.collection = collection_name

                    self._write_entry(entry, temp)
                    if self._should_stop():
                        break

                if evlogdump.returncode:
                    raise Exception("host {}: process '{}' died with exit code {}".format(
                        host, evlogdump_cmd, evlogdump.returncode
                    ))
        except Exception:
            os.unlink(temp_name)
            raise
        finally:
            temp.close()
            devnull.close()

    def _write_entry(self, event, out):
        """
            Write formatted event
            :param event: event object
            :param out: output stream
        """
        output = array('c')
        self.formatter(event, output)
        out.write("{1}{0}{2}\n".format(
            output.tostring().lstrip(),
            self.events_filter['prefix'],
            self.events_filter['suffix']
        ))

        self.count += 1

    def _should_stop(self):
        return 0 < self.events_filter['max'] <= self.count


class _EvlogdumpFormatter(list):
    def __init__(self, ev, *args):
        self.ev = ev
        list.__init__(self, *args)

    def __call__(self, *args):
        for i in self:
            i(*args)

    @staticmethod
    def append_all(event, output):
        output.fromstring(str(event))

    @staticmethod
    def append_time(event, output):
        output.fromstring("\t{0}".format(event.time))

    @staticmethod
    def append_frame(event, output):
        output.fromstring("\t{0}".format(event.frame))

    @staticmethod
    def append_name(event, output):
        output.fromstring("\t{0}".format(event.name))

    @staticmethod
    def append_collection(event, output):
        output.fromstring("\t{0}".format(event.collection))

    @staticmethod
    def create_fields_appender(fields):

        def _appender(event, output):
            for i in fields:
                output.fromstring("\t{0}".format(event.args[i]))
        return _appender


def run_evlogdump(
    log_names, hosts,
    port=8030,
    compress=True,
    logpath="/usr/local/www/logs/",
    event_ids=None,
    order_events=False,
    events_filter={},
    collection_filter=None,
    start_tstamp=None,
    alternate_evlogdump_sky_id=None,
):
    import api.cqueue
    client = api.cqueue.Client(implementation='cqudp', netlibus=True)
    runner = _RunEvlogdump(
        log_names=log_names,
        port=dict((hosts[i], port[i]) for i in range(0, len(hosts))) if isinstance(port, list) else port,
        compress=compress,
        logpath=logpath,
        event_ids=event_ids,
        order_events=order_events,
        events_filter=events_filter,
        collection_filter=collection_filter,
        start_tstamp=start_tstamp,
        alternate_evlogdump_sky_id=alternate_evlogdump_sky_id,
    )
    logging.info("Running evlogdump using %s on %s", runner.osUser, hosts)
    results = client.run(hosts, runner).wait()
    return results


def del_evlogdump_temp(results):
    import api.cqueue
    client = api.cqueue.Client(implementation='cqudp')
    results = [r for r in results if not r[2]]
    if results:
        hosts = [r[0] for r in results]
        host2file = dict([(r[0], r[1][0]) for r in results])
        client.run(hosts, _CleanupEvlogdumpTemp(host2file)).wait()


class _CleanupEvlogdumpTemp:
    def __init__(self, host2file):
        self.host2file = host2file

    def run(self):
        os.unlink(self.host2file[socket.gethostname()])
        return 0
