#!/usr/bin/env python
# -*- coding: utf-8 -*-

import logging
import subprocess
import select
import os

from sandbox import sdk2
from sandbox.common.types import resource as ctr
from sandbox.projects import resource_types
from sandbox.projects.common import decorators
from sandbox.projects.common import resource_selectors


class FactorsDecompressor(object):
    _command = None
    _error = False

    @staticmethod
    @decorators.retries(3, delay=5)
    def _get_decoder_resource(new_decompressor):
        if new_decompressor:
            return sdk2.Resource.find(
                type=resource_types.COMPRESSED_FACTORS_DUMP_EXECUTABLE,
                state=ctr.State.READY,
                attrs={"new_decompressor": "yes"},
            ).first()
        else:
            return sdk2.Resource[resource_selectors.by_last_released_task(
                resource_types.COMPRESSED_FACTORS_DUMP_EXECUTABLE
            )[0]]

    @staticmethod
    def init(args=(), new_decompressor=False):
        """
            Initialize command for decompress. Sync decompressor binary.
            :param args: iterable with arguments for decoder
            :param new_decompressor: use new decompressor or not
        """
        if FactorsDecompressor._command is None:
            decoder_resource = FactorsDecompressor._get_decoder_resource(new_decompressor)
            logging.debug("FOUND FactorsDecompressor: %s", decoder_resource.id)
            FactorsDecompressor._command = [str(sdk2.ResourceData(decoder_resource).path)] + list(args)
        logging.debug("FactorsDecompressor command: %s", FactorsDecompressor._command)

    @staticmethod
    def extract(value):
        try:
            FactorsDecompressor.init()
            cmd = FactorsDecompressor._command + ["--input", value]
            # Do not use `process.run_process` here, because it produces insanely huge logs
            proc = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
            out, err = proc.communicate()
            if proc.returncode:
                if not FactorsDecompressor._error:
                    logging.debug("Failed to execute COMPRESSED_FACTORS_DUMP_EXECUTABLE: %s, Error %s", cmd, err)
                    FactorsDecompressor._error = True
                return value

            return out
        except Exception as error:
            if not FactorsDecompressor._error:
                logging.debug("Failed to call extract_compressed_factors(): %s", error)
                FactorsDecompressor._error = True

        return value


def extract_compressed_factors(value):
    return FactorsDecompressor.extract(value)


class PipedFactorsDecompressor(object):
    _binary = None
    _pipe_read_size = 65536

    @staticmethod
    @decorators.retries(3, delay=5)
    def _get_decoder_resource():
        return sdk2.Resource.find(
            type=resource_types.COMPRESSED_FACTORS_DUMP_EXECUTABLE,
            state=ctr.State.READY,
            attrs={"pipe_support": "yes"},
        ).first()

    @staticmethod
    def init():
        """
            Sync decompressor binary.
            Useful to avoid downloading the same resource in several threads.
        """
        if PipedFactorsDecompressor._binary is None:
            decoder_resource = PipedFactorsDecompressor._get_decoder_resource()
            logging.info("FOUND PipedFactorsDecompressor: %s", decoder_resource.id)
            PipedFactorsDecompressor._binary = str(sdk2.ResourceData(decoder_resource).path)

    def __bool__(self):
        return PipedFactorsDecompressor._binary is not None

    def __init__(self, input_mode="fs"):
        self.pipe_ok = False
        self.input_mode = input_mode

    def open_pipe(self):
        PipedFactorsDecompressor.init()  # no-op if already called
        self.process = subprocess.Popen(
            args=[PipedFactorsDecompressor._binary, "--output-mode", "oneline", "--input-mode", self.input_mode],
            stdin=subprocess.PIPE,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE)
        self.pipe_ok = True

    def _log_child_stderr(self, stderr, write_incomplete):
        while True:
            pos = stderr.find('\n')
            if pos == -1:
                if write_incomplete and stderr:
                    logging.error("compressed_factors_decoder stderr: {}".format(stderr))
                    return ""
                return stderr
            logging.error("compressed_factors_decoder stderr: {}".format(stderr[:pos]))
            stderr = stderr[pos + 1:]

    def _read_portion(self, should_live):
        fullerr = ""
        while True:
            retcode = self.process.poll()
            if retcode is not None:
                if should_live or retcode != 0:
                    logging.error("compressed_factors_decoder died unexpectedly with status {}".format(retcode))
                while True:
                    err = os.read(self.process.stderr.fileno(), self._pipe_read_size)
                    if not err:
                        break
                    fullerr += err
                self._log_child_stderr(fullerr, True)
                self.pipe_ok = False
                return None
            ready, _, _ = select.select([self.process.stdout, self.process.stderr], [], [], 0.1)
            if self.process.stderr in ready:
                fullerr += os.read(self.process.stderr.fileno(), self._pipe_read_size)
                fullerr = self._log_child_stderr(fullerr, False)
                continue
            if self.process.stdout in ready:
                self._log_child_stderr(fullerr, True)
                return os.read(self.process.stdout.fileno(), self._pipe_read_size)

    def extract(self, value):
        if not self.pipe_ok:
            return value
        try:
            self.process.stdin.write(value + '\n')
            self.process.stdin.flush()
        except IOError:
            logging.error("failed to communicate with compressed_factors_decoder")
            self.close_pipe()  # print remaining stderr to logs
            return value
        fullout = ""
        while True:
            out = self._read_portion(True)
            if out is None:
                return value
            fullout += out
            pos = fullout.find('\n')
            if pos == len(fullout) - 1:
                return fullout[:pos]
            if pos != -1:
                logging.error("output of compressed_factors_decoder is expected to be a single line")
                self.close_pipe()
                return value

    def close_pipe(self):
        if not self.pipe_ok:
            return
        self.process.stdin.close()
        while self._read_portion(False) is not None:
            pass
        self.pipe_ok = False
