import itertools
from six.moves import range

from sandbox.projects import resource_types
from sandbox.projects.websearch.middlesearch import resources as ms_resources
from sandbox.projects.common.search import requester as search_requester

from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import parameters


BASESEARCH_API = (
    (
        '/yandsearch?info=getconfig',
        (r'^ *RequestThreads +[1-9]+', r'^ *SnippetThreads +[1-9]+'),
    ),
    (
        '/yandsearch?info=checkconfig:read-only:1;deep:0',
        (r'base-linux-bin-md5', r'base-cfg-md5', r'base-frm-md5', r'base-path-revision'),
    ),
    (
        '/yandsearch?info=getversion',
        (r'Top src dir:', r'Hostname:', r'Compiler:')
        # Note: Distbuild results doesn't contain 'URL:', 'Last Changed Rev:', and 'Last Changed Author:' fields
    ),
)


BASESEARCH_CRASHERS = (
    (
        # this coredumped old implementation...
        '/yandsearch?DF=da&dh=100:1&ms=proto',
        (r'', )
    ),
    (
        # ... and this one coredumped new implementation
        '/yandsearch?DF=da&dh=100:1&ms=proto&hr=da',
        (r'', )
    ),
    (
        # SEARCH-2516 crash
        '/yandsearch?info=rf:docid:1',
        (r'No qtree', )
    ),
    (
        # SEARCH-2516 crash
        '/yandsearch?info=rf:docid:1&text=1&ms=proto&hr=da',
        (r'No qtree', )
    ),
)


IMAGES_MIDDLESEARCH_API = (
    (
        '/admin?action=clearcache&id=quick',
        (r'^Cache is cleared with code 0',)
    ),
    (
        '/admin?action=clearcache&id=imgscbir',
        (r'^Cache is cleared with code 0',)
    ),
)


VIDEO_MIDDLESEARCH_API = (
    (
        '/admin?action=clearcache&id=videoquick',
        (r'^Cache is cleared with code 0',)
    ),
)


BASE = "base"
MIDDLE = "middle"
VIDEO = 'video'
IMAGES = 'images'
WEB = 'web'
QUICK = 'quick'


INFO_REQUESTS_BASE = {
    WEB: BASESEARCH_API,
    IMAGES: BASESEARCH_API,
    VIDEO: BASESEARCH_API,
    QUICK: BASESEARCH_API,
}

CRASH_REQUESTS_BASE = {
    WEB: BASESEARCH_CRASHERS,
    IMAGES: BASESEARCH_CRASHERS,
    VIDEO: BASESEARCH_CRASHERS,
    QUICK: BASESEARCH_CRASHERS,
}

INFO_REQUESTS_MIDDLE = {
    VIDEO: VIDEO_MIDDLESEARCH_API,
    IMAGES: IMAGES_MIDDLESEARCH_API,
}


class QueriesParameter(parameters.ResourceSelector):
    name = 'queries_resource_id'
    description = 'Queries'
    resource_type = [
        resource_types.PLAIN_TEXT_QUERIES,
        ms_resources.WebMiddlesearchPlainTextQueries,
        resource_types.IMAGES_MIDDLESEARCH_PLAIN_TEXT_REQUESTS,
        resource_types.VIDEO_MIDDLESEARCH_PLAIN_TEXT_REQUESTS,
    ]
    required = False


class QueriesLimitParameter(parameters.SandboxIntegerParameter):
    name = 'queries_limit'
    description = 'Limit number of queries to read (0 = all)'
    default_value = 500


def run_component_requesters(
    task,
    component,
    info_requests=None,
    info_sessions=5,
    ignore_error=False,
    retry_limit=10
):
    with component:
        if info_requests:
            for _ in range(info_sessions):
                info_requester = search_requester.InfoRequester(task.ctx, info_requests)
                info_requester.re_try_limit = retry_limit
                __run_requester(info_requester, component, ignore_error=ignore_error)

        if task.ctx[QueriesParameter.name]:
            simple_requester = search_requester.SimpleRequester(task.ctx, __make_requests_iterator(task))
            simple_requester.re_try_limit = 10
            __run_requester(simple_requester, component, ignore_error=ignore_error)


def __run_requester(requester, target, ignore_error=False):
    requester.use_component(target)
    if requester.error and not ignore_error:
        raise errors.SandboxTaskFailureError("Error in requester: {}".format(requester.error))


def __make_requests_iterator(task):
    queries_path = task.sync_resource(task.ctx[QueriesParameter.name])
    queries_limit = task.ctx[QueriesLimitParameter.name]
    with open(queries_path) as queries_file:
        if queries_limit:
            queries_file = itertools.islice(queries_file, queries_limit)
        for query in queries_file:
            query = query.strip()
            if query.startswith("/"):
                yield query
            else:
                yield "/yandsearch" + query
