import os
import logging
import shutil

from sandbox.common import utils as common_utils
import sandbox.common.types.task as ctt
import sandbox.common.types.resource as ctr
import sandbox.projects.common.constants as consts

from sandbox import sdk2
from sandbox.sandboxsdk.channel import channel

from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common.sdk_compat import task_helper as th
from sandbox.projects import resource_types

from sandbox.projects.resource_types import releasers as resource_releasers
from sandbox.projects.balancer import resources as balancer_resources


class GDB_SEARCH_TOOLKIT(sdk2.resource.AbstractResource):
    """
        GDB bundle released with Search runtime (SEPE-7767)
    """
    releasable = True
    auto_backup = True
    any_arch = False
    releasers = resource_releasers.unique_list(
        resource_releasers.base_releasers +
        resource_releasers.middle_releasers +
        resource_releasers.geowizard_releasers +
        resource_releasers.saas_releasers +
        resource_releasers.apphost_releasers +
        resource_releasers.noapacheupper_releasers +
        balancer_resources.balancer_releasers +
        resource_releasers.voiceserv_asr_releasers +
        resource_releasers.voiceserv_tts_releasers +
        resource_releasers.vh_frontend_releasers +
        resource_releasers.video +
        [
            "alborisov",
            "drauggi",
            "robot-srch-releaser"
        ]
    )


def append_to_stable_only(task, arch=None, arcadia_url_key=consts.ARCADIA_URL_KEY):
    """
        SDK1/SDK2-compatible wrapper for GDB appending.
        Appends GDB_SEARCH_TOOLKIT (gdb bundle for production) to task' resources
        if it is built from stable tag (or branch).
        See SEARCH-8381 and SEPE-7767 for details.

        :param task: task object
        :param arch: None means arch of current system
        :param arcadia_url_key: input parameter with arcadia url
    """
    checkout_arcadia_from_url = th.input_field(task, arcadia_url_key, '')
    if '/stable-' not in checkout_arcadia_from_url:
        logging.info("GDB appending skipped: arcadia url was `%s`", checkout_arcadia_from_url)
        return

    if th.is_sdk2(task):
        append_to_release_sdk2(task, arch)
    else:
        append_to_release(task, arch)


def append_to_release(task, arch=None):
    """
        Appends GDB_SEARCH_TOOLKIT (gdb bundle for production) to task' resources (sdk1 version)
        See SEPE-7767 for details.
        @param task: task object
        @param arch: None means arch of current system
    """
    # This code may fail by various reasons and we don't want to fail entire task in this case.
    # So please do not remove exception handler wrapper.
    try:

        my_toolkits = channel.sandbox.list_resources(
            resource_type=GDB_SEARCH_TOOLKIT,
            task_id=task.id,
            status='READY',
        )
        if my_toolkits:
            logging.info("We already have at least one GDB toolkit: resource:%s", my_toolkits[0].id)
            return

        # only lucid platforms are suitable (see SEARCH-4963)
        # so we get a bunch of versions and select proper one
        gdb_toolkits = channel.sandbox.list_resources(
            resource_type=resource_types.GDB_TOOLKIT,
            status='READY',
            all_attrs={
                "released": "stable",
            },
            arch=arch,
            limit=20,
        )

        gdb_resource = None
        for gdb_toolkit in gdb_toolkits:
            logging.debug("GDB candidate: %s", gdb_toolkit.id)
            if 'platform' in gdb_toolkit.attributes and 'lucid' in gdb_toolkit.attributes['platform']:
                # we found him
                gdb_resource = gdb_toolkit
                break

        if gdb_resource is None:
            raise Exception(
                "There is no stable lucid GDB_TOOLKIT for arch {} (None means arch of current system)".format(arch)
            )
        gdb_toolkit = task.sync_resource(gdb_resource.id)
        new_name = '{}-{}'.format(gdb_resource.arch, os.path.basename(gdb_toolkit))
        gdb_toolkit_copy = os.path.join(task.abs_path(), new_name)
        # yes, we need to copy file to task' folder, otherwise resource won't be created
        shutil.copyfile(gdb_toolkit, gdb_toolkit_copy)

        search_gdb_resource = task.create_resource(
            "Last stable released GDB toolkit from resource:{}".format(gdb_resource.id),
            resource_path=gdb_toolkit_copy,
            resource_type=GDB_SEARCH_TOOLKIT,
            arch=gdb_resource.arch,
        )
        task.mark_resource_ready(search_gdb_resource)
    except Exception as err:
        eh.log_exception("Failed to append gdb", err)


def append_to_release_sdk2(task, arch=None):
    """
        Appends GDB_SEARCH_TOOLKIT (gdb bundle for production) to task' resources (sdk2 version)
        See SEPE-7767 for details.
        @param task: task object
        @param arch: None means arch of current system
    """
    # This code may fail by various reasons and we don't want to fail entire task in this case.
    # So please do not remove exception handler wrapper.
    try:
        search_toolkit = sdk2.Resource.find(
            type=str(GDB_SEARCH_TOOLKIT), task_id=task.id, state=ctr.State.READY
        ).first()

        if search_toolkit:
            logging.info("We already have at least one GDB toolkit: resource: %s", search_toolkit.id)
            return

        arch = arch if arch else common_utils.get_sysparams()["arch"]

        gdb_toolkits = sdk2.Resource.find(
            type=str(resource_types.GDB_TOOLKIT),
            arch=arch,
            attrs={
                "released": ctt.ReleaseStatus.STABLE,
            },
        ).order(-sdk2.Resource.id).limit(20)

        gdb_resource = None
        for gdb_toolkit in gdb_toolkits:
            logging.debug("GDB candidate: %s", gdb_toolkit.id)
            if hasattr(gdb_toolkit, 'platform') and 'lucid' in gdb_toolkit.platform:
                # we found him
                gdb_resource = gdb_toolkit
                break

        if not gdb_resource:
            raise Exception(
                "There is no stable GDB_TOOLKIT for arch {} (None means arch of current system)".format(arch)
            )

        logging.debug('Got gdb toolkit: %s', gdb_resource.id)
        gdb_toolkit = str(sdk2.ResourceData(gdb_resource).path)
        new_name = '{0}-{1}'.format(gdb_resource.arch, os.path.basename(gdb_toolkit))
        shutil.copyfile(gdb_toolkit, new_name)

        GDB_SEARCH_TOOLKIT(
            task, "Last stable released GDB toolkit from resource:{}".format(gdb_resource.id),
            new_name, arch=gdb_resource.arch
        )

    except Exception as err:
        eh.log_exception("Failed to append gdb", err)
