# -*- coding: utf-8 -*-

import os.path

import sandbox.common.types.client as ctc

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk import parameters as sp

from sandbox.projects import resource_types
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils
from sandbox.projects.common.search import bugbanner
from sandbox.projects.common.base_search_quality import dolbilka_response_parser
from sandbox.projects.common.base_search_quality import dolbilka_response_diff
from sandbox.projects.images.models import resources as images_models_resources


_FACTOR_NAMES_FILE = 'factor_names.txt'
"""
    Имя файла с факторами
"""

_OUTPUT_FILE = 'output.txt'
"""
    Имя файла с выходными данными от основного
    запуска idx_ops
"""

_OUTPUT2_FILE = 'output2.txt'
"""
    Имя файла с выходными данными от повторного
    запуска idx_ops
"""

_EXECUTABLE_RESOURCE_ID = 'executable_resource_id'
"""
    Название в контексте идентификатора ресурса
    исполняемого файла
"""

_DATABASE_RESOURCE_ID = 'search_database_resource_id'
"""
    Название в контексте идентификатора ресурса
    базы данных
"""


class QueriesParameter(sp.LastReleasedResource):
    """
        Файл запросов
    """
    name = 'queries_resource_id'
    description = 'Queries'
    resource_type = resource_types.PLAIN_TEXT_QUERIES


class QueriesLimitParameter(sp.SandboxIntegerParameter):
    """
        Ограничение на количество обрабатываемых запросов
    """
    name = 'queries_limit'
    description = 'Limit number of queries to read (0 = all)'
    default_value = 0


class MinOutputSizeParameter(sp.SandboxIntegerParameter):
    """
        Минимальный размер выходного файла
    """
    name = 'min_output_size'
    description = 'Min output size'
    default_value = 0


class RecheckResultsParameter(sp.SandboxBoolParameter):
    """
        Запускать процесс дважды для сверки результатов
    """
    name = 'recheck_results'
    description = 'Recheck results'
    default_value = False


class SavePatchedQueriesParameter(sp.SandboxBoolParameter):
    """
        Следует ли сохранять файлик с пропатченными запросами (номер запроса TAB запрос)
    """
    name = 'save_patched_queries'
    description = 'Save patched queries'
    default_value = False


class ModelsParameter(sp.ResourceSelector):
    """
        Модели MatrixNet
    """
    name = 'models_resource_id'
    description = 'Dynamic models archive'
    resource_type = [
        resource_types.DYNAMIC_MODELS_ARCHIVE,
        images_models_resources.IMAGES_DYNAMIC_MODELS_ARCHIVE,
        resource_types.VIDEO_DYNAMIC_MODELS_ARCHIVE
    ]


class IdxOpsFactorNamesArgs(sp.SandboxStringParameter):
    """
    Additional parameters for idx_ops factor_names run (comma separated)

    """
    name = "idx_ops_factor_names_args"
    description = "Add args for idx_ops factor_names run (separated by ',')"
    default_value = ""


class BaseGetIdxOpsResponses(bugbanner.BugBannerTask):
    """
        Базовый класс для получения вывода
        с утилиты idx_ops
    """

    client_tags = ctc.Tag.LINUX_PRECISE

    idx_ops_serp_args = ['--printdocid']

    input_parameters = (
        IdxOpsFactorNamesArgs,
        QueriesParameter,
        QueriesLimitParameter,
        ModelsParameter,
        MinOutputSizeParameter,
        RecheckResultsParameter,
        SavePatchedQueriesParameter,
    )

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)

        channel.task = self
        resource = self.create_resource(
            self.descr,
            'idx_ops_output',
            resource_types.IDX_OPS_OUTPUT,
            arch='any'
        )
        self.ctx['out_resource_id'] = resource.id

    def on_execute(self):
        self.add_bugbanner(bugbanner.Banners.WebBaseSearch, add_responsibles=["gotmanov"])
        output_directory = channel.sandbox.get_resource(self.ctx['out_resource_id']).path
        paths.make_folder(output_directory, delete_content=True)

        queries_path = self.sync_resource(self.ctx[QueriesParameter.name])
        patched_queries_path = os.path.join(self.abs_path(), 'patched_queries.txt')
        _patch_queries(
            queries_path,
            patched_queries_path,
            queries_limit=utils.get_or_default(self.ctx, QueriesLimitParameter),
        )

        if self.ctx.get(SavePatchedQueriesParameter.name):
            r = self.create_resource('patched queries', patched_queries_path, resource_types.PLAIN_TEXT_QUERIES)
            self.mark_resource_ready(r)

        factors_count = self._get_factor_names(os.path.join(output_directory, _FACTOR_NAMES_FILE))

        output = self._get_output(
            os.path.join(output_directory, _OUTPUT_FILE),
            patched_queries_path,
            factors_count
        )
        if self.ctx[RecheckResultsParameter.name]:
            output2 = self._get_output(
                os.path.join(output_directory, _OUTPUT2_FILE),
                patched_queries_path,
                factors_count
            )
            self._compare_outputs(output_directory, queries_path, output, output2)

    def _get_factor_names(self, factors_path):
        """
            Создаёт файл с именами факторов в каталоге с выходными данными
        """

        executable_path = self.sync_resource(self.ctx[_EXECUTABLE_RESOURCE_ID])

        with open(factors_path, 'w+') as factors_file:
            command = [executable_path, "factor_names"]
            if self.ctx.get(IdxOpsFactorNamesArgs.name):
                command += self.ctx[IdxOpsFactorNamesArgs.name].split(",")
            process.run_process(
                command,
                stdout=factors_file,
                log_prefix='idx_ops_getfactornames',
                wait=True,
                timeout=20,
                outputs_to_one_file=False,
            )
            factors_file.seek(0)
            return sum(1 for _ in factors_file)

    def _get_output(self, output_path, queries_path, factors_count):
        """
            Создаёт файл с выводом от утилиты idx_ops
        """

        executable_path = self.sync_resource(self.ctx[_EXECUTABLE_RESOURCE_ID])
        db_path = self.sync_resource(self.ctx[_DATABASE_RESOURCE_ID])
        # debug binary wastes more time
        exec_timeout = 7200 if '_DEBUG_' in self.descr else 3600

        with open(output_path, 'w') as output_file:
            if self.ctx.get(ModelsParameter.name):
                self.idx_ops_serp_args += ['--models', self.sync_resource(self.ctx[ModelsParameter.name])]
            process.run_process(
                [
                    executable_path,
                    'serp',
                ] + self.idx_ops_serp_args + [
                    '--threads', str(self.client_info['ncpu']),
                    '--queries', queries_path,
                    db_path,
                ],
                stdout=output_file,
                log_prefix='idx_ops_getoutput',
                wait=True,
                timeout=exec_timeout,
                outputs_to_one_file=False,
            )

        # verify output
        output = dolbilka_response_parser.parse_and_group_by_query(output_path, False, factors_count=factors_count)
        eh.ensure(output, 'Output from idx_ops is empty')

        min_output_size = self.ctx[MinOutputSizeParameter.name]
        if min_output_size > 0 and os.path.getsize(output_path) < min_output_size:
            eh.check_failed('Output size is too small (should be at least {} bytes)'.format(min_output_size))

        return output

    def _compare_outputs(self, output_directory, queries_path, output1, output2):
        """
            Сравнить выводы от двух последовательных запусков утилиты
        """

        diffs = dolbilka_response_diff.CompareResults(output1, output2)

        if diffs.HasDifferences():
            factors_list = fu.read_lines(os.path.join(output_directory, _FACTOR_NAMES_FILE))
            queries_list = fu.read_lines(queries_path)

            diffs_resource = self.create_resource(
                self.descr,
                'unstable_diff',
                resource_types.EXTENDED_INFO,
                arch='any'
            )

            paths.make_folder(diffs_resource.path)
            dolbilka_response_diff.WriteDiffs(diffs, factors_list, queries_list, diffs_resource.path)
            self.mark_resource_ready(diffs_resource)

            eh.check_failed('idx_ops is not stable, see the unstable_diff dir for diffs')


def create_input_parameters(executable_resource, database_resource, database_required=True):
    """
        Конструирует набор параметров для задачи BaseGetIdxOpsResponses
    """

    class Params(object):
        class ExecutableParameter(sp.LastReleasedResource):
            name = _EXECUTABLE_RESOURCE_ID
            description = 'Executable'
            resource_type = executable_resource

        class SearchDatabaseParameter(sp.LastReleasedResource):
            name = _DATABASE_RESOURCE_ID
            description = 'Database'
            resource_type = database_resource
            required = database_required

        params = (ExecutableParameter, SearchDatabaseParameter)

    return Params


def _patch_queries(in_file_path, out_file_path, queries_limit=0):
    """
        Создаёт файл запросов для idx_ops
        на основе стандартного файла запросов для базового поиска
    """

    with open(in_file_path, 'r') as in_file:
        with open(out_file_path, 'w') as out_file:
            for counter, line in enumerate(in_file):
                if 0 < queries_limit < counter:
                    break
                out_file.write("{}\t{}".format(counter, line))
