# -*- coding: utf-8 -*-

import six
import logging
import json
import time
import urllib

from sandbox.sandboxsdk import errors
from sandbox.projects.common import decorators
from sandbox.projects.common import requests_wrapper
from sandbox.projects.common.nanny import nanny


_CLUSTERSTATE_API = 'http://clusterstate.yandex-team.ru/'

_LIST_CONFIGS_API = _CLUSTERSTATE_API + '?term=C@{config}&page=kernels&json=da'
_LIST_CONFIG_INSTANCES_API = _CLUSTERSTATE_API + 'search?term={term}&page=instances&json=da'

_SEARCH_MAP_API = _CLUSTERSTATE_API + 'search_map/host/{hostname}/port/{port}/'
_SEARCH_MAP_API_DOWN = _SEARCH_MAP_API + 'down'
_SEARCH_MAP_API_UP = _SEARCH_MAP_API + 'up'


def get_upper_instances(hostname, port):
    """
    Seems to be unused in 2018
    :return: instances list: [(host, port), (host, port), ...]
    """
    url = _SEARCH_MAP_API_UP.format(hostname=hostname, port=port)
    return _get_instances_from_search_map(url, 'up')


def get_lower_instances(hostname, port):
    """
    Seems to be unused in 2018
    :return: instances list: [(host, port), (host, port), ...]
    """
    url = _SEARCH_MAP_API_DOWN.format(hostname=hostname, port=port)
    return _get_instances_from_search_map(url, 'down')


def _get_instances_from_search_map(url, direction):
    """
    Helper for `get_upper_instances` and `get_lower_instances`.
    """
    logging.debug('Getting info from search map: direction "%s"', direction)
    json_response = _get_json_response(url)
    instance_list = [(_fix_host_name(i['host']), i['port']) for i in json_response[direction]]
    instance_list.sort(key=lambda x: x[0])
    logging.debug('Instance list: %s', instance_list)
    return instance_list


def get_instances_by_config(config_name, instance_tag_name=None, nanny_oauth_token=None):
    """
    Recommended way to resolve instances

    :param config_name: active cluster configuration name prefix or Nanny service id
    :param instance_tag_name: instance tag name (optional, seems to be deprecated)
    :param nanny_oauth_token: when passed, uses config_name as a Nanny service name
        to get instances list
    :return: instances list: [(host, port), (host, port), ...]
    """
    if nanny_oauth_token:
        return _get_nanny_instances(config_name, nanny_oauth_token)

    active_config = _get_active_config_by_prefix(config_name)
    instance_list = get_instances_by_active_config(active_config, instance_tag_name)
    return instance_list


def _get_nanny_client(nanny_token):
    return nanny.NannyClient(api_url='http://nanny.yandex-team.ru/', oauth_token=nanny_token)


@decorators.retries(max_tries=3, delay=10)
def _get_nanny_instances(service_id, token):
    nanny_client = _get_nanny_client(token)
    return [
        (instance['container_hostname'], instance['port'])
        for instance in nanny_client.get_service_current_instances(service_id)['result']
    ]


def get_svn_tag_by_config(config_name, instance_tag_name=None):
    """
        Example: base/stable-124-2/arcadia
    """
    active_config = _get_active_config_by_prefix(config_name)
    svn_tag = _get_svn_tag_by_active_config(active_config, instance_tag_name)
    return svn_tag


def get_joined_instances_by_config(configuration_prefix, nanny_oauth_token=None):
    """
    Obtain instances list by configuration prefix (quorum majority)
    :param configuration_prefix: configuration name prefix
    :param nanny_oauth_token: if passed - instances will be obtained
    with configuration_prefix as a service name from nanny
    :return list of strings with instances (in "host:port" format)
    """
    return _join_instance_names(get_instances_by_config(configuration_prefix, nanny_oauth_token=nanny_oauth_token))


def _get_active_config_by_prefix(prefix):
    logging.info('Listing configs by config prefix: %s', prefix)
    url = _LIST_CONFIGS_API.format(config=prefix)
    confs = {}
    for i in range(0, 10):
        logging.debug("Read configuration from %s, attempt %s", url, i)
        json_response = _get_json_response(url)
        confs = json_response['confs']
        # when clusterstate flaps, it sometimes return zero configurations instead of
        # normal count, see SEARCH-3499 for reference. So we just ask it twice or more.
        if confs:
            break

        wait_time = 5 + i
        logging.error(
            "No configurations encountered, probably clusterstate is flapping [see SEARCH-3499], "
            "wait for %s seconds...", wait_time
        )
        time.sleep(wait_time)

    if not confs:
        raise errors.SandboxTaskUnknownError("Empty configuration list for prefix {}, cannot continue".format(prefix))
    configurations = {conf: confs[conf]['hosts_active_perc'] for conf in confs}
    active_conf = max(configurations.items(), key=lambda conf: conf[1])[0]
    logging.info('Active config name is: %s', active_conf)
    return active_conf


def _fix_host_name(host_name):
    if 'yandex' not in host_name:
        return host_name + ".search.yandex.net"
    return host_name


def get_instances_by_active_config(active_config=None, instance_tag_name=None, terms=list()):
    """
    :param active_config: active cluster configuration (e.g. 'ONLINE')
    :param instance_tag_name: instance tag name (or list/tuple of names)
    :param terms: aux clusterstate terms list (to be joined via '.')
    :return: instances list: [(host, port), (host, port), ...]
    """
    logging.info(
        "Get instances for config '%s', instance tag '%s' and terms '%s'",
        active_config, instance_tag_name, terms
    )
    url = _make_instance_request_url(active_config, instance_tag_name, terms)
    json_response = _get_json_response(url)
    try:
        instances = [i.split(':') for i in json_response['instances']['cgroups']['']]
        instance_list = [(_fix_host_name(i[0]), i[1]) for i in instances]
    except LookupError as exc:
        logging.error(
            "Bad clusterstate json response (got LookupError: %s)\n%s", exc, json.dumps(json_response, indent=2))
        raise
    logging.debug('Instance list: %s', instance_list)
    return instance_list


def _get_svn_tag_by_active_config(active_config, instance_tag_name=None):
    logging.info('Getting svn tag from config: %s and instance tag: %s', active_config, instance_tag_name)
    url = _make_instance_request_url(active_config, instance_tag_name)
    json_response = _get_json_response(url)
    try:
        svn_tags = json_response['instances']['revisions']
    except LookupError:
        logging.info("Bad json response:\n%s", json.dumps(json_response, indent=2))
        raise
    logging.debug('Svn tags: %s', svn_tags)
    return max(svn_tags.iteritems(), key=lambda x: x[1])[0]


def _make_instance_request_url(active_config=None, instance_tag_name=None, terms=list()):
    """
        :param active_config: cluster configuration name (or name prefix)
        :param instance_tag_name: instance tag name (or list/tuple of names)
        :param terms: aux terms in arbitrary form
        :return: clusterstate request url
    """
    term = []
    if active_config:
        term = ["C@{}".format(active_config)]
    if instance_tag_name:
        if isinstance(instance_tag_name, six.string_types):
            term.append("I@{}".format(instance_tag_name))
        elif isinstance(instance_tag_name, (list, tuple)):
            term.extend(["I@{}".format(i) for i in instance_tag_name])
    if terms:
        term.extend(terms)

    url = _LIST_CONFIG_INSTANCES_API.format(term=urllib.quote_plus(" . ".join(term)))
    return url


@decorators.retries(max_tries=3, delay=180, raise_class=errors.SandboxTaskUnknownError)
def _get_json_response(url):
    """
    Obtain service response in JSON.

    Note about retries: when service is offline or full of HTTP 500, we should wait
    for a while. See also RMINCIDENTS-76.
    :param url: url to query
    :return: parsed json object
    """
    logging.info('Requesting url: %s', url)
    data = requests_wrapper.get(url)
    data.raise_for_status()
    return data.json()


def get_instances_by_shardid(shardid):
    return _skynet_resolve_instances('shard@{}'.format(shardid))


def get_instances_by_itag(itag):
    return _skynet_resolve_instances('itag@{}'.format(itag))


@decorators.retries(max_tries=3, raise_class=errors.SandboxTaskUnknownError)
def _skynet_resolve_instances(arg):
    from library.sky.hostresolver import Resolver

    logging.debug('Command "skynet listinstances %s"', arg)
    skynet_response = Resolver().resolveInstances(arg)
    logging.debug('Skynet response: %s', skynet_response)
    instance_list = _parse_instances_from_skynet(skynet_response)
    logging.debug('Instance list: %s', instance_list)
    return instance_list


def _parse_instances_from_skynet(skynet_response):
    result = []
    for fqdn, instances_on_host in skynet_response.iteritems():
        instance_list = [i[1].split('@')[0] for i in instances_on_host]
        for inst in instance_list:
            hostname, port = inst.split(':')
            result.append((fqdn, int(port)))
    return result


def _join_instance_names(instances):
    return ['{}:{}'.format(host, port) for host, port in instances]
