# -*- coding: utf-8 -*-
"""
    Memcheck (Valgrind) module.
    Task wrapper for for search components
    Author: alexeykruglov@
    Maintainers: mvel@
"""


import xml.dom.minidom
import logging
import os

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import parameters as sp
from sandbox.projects.common.environments import ValgrindEnvironment
from sandbox.projects.common.search import components as sc
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils
from sandbox.projects.common import utils2
from sandbox.projects import resource_types


def finish_log(file_name):
    """
        Sometimes memcheck itself crashes and leaves his log unfinished.
        This leads to XML errors during parse.
    """
    with open(file_name, "r+") as f:
        f.seek(0)
        prev_line = ""
        while True:
            line = f.readline()
            if not line:
                # the end of the file has been reached
                if prev_line.strip() in ["</error>", "</errorcounts>", "</status>"]:
                    # the log is not finished but we can make it up to be parseable
                    logging.info("add </valgrindoutput> to %s", file_name)
                    f.write("</valgrindoutput>\n")
                    return True
                return False
            if '</valgrindoutput>' in line:
                # Valgrind often has garbage after </valgrindoutput> upon crash.
                logging.info("truncate %s", file_name)
                f.truncate()
                return True
            if line.strip():
                prev_line = line


def parse_file(file_name):
    data = fu.read_file(file_name)

    xml_response = xml.dom.minidom.parseString(data)
    error_tags = xml_response.getElementsByTagName('error')

    errors_by_kind = {}

    for e in error_tags:
        kind = _get_text_of(e, 'kind')

        errors_by_kind[kind] = errors_by_kind.get(kind, 0) + 1

    return errors_by_kind


def _get_text_of(node, name):
    node = node.getElementsByTagName(name)[0]
    return "".join([n.data for n in node.childNodes if n.nodeType == n.TEXT_NODE])


_XML_FILE_NAME = 'valgrind.xml'
_TXT_FILE_NAME = 'valgrind.txt'
_VALGRIND_GROUP = "Valgrind parameters"

MEMCHECK_OUTPUT_CTX_KEY = 'memcheck_out_resource_id'
VALGRIND_ERRORS_CTX_KEY = 'valgrind_memcheck_errors'
VALGRIND_ERROR_EXITCODE = 178


class ValgrindSuppressionsParameter(sp.SandboxSvnUrlParameter):
    name = 'valgrind_suppressions'
    description = 'Valgrind suppressions svn url'
    group = _VALGRIND_GROUP
    default_value = ""


class XmlOutputParameter(sp.SandboxBoolParameter):
    name = 'valgrind_xml_output'
    description = 'Valgrind XML output'
    group = _VALGRIND_GROUP
    default_value = True


ERROR_PATTERNS = [
    "depends on uninitialised value",
    "Invalid write of size",
    "Invalid read of size",
]


def get_valgrind_command_params(task):
    valgrind_cmd = [
        'valgrind',
        '--tool=memcheck',
        '--leak-check=full',
        '--track-origins=yes',
        '--freelist-vol=1000000000',
        '--num-callers=100',
        '--time-stamp=yes',
        # '--verbose',  # this produces lots of debug memcheck output
        '--gen-suppressions=all',
        '--error-exitcode={}'.format(VALGRIND_ERROR_EXITCODE),
    ]
    valgrind_cmd += get_cmd_output_params(task.ctx)
    valgrind_cmd = add_suppressions(task, valgrind_cmd)
    return valgrind_cmd


def get_output_file_name(ctx):
    return _XML_FILE_NAME if utils.get_or_default(ctx, XmlOutputParameter) else _TXT_FILE_NAME


def get_cmd_output_params(ctx):
    if utils.get_or_default(ctx, XmlOutputParameter):
        return ['--xml=yes', '--xml-file=' + _XML_FILE_NAME]
    else:
        return ['--log-file=' + _TXT_FILE_NAME]


def create_output(task):
    task.ctx[MEMCHECK_OUTPUT_CTX_KEY] = task.create_resource(
        task.descr,
        get_output_file_name(task.ctx),
        resource_types.VALGRIND_MEMCHECK_OUTPUT,
        arch='any',
    ).id


def add_suppressions(task, params):
    suppressions = utils.get_or_default(task.ctx, ValgrindSuppressionsParameter)
    if suppressions:
        path_to_supp = task.abs_path("memcheck_suppressions")
        Arcadia.export(suppressions, path_to_supp)
        params.append('--suppressions={}'.format(path_to_supp))
    return params


def get_short_task_result(task):
    if not utils.get_or_default(task.ctx, XmlOutputParameter):
        return None
    if not task.is_completed():
        return None

    err_count = sum(task.ctx.get(VALGRIND_ERRORS_CTX_KEY).values())
    return '{0} err'.format(err_count)


def on_memcheck_errors(task, report):
    """
        Мы считаем, что если valgrind находит ошибки в программе,
        их надо либо устранять, либо глушить с помощью suppressions.
        Наблюдать за "графиком роста ошибок" memcheck никто не будет.
        Поэтому мы валим задачу с ошибкой, чтобы меры принимались незамедлительно.
        См. также SEARCH-887 в исторически-познавательных целях.
    """
    if report.status == 'NOT_READY':
        # when task fails, resource will be marked as BROKEN, we don't want it
        task.mark_resource_ready(report.id)
    error_message = 'Memcheck errors encountered, see <a href="{0}">{1}</a>'.format(
        utils2.resource_redirect_url(report.id),
        get_output_file_name(task.ctx),
    )
    task.set_info(error_message, do_escape=False)
    eh.check_failed('Memcheck errors encountered')


def chmod_log(ctx):
    """
    SEARCH-1010: Chmod memcheck output to be world-readable
    """
    errors_file = get_output_file_name(ctx)
    if not os.path.exists(errors_file):
        logging.info("Memcheck errors file %s does not exist", errors_file)
        return
    try:
        os.chmod(errors_file, 0o644)
    except OSError as e:
        logging.info("Cannot chmod memcheck output file: %s", str(e))


def analyze_errors(task):
    report = channel.sandbox.get_resource(task.ctx[MEMCHECK_OUTPUT_CTX_KEY])
    if not os.path.exists(report.path):
        # no file means nothing to check
        logging.warning("Memcheck did not create any output at '%s'", report.path)
        return

    # to make it finally readable
    chmod_log(task.ctx)

    if utils.get_or_default(task.ctx, XmlOutputParameter):
        finish_log(report.path)
        errors_by_kind = parse_file(report.path)
        task.ctx[VALGRIND_ERRORS_CTX_KEY] = errors_by_kind
        if errors_by_kind:
            on_memcheck_errors(task, report)
    else:
        # detect errors in text (will work for default locale only, but who cares)
        report_text = fu.read_file(report.path)
        eh.verify('definitely lost: ' in report_text, "Invalid memcheck output")
        if 'definitely lost: 0' not in report_text:
            task.ctx[VALGRIND_ERRORS_CTX_KEY] = 1
            eh.check_failed("Some memory leaked")

        for pattern in ERROR_PATTERNS:
            if pattern in report_text:
                logging.info("Detected memcheck error pattern '%s'", pattern)
                on_memcheck_errors(task, report)


def generate_task(
    params_collection,
    base_class,
    start_timeout=sc.DEFAULT_START_TIMEOUT * 4,
    shutdown_timeout=sc.DEFAULT_START_TIMEOUT * 4,
    valgrind_version=None,
):
    """
        Generates task class based on parent task class
        to run search component with valgrind memcheck tool
        base class should have stub method init_search_component(self, component)
    """

    class MemcheckTaskPattern(base_class):
        type = None

        input_parameters = sc.tune_search_params(
            params_collection, base_class.input_parameters,
            start_timeout=start_timeout,
        ) + [
            ValgrindSuppressionsParameter,
            XmlOutputParameter,
        ]

        @property
        def footer(self):
            base_foot = [{
                'helperName': '',
                'content': "<h4>Errors: <span style='color:red'>{}</span></h4>".format(
                    self.ctx.get('valgrind_memcheck_errors')
                )
            }]
            try:
                # catch failed base footers
                add_foot = base_class.footer.fget(self)
                if isinstance(add_foot, dict):
                    if 'content' not in add_foot:
                        base_foot.append({
                            'helperName': '',
                            'content': add_foot
                        })
                    else:
                        base_foot.append(add_foot)
                elif isinstance(add_foot, (list, tuple)):
                    base_foot.extend(add_foot)
                else:
                    base_foot.append({
                        'helperName': '',
                        'content': "<h4>Something wrong with base class footer</h4>"
                    })
            except Exception as e:
                base_foot.append({
                    'helperName': '',
                    'content': "<h4>Exception in base footer:</h4>\n{}".format(e)
                })
            return base_foot

        def on_enqueue(self):
            base_class.on_enqueue(self)
            create_output(self)

        def on_execute(self):
            try:
                base_class.on_execute(self)
            except errors.SandboxSubprocessError as e:
                if e.returncode == VALGRIND_ERROR_EXITCODE:
                    resource = channel.sandbox.get_resource(self.ctx[MEMCHECK_OUTPUT_CTX_KEY])
                    on_memcheck_errors(self, resource)
                else:
                    raise
            analyze_errors(self)

        def _get_middlesearch_additional_params(self):
            """
                Fight with problems like in SEARCH-3159: set longer shutdown timeout
            """
            return {
                "shutdown_timeout": shutdown_timeout,
            }

        def init_search_component(self, component):
            base_class.init_search_component(self, component)

            component.replace_config_parameter("Collection/UserParams/MemoryMappedArchive", None)
            component.replace_config_parameter("Collection/UserParams/ArchiveOpenMode", None)
            component.replace_config_parameter("Collection/UserParams/FileArchive-NoReuse", "")
            component.replace_config_parameter("Collection/CalculateBinaryMD5", 'no')
            component.replace_config_parameter("Collection/AbortTout", '4800000000')  # hotfix for SEARCH-1134

            def cmd_patcher(cmd):
                if valgrind_version:
                    ValgrindEnvironment(version=valgrind_version).prepare()
                else:
                    ValgrindEnvironment().prepare()
                return get_valgrind_command_params(self) + cmd

            component.run_cmd_patcher = cmd_patcher

            def after_start():
                chmod_log(self.ctx)
            component.after_start = after_start

        def get_results(self):
            if not utils.get_or_default(self.ctx, XmlOutputParameter):
                return None
            if not self.is_completed():
                return 'Results are not ready yet.'

            return 'Errors: {0}'.format(self.ctx.get(VALGRIND_ERRORS_CTX_KEY))

        def get_short_task_result(self):
            return get_short_task_result(self)

    return MemcheckTaskPattern
