import logging
import os
import urllib
from six.moves import range

import sandbox.common.types.client as ctc

from sandbox.sandboxsdk import sandboxapi
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import process

from sandbox.projects.common import dolbilka
from sandbox.projects.tank import executor as tank_executor
from sandbox.projects.tank.load_resources import resources as tank_resources

_MEMORY_RSS_KEY = 'memory_rss'
_MEMORY_VSZ_KEY = 'memory_vsz'


class BaseShootingTask:
    """
        Mixin class with common routines for shooting via tank
    """
    client_tags = ctc.Tag.GENERIC & ctc.Tag.LINUX_PRECISE

    shoot_input_parameters = (
        tank_executor.DolbiloPlugin.create_params() +
        tank_executor.TankExecutor.create_params() +
        tank_executor.LunaparkPlugin.create_params(enabled=False) +
        tank_executor.OfflinePlugin.create_params()
    )

    def _init_virtualenv(self, tank_resource_type=tank_resources.YANDEX_TANK_VIRTUALENV_19):
        tank_executor.TankExecutor.init_virtualenv(self, tank_resource_type=tank_resource_type)

    def _default_plugins(self):
        return [
            tank_executor.JsonReportPlugin(self),
            tank_executor.LunaparkPlugin(self),
            tank_executor.OfflinePlugin(self),
            tank_executor.TelegrafPlugin(self),
        ]

    def _shoot(self, executor, variant, autostop_expected=False):
        description = "{}, basesearch {}".format(self.descr, variant)
        work_dir = self.abs_path("fire.{}".format(urllib.quote(variant, safe="")))

        paths.make_folder(work_dir, delete_content=True)
        stats_resource = self.create_resource(
            description, work_dir, tank_resources.YANDEX_TANK_LOGS, arch=sandboxapi.ARCH_ANY)

        # Note: cgroup is too restrictive to obtain maximum possible rps
        artifacts_dir = executor.fire(
            work_dir,
            job_name=description,
            cgroup=None,
            autostop_expected=autostop_expected,
        )
        return stats_resource, artifacts_dir

    @staticmethod
    def _format_report(num, results, add_hash=True, strong=False):
        if "report_url" in results:
            report_url = results["report_url"]
        elif "report_resource" in results:
            report_url = "//proxy.sandbox.yandex-team.ru/{}/{}".format(results["report_resource"], results["report_path"])
        else:  # No additional formatting
            report_url = None

        num = "#{}".format(num) if add_hash else num
        report = "<a href='{}'>{}</a>".format(report_url, num) if report_url else num
        return "<strong>{}</strong>".format(report) if strong else report


class OldShootingTask(BaseShootingTask):
    """
        Old style shooting of search components
    """

    shoot_input_parameters = (dolbilka.DolbilkaSessionsCount,) + BaseShootingTask.shoot_input_parameters

    stats_key = "results"

    # stats from dolbilka.DolbilkaPlanner.fill_rps_ctx
    stats_types = (
        ("requests_per_sec", "RPS", "{:0.2f}"),
        ("fail_rates", "Fail rate", "{:0.2f}"),
        ("memory_rss", "RSS memory", "{}"),
        ("memory_vsz", "VSZ memory", "{}"),
    )

    # stats from tank_executor.DolbiloPlugin
    new_stats_types = (
        ("latency_0.5", "Latency, q50 (usec)", "{:0.2f}"),
        ("latency_0.95", "Latency, q95 (usec)", "{:0.2f}"),
        ("resp_size_quantile_0.5", "Response size median (bytes)", "{:0.1f}"),
        ("max_size", "Max response size (bytes)", "{:0.1f}"),
    )

    # use old style dumper stats for rps calculation
    old_style_stats = False

    def _format_stat_value(self, key, fmt, value):
        if value is None:
            return "-"
        if key == "requests_per_sec" and value == self.ctx["max_rps"]:
            fmt = "<b style='color:red'>{}</b>".format(fmt)
        return fmt.format(value)

    @property
    def footer(self):
        for key, title, fmt in self.stats_types:
            if key not in self.ctx:
                return {"Calculating...": ""}

        header = [{"key": "num", "title": "Session"}] + [
            {"key": key, "title": title} for key, title, _ in (self.stats_types + self.new_stats_types)
        ]

        body = {
            "num": [self._format_report(num, results) for num, results in enumerate(self.ctx[self.stats_key])]
        }
        for key, title, fmt in self.stats_types:
            body[key] = [self._format_stat_value(key, fmt, value) for value in self.ctx[key]]

        for key, title, fmt in self.new_stats_types:
            body[key] = [self._format_stat_value(key, fmt, result.get(key)) for result in self.ctx.get(self.stats_key)]

        return {
            "<h3>Performance stats</h3>": {
                "header": header,
                "body": body,
            }
        }

    def _old_shoot(self, target, plan_resource_id):
        results = []
        for session in range(self.ctx[dolbilka.DolbilkaSessionsCount.name]):
            stats_resource, artifacts_dir = self._shoot(
                tank_executor.TankExecutor(
                    tank_executor.DolbiloPlugin(self, plan_resource_id, "localhost", target.port),
                    *self._default_plugins()
                ),
                "session {}".format(session)
            )
            self._save_memory_usage(target)

            stats_results = tank_executor.DolbiloPlugin.get_dumper_stats(artifacts_dir)
            stats_results.update({
                "report_resource": stats_resource.id,
                "report_path": os.path.relpath(tank_executor.TankExecutor.get_report_path(artifacts_dir), stats_resource.path),
            })
            lunapark_url = tank_executor.LunaparkPlugin.get_job_url(self)
            if lunapark_url:
                stats_results["report_url"] = lunapark_url
            results.append(stats_results)
        self._fill_rps_ctx(results)

    def _fill_rps_ctx(self, results):
        # TODO: use 'variant.field' style for new fields from dumper stats

        def _get_rate(result, field_name):
            try:
                rate = 0
                count = result.get(field_name, None)
                if count:
                    # reasonable request count order is k*10^5, so round up to 6 digits is OK
                    rate = round(float(count) / int(result['total_requests']), 6)
            except (ValueError, ZeroDivisionError, KeyError):
                rate = 2  # 200% of errorness in output
                logging.error(
                    ("rate calculation failed: {%s}, and {requests} requests"
                     % field_name).format(**result)
                )
            return rate

        self.ctx['results'] = results
        self.ctx['requests_per_sec'] = []
        self.ctx['fail_rates'] = []
        self.ctx['notfound_rates'] = []
        for result in results:
            try:
                rps = float(result['old_dumper.rps' if self.old_style_stats else 'dumper.rps'])
            except (ValueError, KeyError):
                rps = 0.0
            fail_rate = _get_rate(result, "5xx_requests")
            notfound_rate = _get_rate(result, "404_requests")
            self.ctx['requests_per_sec'].append(rps)
            self.ctx['fail_rates'].append(fail_rate)
            self.ctx['notfound_rates'].append(notfound_rate)
        self.ctx['max_rps'] = max(self.ctx['requests_per_sec'])
        self.ctx['max_fail_rate'] = max(self.ctx['fail_rates'])
        self.ctx['min_fail_rate'] = min(self.ctx['fail_rates'])
        self.ctx['min_notfound_rate'] = min(self.ctx['notfound_rates'])

    def _save_memory_usage(self, target):
        info = process.get_process_info(target.process.pid, ['rss', 'vsz'], ignore_errors=False)

        self.ctx.setdefault(_MEMORY_RSS_KEY, []).append(int(info['rss']))
        self.ctx.setdefault(_MEMORY_VSZ_KEY, []).append(int(info['vsz']))


class NewShootingTask(BaseShootingTask):
    """
        New style shooting of search components
    """

    new_stats_key = "new_stats"
    bold_stats_key = "bold_stat_name"

    new_stats_types = (
        ("shooting.rps_0.5", "RPS P50", "{:0.2f}"),
        ("shooting.rps_avg", "RPS average", "{:0.2f}"),
        ("shooting.rps_stddev", "RPS stddev", "{:0.2f}"),
        ("shooting.latency_0.5", "Latency P50", "{:0.2f}"),
        ("shooting.latency_0.99", "Latency P99", "{:0.2f}"),
        ("shooting.errors", "Errors", "{}"),
        ("monitoring.cpu_user", "CPU P50", "{:0.2f}"),
    )

    dolbilo_warmup = 70  # seconds
    dolbilo_shutdown = 3  # seconds

    shoot_input_parameters = BaseShootingTask.shoot_input_parameters

    @property
    def footer(self):
        if self.new_stats_key not in self.ctx:
            return {"Calculating...": ""}

        stats = self.ctx[self.new_stats_key]
        bold_stat = self.ctx.get(self.bold_stats_key)
        variants = sorted(stats.keys())

        header = [
            {"key": "num", "title": "&nbsp;"}
        ] + [
            {"key": key, "title": title} for key, title, _ in self.new_stats_types
        ]

        body = {"num": []}
        for variant in variants:
            is_bold = bold_stat == variant
            body["num"].append(self._format_report(variant, stats[variant], add_hash=False, strong=is_bold))
            for key, title, fmt in self.new_stats_types:
                body.setdefault(key, []).append(self._format_stats(fmt, stats, variant, key))

        return {
            "<h3>Performance stats</h3>": {
                "header": header,
                "body": body,
            }
        }

    def _in_stats(self, key):
        for stat_key, stat_title, stat_fmt in self.new_stats_types:
            if key == stat_key:
                return True
        return False

    def _format_stats(self, fmt, stats, variant, key):
        if key not in stats[variant]:
            return "-"
        return fmt.format(stats[variant][key])

    def _dolbilo_shoot(self, target, plan_resource_id, variant, augment_url=None):
        with target:
            stats_resource, artifacts_dir = self._shoot(
                tank_executor.TankExecutor(
                    tank_executor.DolbiloPlugin(self, plan_resource_id, "localhost", target.port, augment_url=augment_url),
                    *self._default_plugins()
                ),
                variant
            )

        # new stats
        new_stats_results = tank_executor.DolbiloPlugin.get_stats(artifacts_dir)
        new_stats_results.update({
            "report_resource": stats_resource.id,
            "report_path": os.path.relpath(tank_executor.TankExecutor.get_report_path(artifacts_dir), stats_resource.path),
        })

        lunapark_url = tank_executor.LunaparkPlugin.get_job_url(self)
        if lunapark_url:
            new_stats_results["report_url"] = lunapark_url

        self.ctx.setdefault(self.new_stats_key, {})[variant] = new_stats_results


def delta_percent(a, b):
    return (float(b - a) / a) * 100 if a else 0
