# coding: utf-8

import logging

from sandbox import sdk2
import sandbox.common.types.resource as ctr

from sandbox.projects.common import apihelpers
from sandbox.sandboxsdk import sandboxapi
from sandbox.sandboxsdk import channel

from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils
from sandbox.projects.common.search import instance_resolver

from sandbox.projects.common.nanny.nanny import NannyClient
from sandbox.projects.release_machine import security as rm_sec
from sandbox.projects.release_machine.core import const as rm_const


def get_prod_binary(binary_res_type, clusterstate_config, arch=sandboxapi.ARCH_LINUX):
    """
        Get binary, which now really in prod
        :param binary_res_type: resource type of prod binary
        :param clusterstate_config: Ex.: "jupiter_mmeta-", "jupiter_base-"
        NOTE: Please do not hardcode cluster names in code, use common.search.cluster_names module.
        :param arch: binary arch, linux by default
        :return: SandboxResource or None
    """
    logging.info("Get binary, which equals current production")
    try:
        prod_svn_tag = instance_resolver.get_svn_tag_by_config(clusterstate_config)
        logging.info("Current prod svn tag: %s", prod_svn_tag)
        # Sometimes, the testing robot releases an already-released resource to testing
        # after it was released to production, in this case the final status will be "testing".
        # So, look for both released statuses; if the target resource is found as "testing",
        # then either the testing robot was too late (more likely),
        # or somebody has deployed released:testing resource to production (and in production we trust).
        released_resources = []
        for status in (sandboxapi.RELEASE_STABLE, sandboxapi.RELEASE_TESTING):
            released_resources.extend(
                release.resources for release in channel.channel.sandbox.list_releases(
                    resource_type=binary_res_type,
                    arch=arch,
                    limit=200,  # hopefully, prod tag will be in last 200 stable releases
                    release_status=status,
                )
            )
        for res in utils.flatten(released_resources):
            if utils.get_binary_version(res) in prod_svn_tag:
                logging.info("Got binary: %s", res.id)
                return res
    except Exception as err:
        eh.log_exception("Unable to get binary for production svn tag", err)

    # SEARCH-2913
    channel.channel.task.set_info(
        "Unable to get {} production binary using clusterstate! Try to get last released resource".format(
            binary_res_type
        )
    )
    res = apihelpers.get_last_released_resource(
        resource_type=binary_res_type,
        arch=sandboxapi.ARCH_ANY if binary_res_type.any_arch else sandboxapi.ARCH_LINUX,
        release_status=sandboxapi.RELEASE_STABLE,
    )
    logging.info("Got last stable binary: %s", res.id)
    return res


def get_prod_binary_sdk2(binary_res_type, clusterstate_config, arch=sandboxapi.ARCH_LINUX):
    """
        Get binary that is now really in production (SDK2-ready)
        :param binary_res_type: resource type of prod binary
        :param clusterstate_config: Ex.: "jupiter_mmeta-", "jupiter_base-"
        NOTE: Please do not hardcode cluster names in code, use constants
        from common.search.cluster_names module.
        :param arch: binary arch, linux by default
        :return: resource or None
    """
    logging.info("Get binary, which equals current production")
    prod_svn_tag = instance_resolver.get_svn_tag_by_config(clusterstate_config)
    logging.info("Current prod svn tag: %s", prod_svn_tag)
    # see the comment in get_prod_binary
    for status in ('stable', 'testing'):
        for res in sdk2.Resource.find(
            resource_type=binary_res_type,
            state=ctr.State.READY,
            attr_name='released',
            attr_value=status,
            arch=arch,
        ).limit(15):
            if utils.get_binary_version(res) in prod_svn_tag:
                logging.info("Got binary: %s", res.id)
                return res


def get_prod_binary_nanny(binary_res_type, nanny_service_id, arch=sandboxapi.ARCH_LINUX, just_return_binary=False):
    try:
        NANNY_TOKEN = rm_sec.get_rm_token(None, 'nanny_oauth_token')
        nanny_client = NannyClient(api_url=rm_const.Urls.NANNY_BASE_URL, oauth_token=NANNY_TOKEN)
        resources = nanny_client.get_service_resources(nanny_service_id)
        binary = None
        for file in resources['content']['sandbox_files']:
            if file['resource_type'] == binary_res_type:
                binary = channel.channel.sandbox.get_resource(file['resource_id'])
                if just_return_binary:
                    return binary
        if binary:
            prod_svn_tag = utils.get_binary_version(binary)
            stable_released_resources = [
                release.resources for release in channel.channel.sandbox.list_releases(
                    resource_type=binary_res_type,
                    arch=arch,
                    limit=10,
                    release_status=sandboxapi.RELEASE_STABLE,
                )
                ]
            for res in utils.flatten(stable_released_resources):
                if utils.get_binary_version(res) in prod_svn_tag:
                    return res
    except Exception as err:
        eh.log_exception("Unable to get binary for production svn tag", err)

    channel.channel.task.set_info(
        "Unable to get {} production binary using nanny! Try to get last released resource".format(
            binary_res_type
        )
    )
    res = apihelpers.get_last_released_resource(
        resource_type=binary_res_type,
        arch=sandboxapi.ARCH_ANY if binary_res_type.any_arch else sandboxapi.ARCH_LINUX,
        release_status=sandboxapi.RELEASE_STABLE,
    )
    logging.info("Got last stable binary: %s", res.id)
    return res
