# -*- coding: utf-8 -*-
#
# calculate rearrange timings, using RearrangeInfo events from eventlog


class TimingsInfo:
    def __init__(self):
        self.arr = []
        self.sum_time = 0
        self.count = 0
        self.curr_frame = None
        self.curr_time = 0

    def __repr__(self):
        return str(float(self.sum_time)/self.count)


class EventInfo:
    def __init__(self, tm, frame, stage, rearrange, g_size):
        self.tm = tm
        self.frame = frame
        self.stage = stage  # 0,1,2,3
        self.rearrange = rearrange
        self.g_size = g_size

    def is_begin_rearrange(self):
        return self.stage % 2 == 0


def get_timings(rearrange_timings, rearrange_name):
    if rearrange_name not in rearrange_timings:
        rearrange_timings[rearrange_name] = TimingsInfo()
    return rearrange_timings[rearrange_name]


def add_timing(rearrange_timings, ev0, ev1, rearrange=None):
    timings = get_timings(rearrange_timings, rearrange if rearrange is not None else ev1.rearrange)
    if ev1.tm > ev0.tm:
        tm_diff = ev1.tm - ev0.tm
        timings.sum_time += tm_diff
        timings.curr_time += tm_diff
        if timings.curr_frame is None or timings.curr_frame != ev0.frame:
            timings.count += 1
            timings.curr_frame = ev0.frame
            timings.arr.append(timings.curr_time)
            timings.curr_time = 0


def _calc_timings(f, rearrange_timings, parent_event=None, prev_event=None):
    prev_event = prev_event
    for line in f:
        tk = line.rstrip().split('\t')
        tm = int(tk[0])
        frame = int(tk[1])
        ev = tk[2]
        if prev_event is not None and prev_event.frame != frame:
            prev_event = None
        if ev == 'RearrangeInfo':
            stage = int(tk[3])
            rearrange = tk[4]
            g_size = int(tk[5])
            event = EventInfo(tm, frame, stage, rearrange, g_size)
            if prev_event is not None and prev_event.is_begin_rearrange():
                if event.is_begin_rearrange():
                    add_timing(rearrange_timings, prev_event, event, rearrange=prev_event.rearrange)
                    # detect recursive merge, use recursive _calc_timings
                    event = _calc_timings(f, rearrange_timings, parent_event=prev_event, prev_event=event)
                else:
                    # half common case
                    if event.rearrange != prev_event.rearrange:
                        raise Exception('Invalid events sequence [1]')
                    add_timing(rearrange_timings, prev_event, event)
            else:
                if event.is_begin_rearrange():
                    # TODO:think about case with double run recursive merge
                    prev_event = event
                else:
                    if prev_event is None:
                        raise Exception('Invalid events sequence [2]')
                    # exit from recursion
                    add_timing(rearrange_timings, prev_event, event, rearrange=parent_event.rearrange)
                    return event
            prev_event = event
    return None


def calc_timings_from_evlog(filename='eventlog.txt'):
    rearrange_timings = {}  # rearrange_name -> TimingsInfo(arr, sum_time, count)
    with open(filename) as f:
        if _calc_timings(f, rearrange_timings) is not None:
            raise Exception('Invalid events sequence [0]')
    return rearrange_timings
