# -*- coding: utf-8 -*-
#
# base/common http requester
#
import Queue
import gzip
import httplib
import logging
import re
import struct
import time
import traceback
import urllib
from six.moves import range
from multiprocessing import Process
from multiprocessing import Queue as MtpQueue

from sandbox.sandboxsdk import parameters as sp

from sandbox.projects.common.search import requester_core

_marker_begin = 'window.location.replace("'
_marker_end = '");'
_requester_group = "Requester parameters"


def get_redirect_url(resp):
    """
        dirty hack for redirect via javascript (HAMSTERINCIDENTS-160 & etc)
    """
    for i in range(7):
        tk = resp.split('\n', 1)
        if len(tk) != 2:
            return None
        l, resp = tk
        if l.startswith(_marker_begin) and l.endswith(_marker_end):
            escaped_url = l[len(_marker_begin):-len(_marker_end)]
            it = iter(escaped_url)
            url = ''
            for c in it:
                if c == '\\':
                    c = it.next()
                    if c == 'x':
                        shex = it.next()
                        shex += it.next()
                        url += chr(int(shex, 16))
                        continue
                url += c
            return url
        if not resp:
            break


class Worker(object):
    """
        Класс реализует процесс отправки/получения запросов
        Наследников можно использовать для реализации своих способов отправки запроса или
        обработки результа запроса в потоке-worker-е, что-бы возвращать в основной процесс
        уже обработанный результат
    """

    def __init__(self, worker_id, qreqs, qresps):
        self.worker_id = worker_id
        self.qreqs = qreqs
        self.qresps = qresps

        self.max_redirect = 5
        self.re_try_limit = 0

    def run(self):
        while True:
            try:
                req_job = self.qreqs.get()
                if req_job is None:
                    break
                nreq = req_job['nreq']
                req = req_job['req']
                orig_req = req
                re_try_limit = req_job.get('re_try_limit', 0)
                request_timeout = req_job.get('request_timeout', 60)
            except Queue.Empty:
                break

            u = None
            try:
                while True:  # retry loop
                    try:
                        req = orig_req
                        resp = None
                        for r in range(self.max_redirect):  # redirect loop
                            u = self._on_urlopen(req, request_timeout)
                            try:
                                resp = u.content
                                break
                            except httplib.IncompleteRead as e:
                                if req_job.get('ignore_incomplete_read', False):
                                    logging.debug('incomplete_read: %s', e.partial)
                                    resp = str(e.partial)
                                else:
                                    raise

                            rurl = get_redirect_url(resp)
                            if rurl:
                                req = rurl
                            else:
                                break
                        self._on_response(nreq, resp)
                        break
                    except Exception as e:
                        req_info = req if req == orig_req else '{} orig_req({})'.format(req, orig_req)
                        logging.debug(
                            'Worker exception "%s" on request #%s: %s re_try_limit=%s\n%s',
                            str(e), nreq, req_info, re_try_limit, traceback.format_exc())
                        if re_try_limit == 0:
                            raise
                        re_try_limit -= 1
                        time.sleep(1)
            except Exception as e:
                self.qresps.put((nreq, False, str(e)))
                break
            finally:
                if u:
                    u.close()

    @staticmethod
    def _on_urlopen(req, timeout=60):
        return requester_core.url_open_requests(req, timeout)

    def _on_response(self, nreq, resp):
        """
            Можно переопределить для собственной обработки запроса,
            - если здесь бросить exception, будет сделана очередная попытка
            выполнить данный запрос (пока re_try_limit != 0)
        """
        # logging.debug('worker_end_req %s:\n%s', nreq, resp)
        self.qresps.put((nreq, True, resp))


def default_requester_worker(qreqs, qresps, worker_id):
    """
        Метод для выполнения в отдельных процессах
        (собственно здесь и производится отправка/получение запроса)
    """
    Worker(worker_id, qreqs, qresps).run()


class Params:
    class RequestsLimit(sp.SandboxIntegerParameter):
        name = 'requests_limit'
        description = 'Limit number of used requests (0 = all)'
        group = _requester_group
        default_value = 0

    class WorkersCount(sp.SandboxIntegerParameter):
        name = 'workers_count'
        description = 'Use workers (processes) for sending requests'
        group = _requester_group
        default_value = 20

    class RequestTimeout(sp.SandboxIntegerParameter):
        name = 'request_timeout'
        description = 'Request timeout (valgrind crutch)'
        group = _requester_group
        default_value = 60

    class SleepAfterStart(sp.SandboxIntegerParameter):
        name = 'sleep_after_start'
        description = 'Sleep after start (valgrind crutch)'
        group = _requester_group
        default_value = 0

    lst = (
        RequestsLimit,
        WorkersCount,
        SleepAfterStart,
        RequestTimeout,
    )


class Requester(object):
    """
        Класс для организации опроса (через http/https как правило) поисковых сервисов (например, компоненты).
        Опрос производится в несколько потоков (точнее, процессов), через передачу им задач на опрос, - результат
        возвращается в основной поток (если используется 'тяжёлая' обработка результата, рекомендуется
        делать её в наследнике от Worker, переопределив _on_response - тогда нагрузка на CPU будет рапроделена
        по процессам-worker-ам)
        Можно использовать requester для опроса локальной поисковой компоненты через use()
        или опрашивать сторонний сервис через init() + send_requests()
    """
    class WorkerError(Exception):
        def __init__(self, err, nreq):
            Exception.__init__(self, err)
            self.nreq = nreq

    def __init__(self, requester_worker=default_requester_worker):
        self.search_component = None
        self.workers = None
        self.in_fly = None
        self.requests_iterator = None
        self.process_count = None
        self.requests_limit = None
        self.request_timeout = None
        self.error = None
        self.qreqs = None
        self.qresps = None
        self.responses_counter = 0
        self.ignore_incomplete_read = False
        self.re_try_limit = 0
        self.requester_worker = requester_worker

    def use(self, requests_iterator, search_component, process_count=5, requests_limit=0, ctx=None, start_stop=True):
        """
            Отправляем получаемые через request_iterator запросы в указанный компонент
            (для этого запускаем/останаваливаем компонент)
        """
        def ctx_or_val(key, val):
            if not ctx:
                return val
            return ctx.get(key, val)

        self.init(
            requests_iterator,
            process_count=ctx_or_val(Params.WorkersCount.name, process_count),
            requests_limit=ctx_or_val(Params.RequestsLimit.name, requests_limit),
            request_timeout=ctx_or_val(Params.RequestTimeout.name, Params.RequestTimeout.default_value),
        )
        if start_stop:
            search_component.start()
            search_component.wait()

        # give more time for start if use valgrind
        sleep_after_start = ctx_or_val(Params.SleepAfterStart.name, Params.SleepAfterStart.default_value)
        if sleep_after_start:
            time.sleep(sleep_after_start)

        self.use_component(search_component)

        if start_stop:
            search_component.stop()

    def init(self, requests_iterator, process_count=5, requests_limit=0, request_timeout=60):
        self.requests_iterator = requests_iterator
        self.process_count = process_count
        self.requests_limit = requests_limit
        self.request_timeout = request_timeout
        self.error = None
        self.responses_counter = 0

    def _requests(self):
        nreq = 0
        for req in self.requests_iterator:
            yield self.build_request(req)
            nreq += 1
            if self.requests_limit and nreq == self.requests_limit:
                break

    def build_request(self, req):
        return requester_core.build_request("http://localhost:{}".format(self.search_component.get_port()), req)

    def use_component(self, search_component):
        self.search_component = search_component
        try:
            search_component.use_component(self.send_requests)
        finally:
            self.search_component = None

    def send_requests(self):
        """
            Создаём пул процессов для отправки запросов/обработки ответов
            (данные для запросов уже должны быть в наличии в виде итератора self.requests_iterator)
        """
        # общаемся с пулом worker-ов через очереди
        # отправляем "пакет"
        # {
        #   'nreq': номер_запроса,
        #   'req': запрос
        #   'ignore_incomplete_read': True/False
        # }
        self.qreqs = MtpQueue(1024)
        self.qresps = MtpQueue(1024)  # получаем (номер_запроса, флаг_успешности, результат_ошибка_или_ответ)
        self.workers = []

        for wid in range(0, self.process_count):
            proc = Process(target=self.requester_worker, args=(self.qreqs, self.qresps, wid))
            self.workers.append(proc)
            proc.start()
        nreq = 0
        self.in_fly = {}
        try:
            for req in self._requests():
                while self.workers:
                    try:
                        self.qreqs.put_nowait(dict(
                            nreq=nreq,
                            req=req,
                            ignore_incomplete_read=self.ignore_incomplete_read,
                            re_try_limit=self.re_try_limit,
                            request_timeout=self.request_timeout,
                        ))
                        if nreq % 100 == 0:
                            logging.debug('Enqueued %s requests...', nreq)
                        self.in_fly[nreq] = req
                        nreq += 1
                        break
                    except Queue.Full:
                        self.handle_workers_result()
            logging.debug('Total enqueued requests amount: %s', nreq)
            self.finish_workers()
        except Requester.WorkerError as we:
            logging.error('Worker error at req[%s]: %s', we.nreq, we)
            if not self.error:
                self.error = str(we)
            self.finish_workers(True)
        self.qreqs = None
        self.qresps = None

    def handle_workers_result(self, timeout=3):
        """
            получаем от worker-ов результаты работы
            (если результатов нет, то залипаем на несколько сек. и возвращаем управление)
        """
        try:
            while True:
                nresp, ok, result = self.qresps.get(True, timeout)
                if not ok:
                    self.on_fail_request(nresp, result)
                    return
                self.on_response(nresp, result)
        except Queue.Empty:
            self.refresh_workers()

    def on_fail_request(self, nreq, err):
        """
            Обработчик окончательной ошибки при попытке выполнения запроса
            Логируем здесь проблемный запрос/проблему, вызываем hook выброса исключения
            для завершения процесса опроса
        """
        if self.error:
            return
        self.error = 'error: {} on request: {}'.format(err, self.in_fly[nreq])
        logging.info(self.error)
        del self.in_fly[nreq]
        self.raise_on_fail_request(nreq, err)

    def raise_on_fail_request(self, nreq, err):
        """
            Hook выброса исключения для завершения процесса опроса
            (можно перегрузить для предотвращения прерывания процесса опроса на определённых ошибках)
        """
        raise Requester.WorkerError(err, nreq)

    def on_response(self, nreq, result):
        """
            Метод нужно перегрузить, если нужно обрабатывать содержимое ответа!
            Override for do something with response
        """
        # logging.debug('result[%s]: %s', nreq, result)
        self.responses_counter += 1
        del self.in_fly[nreq]

    def refresh_workers(self):
        """
            Обновляем список текущих работающих worker-ов
        """
        self.workers = [p for p in self.workers if p.is_alive()]

    def finish_workers(self, force_finish=False):
        """
            Мягко останавливаем worker-ы
                - посылаем им маркер конца очереди с job-ами
                - дожидаемся завершения всех worker-ов
        """
        logging.debug('Workers finalization started')
        self.refresh_workers()
        while self.workers:
            self.handle_workers_result()
            for _ in self.workers:
                try:
                    if force_finish:
                        _.terminate()
                        logging.debug('terminate signal sended to worker')
                    else:
                        self.qreqs.put_nowait(None)
                        logging.debug('Finish marker sended to worker')
                except Queue.Full:
                    logging.debug('Queue overfull')
                    break
            for proc in self.workers:
                proc.join(1)
            self.refresh_workers()
            logging.debug('Workers finalization continues...')
        self.handle_workers_result()  # довычитываем данные из очереди ответов (ежели там ещё что-то есть)


def format_users_queries_line(line):
    """
        Форматируем строку из ресурса USERS_QUERIES в cgi-параметры запроса на report: &<request-cgi-params>
        (если регион для запроса не указан, используем lr=213)
    """
    line = line.rstrip('\n')
    tk = line.split('\t')
    text = urllib.quote(tk[0], '')
    lr = tk[1] if len(tk) >= 2 and tk[1] else "213"
    q = "&text={}&lr={}".format(text, lr)
    if len(tk) >= 3 and tk[2]:
        if tk[2] != '&':
            q += '&'
        q += tk[2]
    return q


def sequence_binary_data_from_stream(f):
    """
        Генератор последовательности двоичных данных (запросы/ответы),
        считанных из потока (файловый объект)
    """
    while True:
        head = f.read(4)
        if not head:
            break
        if len(head) != 4:
            raise Exception("Broken file content")
        size = struct.unpack("i", head)[0]
        body = f.read(size)
        if len(body) != size:
            raise Exception("Broken file content")
        yield body


def sequence_binary_data(file_name):
    """
        Генератор последовательности двоичных данных (запросы/ответы), считанных из файла
    """
    with open(file_name) as f:
        for data in sequence_binary_data_from_stream(f):
            yield data


def sequence_gzipped_binary_data(file_name):
    """
        Генератор последовательности двоичных данных (запросы/ответы),
        считанных из пожатого gzip-ом файла
    """
    with gzip.open(file_name) as f:
        for data in sequence_binary_data_from_stream(f):
            yield data


def write_binary_data(f, req):
    """
        Сохраняем запрос в файл в формате ui32_sizeof(data) + data
    """
    f.write(struct.pack('i', len(req)))
    f.write(req)


class ResponseSaver(Requester):
    @staticmethod
    def create_params():
        return Params

    def __init__(self, ctx, search_component, custom_sep=None):
        super(ResponseSaver, self).__init__()
        self.res_file = None
        self.ctx = ctx
        self.search_component = search_component
        # buffer for request reordering due to multiprocessing
        self.wait_nreq = 0
        self.buffered_results = {}
        self.custom_sep = str(custom_sep) if custom_sep is not None else "\n" + "-" * 100 + "{}\n"

    def run(self, req_iter, res_file):
        self.res_file = res_file
        self.use(req_iter, self.search_component, ctx=self.ctx)

    def on_response(self, nreq, result):
        super(ResponseSaver, self).on_response(nreq, result)
        if nreq == self.wait_nreq:
            self.save_result(nreq, result)
            self.wait_nreq += 1
            while self.wait_nreq in self.buffered_results:
                self.save_result(self.wait_nreq, self.buffered_results[self.wait_nreq])
                del self.buffered_results[self.wait_nreq]
                self.wait_nreq += 1
        else:
            self.buffered_results[nreq] = result

    def save_result(self, nreq, result):
        self.res_file.write(result.strip("\n\r"))
        self.res_file.write(self.custom_sep.format(nreq + 1))


class BinaryResponseSaver(ResponseSaver):
    next_result_pos = 0
    result_file = None
    index_file = None

    def run(self, req_iter, output_fname, index_fname=None):
        with open(output_fname, 'w') as f:
            self.result_file = f
            self.index_file = open(index_fname, 'w') if index_fname else None
            self.next_result_pos = 0
            try:
                self.use(req_iter, self.search_component, ctx=self.ctx)
            finally:
                self.result_file = None
                if self.index_file is not None:
                    self.index_file.close()
                    self.index_file = None

    def save_result(self, nreq, result):
        if self.index_file is not None:
            self.index_file.write(struct.pack('Q', self.next_result_pos))
        write_binary_data(self.result_file, result)
        self.next_result_pos += 4 + len(result)


class InfoRequester(Requester):
    """
        Sends request and validates response by regex pattern
    """

    def __init__(self, task_ctx, info_requests):
        super(InfoRequester, self).__init__()
        self.__info_requests = info_requests
        self.__info_errors = []
        self.init(
            requests_iterator=(tp[0] for tp in self.__info_requests),
            process_count=task_ctx[Params.WorkersCount.name],
            request_timeout=task_ctx[Params.RequestTimeout.name],
        )

    def on_response(self, nreq, result):
        logging.debug('Request[%s]: %s\nResult: %s', nreq, self.in_fly[nreq], result)
        self.__validate_response(self.in_fly[nreq], result, self.__info_requests[nreq][1])
        super(InfoRequester, self).on_response(nreq, result)

    def use_component(self, search_component):
        super(InfoRequester, self).use_component(search_component)
        if self.__info_errors:
            self.error = (self.error or "") + "\n" + "\n".join(self.__info_errors)

    def __validate_response(self, req, resp, required_lines):
        req_re = {}
        for r in required_lines:
            req_re[r] = re.compile(r)
        for line in resp.split('\n'):
            for r, rm in req_re.iteritems():
                if rm.search(line) is not None:
                    del req_re[r]
                    break
        if len(req_re):
            require_re = ','.join(req_re)
            self.__info_errors.append(
                "Required pattern '{}' not found for request {} in response:\n{}".format(require_re, req, resp)
            )


class SimpleRequester(Requester):
    """
        Simplest requester to validate connection errors only
    """

    def __init__(self, task_ctx, requests_iterator):
        super(SimpleRequester, self).__init__()
        self.init(
            requests_iterator=requests_iterator,
            process_count=task_ctx[Params.WorkersCount.name],
            request_timeout=task_ctx[Params.RequestTimeout.name],
        )
