import itertools
import os

from sandbox.sandboxsdk import errors

from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils
import sandbox.projects.common.base_search_quality.response_saver_params as response_saver_params
from sandbox.projects.common.search.response import cgi as response_cgi
from . import requester
from . import requester_core


def create_params():
    return (
        requester.Params.WorkersCount,
        requester.Params.RequestTimeout
    )


def save_responses(task_ctx, requests_iterator, url_read=requester_core.url_read):
    request_timeout = task_ctx[requester.Params.RequestTimeout.name]
    max_retries = utils.get_or_default(
        task_ctx, response_saver_params.DefaultResponseSaverParams.MaxRetriesForEveryQuery
    )
    requests_data_iterator = itertools.izip(
        requests_iterator,
        itertools.repeat(request_timeout),
        itertools.repeat(max_retries),
    )

    for simple_response in requester_core.simple_response_yielder(
        task_ctx[requester.Params.WorkersCount.name],
        requests_data_iterator,
        url_read,
    ):
        yield simple_response


def generate_queries_old(
    task_ctx, queries_path, target,
    collection="yandsearch",
    need_dbgrlv=True,
    service_url=None
):
    """
    Request processing, compatible with basesearch_quality.response_saver
    """
    drsp = response_saver_params.DefaultResponseSaverParams
    queries_limit = utils.get_or_default(task_ctx, drsp.QueriesLimitParameter)
    use_binary = task_ctx[drsp.UseBinaryResponses.name]
    get_all_factors = utils.get_or_default(task_ctx, drsp.GetAllFactorsParameter)
    use_dcfm = utils.get_or_default(task_ctx, drsp.UseDCFM)
    additional_cgi_params = utils.get_or_default(task_ctx, drsp.AdditionalCgiParams)
    stabilize_responses = utils.get_or_default(task_ctx, drsp.StabilizeResponses)

    with open(queries_path) as queries_file:
        for num, query in enumerate(queries_file, 1):
            custom_params = response_cgi.UrlCgiCustomizer()
            query = query.rstrip("\n")

            # search_info parameter adds a lot of unstable props. See SEARCH-898 for details
            query = query.replace("&search_info=da", "").replace("&search_info=1", "").replace("&search_info=yes", "")

            query = response_cgi.force_timeout(query)

            if query.startswith("?"):
                query = "/{}{}".format(collection, query)

            if use_binary:
                query = query.replace("&hr=da", "")
                query = response_cgi.remove_protobuf_compression(query)
            else:
                custom_params.add_custom_param("hr", "da")

            if not response_cgi.get_request_type(query):
                custom_params.add_custom_param("ms", "proto")

            gta_param = "fsgta" if "&haha=da" in query else "gta"

            if need_dbgrlv:
                custom_params. \
                    add_custom_param("dbgrlv", "da"). \
                    add_custom_param(gta_param, "_RelevFormula")

            if get_all_factors:
                custom_params.add_custom_param(gta_param, "_RelevFactors")
                if use_dcfm:
                    custom_params.add_custom_pron("dcfm")

            if stabilize_responses:
                (
                    custom_params
                    .add_custom_param("waitall", "da")
                    .add_custom_param("dump_groupings", "yes")
                    .add_custom_pron("norandomgroupselection")
                    .add_custom_pron("nosmfa")  # https://st.yandex-team.ru/SEARCH-556#1453757532000
                )

            if additional_cgi_params:
                query += ("" if additional_cgi_params.startswith("&") else "&") + additional_cgi_params

            query = custom_params.apply_to_query(query)

            yield requester_core.build_request(service_url if service_url is not None else "http://localhost:{}".format(target.port), query)
            if queries_limit and num == queries_limit:
                break


def save_responses_old(task_ctx, queries_path, responses_path, target, collection="yandsearch", service_url=None):
    eh.ensure(os.stat(queries_path).st_size, "Input file with queries is empty")

    requests_iterator = generate_queries_old(task_ctx, queries_path, target, collection=collection, service_url=service_url)

    response_separator = "\n" + "-" * 100 + "{}\n"
    use_binary = task_ctx[response_saver_params.DefaultResponseSaverParams.UseBinaryResponses.name]

    with open(responses_path, "w") as responses_file:
        for response_num, response_status, response_data in save_responses(task_ctx, requests_iterator):
            if not response_status:
                raise errors.SandboxTaskFailureError("Failed to query #{}: {}".format(response_num, response_data))
            if use_binary:
                requester.write_binary_data(responses_file, response_data)
            else:
                responses_file.write(response_data.strip("\n\r"))
                responses_file.write(response_separator.format(response_num + 1))
