"""
    Here is core functions of new requester.
    No sandbox dependencies allowed here!
"""
import abc
import six
import base64
import logging
import multiprocessing
import multiprocessing.dummy
import traceback
import time
import urllib2
from sandbox.projects.common import requests_wrapper
import sandbox.common.errors as sb_err
import sandbox.projects.common.search.response.cgi as response_cgi
import sandbox.projects.common.app_host.converter as apphost_converter
import sandbox.projects.common.app_host.patcher as apphost_patcher

# ###########################
# iterating over request file


def apphost_requests_iter(file_name):
    """
        Generates sequence of apphost request contexts from base64-encoded strings
        :param file_name: name of file with queries (WebMiddlesearchApphostRequests resource)
        :return generator with decoded queries
    """
    with open(file_name) as f:
        for base64_line in f:
            base64_line = base64_line.strip('\n')
            if base64_line:
                yield base64.decodestring(base64_line)


# ##################
# preparing requests


def build_request(url, req):
    if isinstance(req, six.string_types):
        return "{}{}".format(url, req)
    elif len(req) == 2:
        return "{}{}".format(url, req[0]), req[1]
    elif len(req) == 3:
        return "{}{}".format(url, req[0]), req[1], req[2]
    raise Exception('Unsupported req: {}'.format(req))


class QueryTransformer(object):
    __metaclass__ = abc.ABCMeta

    @abc.abstractmethod
    def apply(self, query):
        """Apply transformation on query"""


class ApphostQueryTransformer(QueryTransformer):
    """
        Here should be defined ALL options of query transformation
    """

    def __init__(
        self, base_url, converter,
        additional_cgi="",
        stabilize=True,
        need_dbgrlv=True,
        log_src_groupings=False,
        log_src_answer=False,
        log_factors=False,
        get_all_factors=False,
        use_dcfm=True,
        verbose_transform=False,
    ):
        self.base_url = base_url
        self.verbose_transform = verbose_transform
        self.converter = converter
        self.patch = {
            apphost_patcher.REMOVE: {u"search_info": [u"da", u"1", u"yes"]},
            apphost_patcher.ADD: {
                u"rearr": [u"scheme_Local/dump=1"],  # SEARCH-2515
                u"debug": [u"log_rearrange_worked_rule"],  # SEARCH-4301
            },
            apphost_patcher.FORCE_REPLACE: {u"timeout": [u"{}".format(response_cgi.MAX_TIMEOUT)]},
        }
        if additional_cgi:
            for cgi_name, cgi_val in response_cgi.split_to_cgi(additional_cgi):
                self.upd_patch({apphost_patcher.ADD: {six.text_type(cgi_name): [six.text_type(cgi_val)]}})
        if stabilize:
            self.upd_patch({
                apphost_patcher.ADD: {
                    u"pron": [u"norandomgroupselection", u"nosmfa", u"lua_rearrange_time_limit_80"],
                    u"waitall": [u"da"],
                    u"rearr": [u"scheme_Local/DumpGrouping=1"],
                    u"dump_groupings": [u"yes"],
                },
                apphost_patcher.REMOVE: {u"debug": [u"eventlogs"]},
            })
        if need_dbgrlv:
            self.upd_patch({
                apphost_patcher.ADD: {
                    u"dbgrlv": [u"da"],
                    u"gta": [u"_RelevFormula"],
                }
            })
        if log_src_groupings:
            self.upd_patch({apphost_patcher.ADD: {u"debug": [u"logsrcgroupings"]}})
        if log_src_answer:
            self.upd_patch({apphost_patcher.ADD: {u"debug": [u"logsrcanswer"]}})
        if log_factors:
            self.upd_patch({apphost_patcher.ADD: {u"debug": [u"log_factors"]}})
        if use_dcfm:
            self.upd_patch({apphost_patcher.ADD: {u"pron": [u"dcfm"]}})
        if get_all_factors:
            self.upd_patch({apphost_patcher.ADD: {u"gta": [u"_RelevFactors"]}})

    def upd_patch(self, upd):
        for add_rm, params in upd.iteritems():
            for param_name, param_vals in params.iteritems():
                self.patch.setdefault(add_rm, {}).setdefault(param_name, []).extend(param_vals)

    def apply(self, query):
        if self.verbose_transform:
            logging.debug("Initial query:\n%s", query)
        q = apphost_converter.convert_input_request(self.converter, query)
        if self.verbose_transform:
            logging.debug("Converted query:\n%s", q)
        apphost_patcher.patch_global_ctx(q, self.patch)
        if self.verbose_transform:
            logging.debug("Patched query:\n%s", q)
        q = apphost_converter.convert_output_request(self.converter, q)
        if self.verbose_transform:
            logging.debug("Encoded query:\n%s", base64.b64encode(q))
        return build_request(self.base_url, ("/", q))


class CgiQueryTransformer(QueryTransformer):
    """
        Here should be defined ALL options of query transformation
    """

    def __init__(
        self, base_url,
        make_binary=False,
        additional_cgi="",
        stabilize=True,
        need_dbgrlv=True,
        log_src_groupings=False,
        log_src_answer=False,
        log_factors=False,
        get_all_factors=False,
        use_dcfm=True,
        collection="yandsearch",
    ):
        self.common_params = response_cgi.UrlCgiCustomizer(
            base_url=base_url,
            params=additional_cgi,
            del_params=(("search_info", "da"), ("search_info", "1"), ("search_info", "yes"))
        )
        self.common_params.add_custom_param('debug', 'log_rearrange_worked_rule')  # SEARCH-4301
        self.common_params.add_custom_param('rearr', 'scheme_Local/dump=1')  # SEARCH-6515
        self.make_binary = make_binary
        self.need_dbgrlv = need_dbgrlv
        self.get_all_factors = get_all_factors
        self.collection = collection
        if stabilize:
            self.common_params.add_custom_params([
                ("waitall", "da"),
                ("pron", "norandomgroupselection"),
                ("pron", "nosmfa"),  # https://st.yandex-team.ru/SEARCH-556#1453757532000
                ("pron", "lua_rearrange_time_limit_80"),
                ("rearr", "scheme_Local/DumpGrouping=1"),
                ("dump_groupings", "yes"),
            ])
            self.common_params.remove_custom_params([
                ("debug", "eventlogs")
            ])
        if use_dcfm:
            self.common_params.add_custom_pron("dcfm")
        if log_src_groupings:
            self.common_params.add_custom_param("debug", "logsrcgroupings")
        if log_src_answer:
            self.common_params.add_custom_param("debug", "logsrcanswer")
        if log_factors:
            self.common_params.add_custom_param("debug", "log_factors")

    def apply(self, query):
        q = query.rstrip("\n").split("?", 1)[-1]
        q = response_cgi.force_timeout(q)
        all_params = response_cgi.UrlCgiCustomizer(
            params=self.common_params.params,
            del_params=self.common_params.del_params,
        )

        gta_name = "fsgta" if "&haha=da" in q else "gta"
        if self.need_dbgrlv:
            all_params.add_custom_params([
                ("dbgrlv", "da"),
                (gta_name, "_RelevFormula"),
            ])
        if self.get_all_factors:
            all_params.add_custom_param(gta_name, "_RelevFactors")
        if not response_cgi.get_request_type(q):
            all_params.add_custom_param("ms", "proto")

        if self.make_binary:
            q = response_cgi.remove_protobuf_compression(q)
            all_params.remove_custom_param("hr", "da")
        else:
            all_params.add_custom_param("hr", "da")

        q = all_params.apply_to_query(q)
        q = "/{}?{}".format(self.collection, q)

        return build_request(self.common_params.base_url, q)


def simple_request_transformer(queries_iter, transformer):
    """
        Transform sequence of queries in a parallel way
        :param queries_iter: iterator with queries
        :param transformer: instance of QueryTransformer
        :return: generator with transformed queries
    """
    worker_n = multiprocessing.cpu_count()
    logging.info("Transform queries using %s processes", worker_n)
    pool = multiprocessing.Pool(processes=worker_n)
    try:
        for num, req in enumerate(pool.imap(transformer.apply, queries_iter)):
            yield req
            if num % 100 == 0:
                logging.debug('Transformed %s requests', num)
    finally:
        pool.terminate()


# #################
# getting responses


def url_open(req, timeout):
    if isinstance(req, six.string_types):
        return urllib2.urlopen(req, timeout=timeout)
    elif len(req) == 3:
        request = urllib2.Request(req[0], req[1], req[2])
        return urllib2.urlopen(request, timeout=timeout)
    return urllib2.urlopen(req[0], req[1], timeout=timeout)


def url_open_requests(req, timeout):
    if isinstance(req, six.string_types):
        return requests_wrapper.get_r(req, timeout=timeout)
    elif len(req) == 3:
        return requests_wrapper.post_r(req[0], data=req[1], headers=req[2], timeout=timeout)
    return requests_wrapper.post_r(req[0], data=req[1], timeout=timeout)


def url_read(args):
    """
        :param args: tuple with request, timeout and number of maximum retries
        :return: tuple (response_status, response_data).

            In case of correct answer:
                Response status = true
                Response data = result of urllib2.urlopen().read()

            In case of bad answer:
                Response status = false
                Response data = exception message
    """
    req, timeout, max_retries = args
    except_msg = ""
    for i in range(max_retries + 1):
        try:
            u = url_open(req, timeout)
            return True, u.read()
        except Exception as e:
            except_msg = str(e)
            logging.debug("Problems with request '%s' (attempt %s):\n%s", req, i, except_msg)
            time.sleep(2)

    logging.error("Unresolved problems with request '%s':\n%s", req, except_msg)
    return False, except_msg


def simple_response_yielder(worker_n, requests_data_iterator, url_read_func=url_read):
    """
        Yield responses using threads. Why not to use processes?
        Because of requests_data_iterator, which could be a pool.imap().
        So, if that pool already eat all the n_cpu, there will be a deadlock.
        Besides, the speed is not so important for that function.
        :param worker_n: number of threads
        :param requests_data_iterator: iterator with incoming requests data
                                       each iter should be a tuple (request, timeout, max_retries)
        :param url_read_func: function should return tuple of (response_status, response_data)
                              using same rules as url_read
        :return: generator with tuples: (response_number, response_status, response_data)
    """
    worker_n = worker_n or 1
    logging.info("Yield responses using %s threads", worker_n)
    pool = multiprocessing.dummy.Pool(processes=worker_n)
    try:
        for num, response in enumerate(pool.imap(url_read_func, requests_data_iterator)):
            response_status, response_data = response
            yield num, response_status, response_data
            if num % 100 == 0:
                logging.debug('Yielded %s responses', num)
    finally:
        pool.terminate()


def url_read_and_transform(args):
    """
        :return: tuple (request, response_status, response_data)
    """
    request, timeout, max_retries, transformer = args
    if transformer:
        try:
            request = transformer.apply(request)
        except Exception:
            raise sb_err.TaskFailure("Unable to transform request ({}):\n{}".format(request, traceback.format_exc()))
    return (request,) + url_read((request, timeout, max_retries))


def response_yielder(worker_n, requests_data_iterator, url_read_and_transform_func=url_read_and_transform):
    """
        Yield responses using parallel processes. Also, transform incoming queries.
        :param worker_n: number of threads
        :param requests_data_iterator: iterator with incoming requests data
            each iter should be a tuple (request, timeout, max_retries, transformer)
        :param url_read_and_transform_func: function should return tuple of (request, response_status, response_data)
            using same rules as url_read
        :return: generator with tuples: (response_number, request, response_status, response_data)
    """
    worker_n = worker_n or multiprocessing.cpu_count()
    logging.info("Yield responses using %s processes (with preliminary query transformation)", worker_n)
    pool = multiprocessing.Pool(processes=worker_n)
    try:
        for num, response in enumerate(pool.imap(url_read_and_transform_func, requests_data_iterator)):
            yield (num,) + response
            if num % 10 == 0 and num:
                logging.debug('Yielded %s responses', num)
    finally:
        pool.terminate()


def response_yielder_sequential(requests_data_iterator, url_read_and_transform_func=url_read_and_transform):
    """
        For debug
    """
    logging.info("Yield responses sequentially")
    for num, requests_data in enumerate(requests_data_iterator):
        logging.debug('Try to yield %s response', num)
        response = url_read_and_transform_func(requests_data)
        yield (num,) + response
        if num % 100 == 0:
            logging.debug('Yielded %s responses', num)
