# -*- coding: utf-8 -*-

"""
    Here is the code for resource getting from nanny via nanny_client

    Please obey PEP8 coding standards and Sandbox StyleGuide here
    https://wiki.yandex-team.ru/sandbox/codestyleguide
"""
import logging

from sandbox.sandboxsdk.channel import channel

from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import decorators
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import apihelpers
from sandbox.projects.common import requests_wrapper


class ResLocation(object):
    """Base class"""
    location = "UNKNOWN_LOCATION"

    def __init__(self, res_name, res_data):
        self.res_name = res_name
        self.res_type = res_data["res_type"]
        self.checker = res_data["checker"]

    def _get_aggregate(self, res_info):
        """
            Gets appropriate aggregate using custom checker.
            Checker could be the dict with lists of optional check values.
            Check values should be presented in desired order (
                ex.: ["apache.ywsearch.cfg-a_geo_man", "apache.ywsearch.cfg-a_geo_msk",] means, that
                firstly we'll look for "apache.ywsearch.cfg-a_geo_man", and only if not found, look for the next one
            )
            :param res_info: dict with resources info
        """
        logging.info("Getting %s of type '%s' from '%s'", self.res_name, self.res_type, self.location)
        local_info_list = res_info["content"].get(self.location, [])
        eh.ensure(local_info_list, "Location is not valid!")
        for i in range(max([len(c_list) for c_list in self.checker.itervalues()])):
            for item in local_info_list:
                if all([item.get(key) in vals[:i + 1] for key, vals in self.checker.iteritems()]):
                    return item

    def get_res_id(self, res_info):
        raise NotImplementedError()


class SandboxFilesLocation(ResLocation):
    location = "sandbox_files"
    aggregate_key = "task_id"

    def get_res_id(self, res_info):
        agg = self._get_aggregate(res_info)
        if agg is not None:
            return apihelpers.get_task_resource_id(agg[self.aggregate_key], self.res_type)


class StaticFilesLocation(ResLocation):
    location = "static_files"
    aggregate_key = "content"

    def get_res_id(self, res_info):
        agg = self._get_aggregate(res_info)
        if agg is not None:
            resource_path = agg["local_path"]
            fu.write_file(resource_path, agg[self.aggregate_key])
            return channel.task.create_resource(
                resource_path, resource_path, self.res_type,
            ).id


class UrlLocation(ResLocation):
    location = "url_files"
    aggregate_key = "url"

    @decorators.retries(max_tries=5, delay=30)
    def get_res_id(self, res_info):
        agg = self._get_aggregate(res_info)
        if agg is not None:
            resource_path = agg["local_path"]
            response = requests_wrapper.get(agg[self.aggregate_key])
            response.raise_for_status()
            fu.write_file(resource_path, response.text)
            return channel.task.create_resource(
                resource_path, resource_path, self.res_type,
            ).id
