# -*- coding: utf-8 -*-

import re
import six
from six.moves.urllib import parse

"""
    Search-specific cgi request patching parse things.
    Please DO NOT put any sandbox dependencies.
"""

# protobuf compression rules
_PRON_PC_MATCH = re.compile(r'&pron=pc[a-z-]+')
_PC_MATCH = re.compile(r'&pc=[a-z-]+')
_TIMEOUT_MATCH = re.compile(r'&timeout=[^&\s]+')
_DB_TIMESTAMP_MATCH = re.compile(r'&pron=db_timestamp[^&\s]+')

# supermind degradation rules
_SUPERMIND_CONTROLS = re.compile(
    r'&pron='
    r'(garbagesmm_[0-9.-]+'
    r'|smm_[0-9.-]+'
    r'|smdt_[0-9.-]+'
    r'|smlb_[0-9.-]+'
    r'|smncpudt_[0-9.-]+'
    r'|smautodegrade_[a-z0-9._-]+'
    r'|smautodrop_[0-9.-]+'
    r'|smfa'
    r'|nosmfa'
    r')'
)

MAX_TIMEOUT = 4294967295


def get_cgi_extractor(cgi_name):
    """Should return regexp pattern, which extracts cgi as follows: '&cgi_name=cgi_values'"""
    return re.compile(r'&{}=[^&\s]*'.format(cgi_name))


def append_cgi(cgi_name, cgi_val_to_append, reqs_iter):
    """Returns generator of queries with appended cgi param"""
    cgi_extractor = get_cgi_extractor(cgi_name)
    for req in reqs_iter:
        cgi_list = cgi_extractor.findall(req)
        if cgi_list:
            for cgi in cgi_list:
                cgi_name, cgi_val = cgi.split("=", 1)
                if not cgi_val:
                    yield req.replace(cgi, "{}{}".format(cgi, cgi_val_to_append))
                elif cgi_val_to_append not in cgi_val:
                    yield req.replace(cgi, "{};{}".format(cgi, cgi_val_to_append))
                else:
                    # cgi is already exists in query
                    yield req

        else:
            yield "{}&{}={}".format(req, cgi_name, cgi_val_to_append)


class UrlCgiCustomizer(object):
    """
        Use this class for custom url generation purposes.
        Usage example:
            my_url = cgi.UrlCgiCustomizer(base_url="yandex.ru")
            my_url.add_text("hello world").notests().add_custom_param("key", "value")
            my_response = requests_wrapper.get(my_url.base_url, params=my_url.params)
        or just
            query_string = str(my_url)
    """

    def __init__(self, base_url="", params=None, del_params=None):
        self.base_url = base_url
        if base_url.startswith('http://') or base_url.startswith('https://'):
            self.base_url = base_url
        elif base_url:
            self.base_url = 'http://' + base_url
        if params is None:
            self.__add_params = ()
        elif isinstance(params, (six.text_type, six.binary_type)):
            self.__add_params = tuple(split_to_cgi(params))
        else:
            self.__add_params = tuple(params)
        if del_params is None:
            self.__del_params = ()
        elif isinstance(del_params, (six.text_type, six.binary_type)):
            self.__del_params = tuple(split_to_cgi(del_params))
        else:
            self.__del_params = tuple(del_params)

    def replace_tld(self, tld):
        self.base_url = re.sub(r"yandex\.[a-z]+", "yandex.{}".format(tld), self.base_url)
        return self

    def add_region(self, reg):
        self.__add_params += (("lr", reg),)
        return self

    def add_text(self, custom_text):
        self.__add_params += (("text", custom_text),)
        return self

    def add_custom_params(self, params_seq):
        """
            :param params_seq: sequence of tuples (key, value) params
            :return: self
        """
        self.__add_params += tuple(params_seq)
        return self

    def remove_custom_params(self, params_seq):
        """
            :param params_seq: sequence of tuples (key, value) params
            :return: self
        """
        self.__del_params += tuple(params_seq)
        return self

    def add_custom_cgi_string(self, cgi_string):
        """
            :param cgi_string: correctly serialized string of CGI params
            :return: self
        """
        self.__add_params += tuple(split_to_cgi(cgi_string))
        return self

    def add_custom_param(self, key, val):
        self.__add_params += ((key, val),)
        return self

    def remove_custom_param(self, key, val):
        self.__del_params += ((key, val),)
        return self

    def add_custom_pron(self, val):
        self.__add_params += (("pron", val),)
        return self

    def disable_experiments(self):
        self.__add_params += (("no-tests", "1"),)
        return self

    def enable_light_cookie(self):
        self.__add_params += (('sbh', '1'),)
        return self

    def disable_cache(self):
        self.__add_params += (("nocache", "da"),)
        return self

    def disable_mda(self):
        self.__add_params += (("flag", "disable_mda"),)
        return self

    def set_max_timeout(self):
        self.__add_params += (("timeout", str(MAX_TIMEOUT)),)
        return self

    def no_cookie_support(self):
        """Prevent redirects to Passport (pass.yandex.*), s/a SEARCH-2212"""
        self.__add_params += (("nocookiesupport", "yes"),)
        return self

    def apply_to_query(self, query):
        """
            Removes __del_params from query, then appends __add_params to query
        """
        return self.remove_from_query(query) + "".join(self._new_params_for_query(query))

    def remove_from_query(self, query):
        for p in self.__del_params:
            str_p = self._form_param_str(p)
            query = query.replace(str_p, "")
        return query

    def _new_params_for_query(self, query):
        for p in self.__add_params:
            str_p = self._form_param_str(p)
            if str_p not in query:
                yield str_p

    @staticmethod
    def _form_param_str(param):
        key = param[0]
        value = param[1]

        if isinstance(value, int):
            value = str(value)
        elif isinstance(value, str):
            pass  # leave as is
        elif six.PY2 and isinstance(value, six.text_type):
            value = value.encode("utf-8")
        else:
            # FIXME(mvel): Strange fallback. TODO: remove it?
            value = value.encode("ASCII", "replace")

        return "&{}={}".format(parse.quote_plus(key), parse.quote_plus(value))

    def __str__(self):
        if not self.base_url:
            return self.params_to_str()
        else:
            return "?".join([self.base_url, self.params_to_str()])

    @property
    def params(self):
        return self.__add_params

    @property
    def del_params(self):
        return self.__del_params

    def params_to_str(self):
        # please note that urlencode does not encode utf-8 sequences properly, so we have custom code here
        result = ""
        for param in self.__add_params:
            result += UrlCgiCustomizer._form_param_str(param)
        return result


def get_cgi_string(url):
    """
        Return cgi params string from url
        :param url: url string with cgi
        :return: string of cgi params
    """
    return url[url.find("?") + 1:]


def parse_key_value_parm(cgiparam, unquote=True):
    parts = cgiparam.split('=', 1)
    if unquote:
        parts = [parse.unquote_plus(part) for part in parts]
    if len(parts) < 2:
        return parts[0], ''
    else:
        return tuple(parts)


def split_to_cgi(s, unquote=True, url=False, delim='&', encode_to_ascii=False, sub_param_names=None):
    """
        Splits string to cgi tuples.
        :param s: string, contained cgi
        :param unquote: decode url params
        :param url: get cgi string from url
        :param delim: delimiter for cgi parameters
        :param encode_to_ascii: whether to call .encode('ASCII')
        :param sub_param_names: this params will be split by semicolon (';') and parsed as key=value
        :return: generator of tuples
    """
    if sub_param_names is None:
        sub_param_names = set()
    if encode_to_ascii:
        s = s.encode('ASCII')
    if url:
        s = get_cgi_string(s)
    for key_and_value in s.split(delim):
        if not key_and_value:
            continue
        parts = key_and_value.split('=', 1)

        if unquote:
            parts = [parse.unquote(part) for part in parts]

        if parts[0] in sub_param_names:
            sub_params = split_to_cgi(parts[1], unquote=False, delim=';', encode_to_ascii=False, sub_param_names=None)
            parts = (parts[0], list(sub_params))

        if len(parts) < 2:
            yield parts[0], ''
        else:
            yield tuple(parts)


def remove_protobuf_compression(query):
    """
        Remove all protobuf response compression switches from cgi params.
        See also SEARCH-556.
    """
    query = _PRON_PC_MATCH.sub('', query)
    query = _PC_MATCH.sub('', query)
    return query


def remove_supermind_controls(query):
    """
        Remove all supermind-related switches from cgi params to stabilize performance.
        See also SEARCH-2050.
    """
    return _SUPERMIND_CONTROLS.sub('', query)


def remove_db_timestamp(query):
    """
        Remove required db timestamp (pron=db_timestamp).
        See SEARCH-2872.
    """
    return _DB_TIMESTAMP_MATCH.sub('', query)


def remove_timeout(query):
    """
        Remove all timeout settings from query
    """

    return _TIMEOUT_MATCH.sub("", query)


def force_timeout(query, timeout=MAX_TIMEOUT):
    """
        Force timeout settings in query to stabilize results

        By default method will set timeout to maximum possible value.
    """

    return remove_timeout(query) + "&timeout={}".format(timeout)


def _get_cgi_searcher(cgi):
    return re.compile(r''.join([r'(\?|&)', cgi, r'(&|$)']))


def _get_ms_type(typename):
    return _get_cgi_searcher(r'ms=' + typename)


_REQUEST_TYPES = {
    'xml': [_get_cgi_searcher(r'xml=yes'), _get_ms_type(r'xml'), _get_ms_type(r'cloud')],
    'proto': [_get_ms_type(r'proto')]
}


def _get_all_types(query):
    return [typename for typename, tests in _REQUEST_TYPES.items() if any((m.search(query) for m in tests))]


def get_request_type(query):
    types = _get_all_types(query)

    return types[-1] if types else None
