"""
    This file is for custom comparison of KeyVal props in protobuf:
    If you need to compare any field in any unusual way, do it as follows:

    1) write a function, that compares two values you need
    2) add it to CMP_FUNCTIONS with key of your prop

    Please, obey pep8 code style here!
"""

import logging
import time
import json
from six.moves import xrange

from sandbox.projects.common.base_search_quality import snipdiff
from sandbox.projects.common.differ import printers
from sandbox.projects.common.differ import json_differ
from sandbox.projects.common.search import factors_decompressor as fd
from sandbox.projects.common.search.response import patterns as rp

FACTOR_NAMES = [[]] * 2
SOFT_CHECK_FACTORS_INDEXES = set()
SOFT_CHECK_FACTORS_OFFSETS_WITH_SLICES = {}


def cmp_relev_formula(string1, string2, output):
    compressed_string_1 = _compress_relev_formula(string1)
    compressed_string_2 = _compress_relev_formula(string2)
    json_differ.cmp_strings(compressed_string_1, compressed_string_2, output)


def cmp_relev_factors(string1, string2, output):
    factors_str_1, slices1 = rp.compress_factors_with_slices(string1)
    factors_str_2, slices2 = rp.compress_factors_with_slices(string2)
    _cmp_factors((factors_str_1.split(), slices1), (factors_str_2.split(), slices2), output)


def cmp_snippet_explanation(string1, string2, output):
    start_time = time.time()
    result = snipdiff.global_compare(string1, string2)
    for algo_dict_type, algo_dict in result.iteritems():
        if algo_dict:
            output.sched("{} Algo Group".format(algo_dict_type))
            for algo_name, pairs in algo_dict.iteritems():
                output.sched(algo_name)
                output(str(pairs))
                output.desched()
            output.desched()
    usec_delta = int((time.time() - start_time) * 1000000 * 100) / 100
    logging.debug("Snippet parse time: %susec", usec_delta)


def _compress_relev_formula(string):
    value = ""

    match_model = rp.MODEL_TEMPLATE.search(string)
    if match_model:
        value += match_model.groups(0)[0]
    match_polynom = rp.POLYNOM_TEMPLATE.search(string)
    if match_polynom:
        value += " ({})".format(match_polynom.groups(0)[0])

    return value


def _cmp_factors(factors_and_slices_1, factors_and_slices_2, output):
    list1, slices1 = factors_and_slices_1
    list2, slices2 = factors_and_slices_2

    if slices1 and slices2:
        with_slices = True
    else:
        with_slices = False

    if with_slices:
        len_slices1 = len(slices1)
        len_slices2 = len(slices2)
        slices_cnt = min(len_slices1, len_slices2)
        output_slice_info = len_slices1 != len_slices2
        b1 = 0
        b2 = 0
        for cur_slice in xrange(slices_cnt):
            len1 = slices1[cur_slice][1]
            len2 = slices2[cur_slice][1]
            if output_slice_info:
                view_slices1 = "{} {} [{}:{}]".format(slices1[cur_slice][0], len1, b1, b1 + len1)
                view_slices2 = "{} {} [{}:{}]".format(slices2[cur_slice][0], len2, b2, b2 + len2)
                diff_type = printers.DiffType.NODIFF if view_slices1 == view_slices2 else printers.DiffType.CHANGED
                output.sched("slice {}".format(cur_slice))
                output("{} -> {}".format(view_slices1, view_slices2), diff_type)
                output.desched()
            for offset in xrange(max(len1, len2)):
                i1 = b1 + offset
                i2 = b2 + offset
                if offset < min(len1, len2):
                    if (
                        list1[i1] != list2[i2]
                        and (
                            abs(float(list1[i1]) - float(list2[i2])) > 0.0005
                            or offset not in SOFT_CHECK_FACTORS_OFFSETS_WITH_SLICES.get(slices1[cur_slice][0], [])
                        )
                    ):
                        output.sched("{} {}".format(slices1[cur_slice][0], offset))
                        output("{} -> {}".format(list1[i1], list2[i2]), printers.DiffType.CHANGED)
                        output.desched()
                elif offset < len1:
                    output.sched("{} {}".format(slices1[cur_slice][0], offset))
                    output(str(list1[i1]), printers.DiffType.REMOVED)
                    output.desched()
                else:
                    output.sched("{} {}".format(slices2[cur_slice][0], offset))
                    output(str(list2[i2]), printers.DiffType.ADDED)
                    output.desched()
            b1 += len1
            b2 += len2
        for cur_slice in xrange(slices_cnt, len_slices1):
            len1 = slices1[cur_slice][1]
            output.sched("slice {}".format(cur_slice))
            output("{} {} [{}:{}]".format(slices1[cur_slice][0], len1, b1, b1 + len1), printers.DiffType.REMOVED)
            output.desched()
            for offset in xrange(len1):
                i1 = b1 + offset
                output.sched("{} {}".format(slices1[cur_slice][0], offset))
                output(str(list1[i1]), printers.DiffType.REMOVED)
                output.desched()
            b1 += len1
        for cur_slice in xrange(slices_cnt, len_slices2):
            len2 = slices2[cur_slice][1]
            output.sched("slice {}".format(cur_slice))
            output("{} {} [{}:{}]".format(slices2[cur_slice][0], len2, b2, b2 + len2), printers.DiffType.ADDED)
            output.desched()
            for offset in xrange(len2):
                i2 = b2 + offset
                output.sched("{} {}".format(slices2[cur_slice][0], offset))
                output(str(list2[i2]), printers.DiffType.ADDED)
                output.desched()
            b2 += len2
    else:
        len1 = len(list1)
        len2 = len(list2)
        for i in xrange(max(len1, len2)):
            if i < min(len1, len2):
                if (
                    list1[i] != list2[i]
                    and (
                        i not in SOFT_CHECK_FACTORS_INDEXES
                        or abs(float(list1[i]) - float(list2[i])) > 0.0005
                    )
                ):
                    output.sched(i if not FACTOR_NAMES[0] else FACTOR_NAMES[0][i])
                    output("{} -> {}".format(list1[i], list2[i]), printers.DiffType.CHANGED)
                    output.desched()
            elif i < len1:
                output.sched(i if not FACTOR_NAMES[0] else FACTOR_NAMES[0][i])
                output(str(list1[i]), printers.DiffType.REMOVED)
                output.desched()
            else:
                output.sched(i if not FACTOR_NAMES[1] else FACTOR_NAMES[1][i])
                output(str(list2[i]), printers.DiffType.ADDED)
                output.desched()


def cmp_compressed_all_factors(string_1, string_2, output):
    if string_1 != string_2:
        json_differ.cmp_json(
            fd.FactorsDecompressor.extract(string_1),
            fd.FactorsDecompressor.extract(string_2),
            output,
            cmp_lists=True,
        )


def cmp_user_groupings(string_1, string_2, output):
    if string_1 != string_2:
        try:
            scheme1 = json.loads(string_1, encoding='utf-8')
            scheme2 = json.loads(string_2, encoding='utf-8')
            if "user_groupings" not in scheme1 and "user_groupings" not in scheme2:
                return
            groupings1 = scheme1.get("user_groupings", json.loads("{}"))
            groupings2 = scheme2.get("user_groupings", json.loads("{}"))
        except Exception as e:
            logging.debug("Can't compare strings\n%s\nand\n%s\nas jsons:\n%s", string_1, string_2, e)
            json_differ.cmp_strings(string_1, string_2, output)
            return
        json_differ._cmp_value(
            groupings1,
            groupings2,
            output,
            cmp_lists=True,
        )


CMP_FUNCTIONS = {
    "_RelevFormula": cmp_relev_formula,
    "_RelevFactors": cmp_relev_factors,
    "_SnippetsExplanation": cmp_snippet_explanation,
    "_SerpData": json_differ.cmp_json,
    "_SerpInfo": json_differ.cmp_json,
    "_ImagesJson": json_differ.cmp_json,
    "_CompressedAllFactors": cmp_compressed_all_factors,
    "Snippet": json_differ.cmp_json,
    "scheme.json.nodump": cmp_user_groupings,
}
