# coding: utf-8

from six.moves import zip, zip_longest

import hashlib
import logging
import json
import os
import pytest

from sandbox.projects.common import file_utils as fu
from sandbox.projects.common.differ import printers
from sandbox.projects.common.search.response.diff import protodiff


@pytest.fixture(scope="module")
def metapb():
    return protodiff.get_meta_pb2()


@pytest.fixture(scope="module")
def r1(metapb):
    r = metapb.TReport()
    r.ParseFromString(read_input_data_file("r1.pb"))
    return r


@pytest.fixture(scope="module")
def r2(metapb):
    r = metapb.TReport()
    r.ParseFromString(read_input_data_file("r2.pb"))
    return r


@pytest.fixture(scope="module")
def r2c(metapb):
    r = metapb.TReport()
    r.ParseFromString(read_input_data_file("r2c.pb"))
    return r


@pytest.fixture(autouse=True)
def fd_patch(monkeypatch):
    import yatest.common
    from sandbox.projects.common.search.factors_decompressor import FactorsDecompressor, PipedFactorsDecompressor
    logging.warning("fd_patch")
    monkeypatch.setattr(
        FactorsDecompressor, "_command",
        [yatest.common.runtime.work_path("compressed_factors_decoder"), "--output-mode", "json"]
    )
    monkeypatch.setattr(
        PipedFactorsDecompressor, "_binary",
        yatest.common.runtime.work_path("compressed_factors_decoder_piped")
    )


def read_input_data_file(file_name):
    import yatest.common
    with open(yatest.common.runtime.work_path(file_name), "rb") as f:
        return f.read()


def test_evil_compressed_streams(r1, r2c):
    import yatest.common
    path_to_diffs = yatest.common.runtime.work_path("fake")

    printer = printers.PrinterToHtml(path_to_diffs)
    prt = protodiff.Protodiff(printer)
    with pytest.raises(protodiff.ProtoDiffException):
        prt.compare_single_pair(r1, r2c)


def test_differ_old(r1, r2):
    printer = printers.PrinterToList()
    prt = protodiff.Protodiff(printer)

    prt.compare_single_pair(r1, r2)
    printer.finalize()

    full_diff = printer.get_diff()
    assert len(full_diff) == 1

    compact_diff = printer.get_compact_diff()

    # todo: figure out why these two lines are slow and diffy in python3 (probably, some unicode issues)
    assert_jsons_equal('full_diff.json', full_diff[0])
    assert_jsons_equal('compact_diff.json', compact_diff)


def test_differ_new(r1, r2):
    import yatest.common
    path_to_diffs = yatest.common.runtime.work_path("test_differ_new")
    os.mkdir(path_to_diffs)

    printer = printers.PrinterToHtml(
        path_to_diffs,
        pair_head_template="response {obj_index}",
        obj_head_template="query",
    )
    prt = protodiff.Protodiff(printer)

    prt.compare_single_pair(r1, r2)
    printer.finalize()

    # todo: fix
    # expected_full_diff = fu.json_load("real_new_full_diff.json")
    # with open(os.path.join(path_to_diffs, "r_000000-000000.html")) as f:
    #     full_diff = f.read()
    # assert expected_full_diff in full_diff

    assert_jsons_equal("real_new_compact_diff.json", printer.get_compact_diff())


def assert_jsons_equal(file_name, real_result):
    expected_result_str = fu.read_file(file_name)
    real_result_str = json.dumps(real_result, indent=2, sort_keys=True)
    assert expected_result_str == real_result_str


def assert_directories_equal(dir_name):
    import yatest.common
    path_expected = yatest.common.runtime.work_path(dir_name)
    path_real = yatest.common.runtime.work_path('real_{}'.format(dir_name))
    logging.info("path_expected = %s, path_real = %s", path_expected, path_real)
    logging.info("listdir expected: %s", os.listdir(path_expected))
    logging.info("listdir real: %s", os.listdir(path_real))
    for file_name in os.listdir(path_expected):
        full_path_expected = os.path.join(path_expected, file_name)
        full_path_real = os.path.join(path_real, file_name)
        assert os.path.exists(full_path_real), "Path {} does not exist".format(full_path_real)
        assert (
            os.path.isdir(full_path_real) and os.path.isdir(full_path_expected)
            or os.path.isfile(full_path_real) and os.path.isfile(full_path_expected)
        ), "File types mismatch"
        if os.path.isdir(full_path_expected):
            assert_directories_equal(os.path.join(dir_name, file_name))
        else:
            logging.info("Comparing files: %s vs %s", full_path_expected, full_path_real)
            expected_data = fu.read_file(full_path_expected)
            real_data = fu.read_file(full_path_real)
            if hashlib.md5(expected_data) != hashlib.md5(real_data):
                assertion = "{} != {}".format(full_path_real, full_path_expected)
                logging.error(assertion)
                for expected, real in zip_longest(expected_data.split("\n"), real_data.split("\n")):
                    assert expected == real


def skipped_test_full_report(r1, r2):
    for max_file_size, report_dir_name in [
        (5 * 1024 * 1024, "full_report"),  # 5 Mb
        (1, "full_report_2"),
    ]:
        _test_full_report(r1, r2, max_file_size, report_dir_name)


def _test_full_report(r1, r2, max_file_size, report_dir_name):
    import yatest.common
    path_to_diffs = yatest.common.runtime.work_path('real_{}'.format(report_dir_name))
    os.mkdir(path_to_diffs)

    printer = printers.PrinterToHtml(
        path_to_diffs,
        max_file_size=max_file_size,
        pair_head_template="response {obj_index}",
        obj_head_template="query"
    )
    prt = protodiff.Protodiff(printer, only_complete=False, ignore_unanswered=False)
    prt.set_factor_names(["NavLinear", "EshopValue", "AuraDocLogAuthor", "TitleTrigramsQuery"])
    cmp_data_bundle = zip(
        [r1, r1, r1],
        [r2, r2, r1],
        ['query 1', 'query 2', 'query with equal results'],
    )
    prt.compare_pairs(cmp_data_bundle)
    assert_directories_equal(report_dir_name)
