# -*- coding: utf-8 -*-
import re

"""
    Search-specific response parse things.

    Please DO NOT put any sandbox dependencies.
"""


class ResponseParseError(Exception):
    pass


MODEL_TEMPLATE = re.compile("<tr><td>model</td><td><nobr>(.+?)</nobr></td></tr>")
POLYNOM_TEMPLATE = re.compile("<tr><td>polynom</td><td><nobr>(.+?)</nobr></td></tr>")

_TEMPLATE_STR = r"""
    <tr>
    <td>([^<>]+?)</td>                         # factor name
    <td>(?:<nobr>)?([^<>]+?)(?:</nobr>)?</td>  # factor value
    (<td>([^<>]+?)</td>)?                      # factor type
    (?:<td><nobr>[^<>]+?</nobr></td>)?
    (?:<td><nobr>[^<>]+?</nobr></td>)?
    </tr>"""

_FACTORS_TEMPLATE = re.compile(_TEMPLATE_STR, re.X)
_FACTORS_MATCH_TEMPLATE = re.compile("^(" + _TEMPLATE_STR + ")+$", re.X)


def beautify_factor_value(str):
    """
        Приводит значения факторов к более читабельному виду,
        убирая нули и десятичную точку.

        В режиме DCFM не требуется (нормализация производится
        непосредственно при формировании ответа поиска).
    """
    if str == "0":
        return str
    return str.rstrip("0").rstrip(".")


def compress_factors_with_slices(factors_str):
    if factors_str.startswith("[DCFM]"):
        factors_slices = factors_str[6:].split("|")
        slices = None

        if len(factors_slices) == 2:
            # "slice1 4 slice2 6" -> [("slice1", 4), ("slice2", 6)]
            slices_list = factors_slices[1].strip().split()
            slices = list(zip(slices_list[::2], map(int, slices_list[1::2])))

        factors = factors_slices[0].strip()

        return factors, slices
    else:
        return compress_factors(factors_str), None


def compress_factors(factors_str):
    """
        Выдергивает факторы из тэгов
        См. также SEARCH-841. Кидает ResponseParseError
    """
    if not factors_str:
        return ""
    elif factors_str.startswith("[DCFM]"):
        factors_slices = factors_str[6:].split("|")
        return factors_slices[0].strip()

    if not _FACTORS_MATCH_TEMPLATE.match(factors_str):
        return factors_str  # response already processed by this routine

    factors_list = _FACTORS_TEMPLATE.findall(factors_str)
    if not factors_list:
        raise ResponseParseError("_FACTORS_TEMPLATE.findall cannot parse string '{}'".format(factors_str))

    values = [beautify_factor_value(x[1]) for x in factors_list]
    return " ".join(values)
