# -*- coding: utf-8 -*-
from six.moves import range
import re


# Notification about removed snippet,
# format 'RemovedDoc_<grouping_mode><grouping_attr>_<count>', e.g., 'RemovedDoc_1d_0'
_REMOVED_DOC_KEY_PATTERN = re.compile(r"^RemovedDoc_[0-9]+[^_]+_[0-9]+$")


def can_compare(node1, node2):
    return not_respond_state(node1) == not_respond_state(node2)


def not_respond_state(response):
    """
        Принимает на вход Node ответа от поиска
        Возвращает список, описывающий уровень неотвеотов подисточников
        (оценивается с точки зрения noapache - с учётом TDI экпериментов и пр.)
        формат списка довольно сумбурный, основное предзначение - сравнение с другими списками.
    """
    res = []
    unans = []
    debug_info = response._nodes.get("DebugInfo")
    if debug_info:
        saf_list = debug_info[0]._nodes.get("SourceAnswerInfo")
        if saf_list:
            for saf in saf_list:
                nr = '{}:{}'.format(
                    saf.GetPropValue('NotRespondedClientsCount'),
                    saf.GetPropValue('NotGatheredBaseSearchAnswers')
                )
                res.append((saf.GetPropValue('Descr'), nr, saf.GetPropValue('Code')))
            res.sort()

    searcher_props = response._nodes.get('SearcherProp', [])
    for prop in searcher_props:
        korig = prop.GetPropValue('Key')
        kt = _key_after_tdi_id(korig)
        k = kt if kt else korig
        if k.endswith('SerpObject.Timedout') or k.endswith('SerpTaskTimedout'):
            res.append((korig, 1, '1'))
        elif k == 'EntitySearch.ErrorMessage.debug':
            res.append((korig, 1, str(prop.GetPropValue('Value'))))
        elif k.startswith('Unanswer_'):
            unans.append('{}:{}'.format(korig, prop.GetPropValue('Value')))

    src_ig = []
    sp_list = response._nodes.get("SearchProperties", [])
    for prop in sp_list:
        sp_id = prop.GetPropValue('Id')
        prop_list = prop._nodes.get("Properties", [])
        for prop in prop_list:
            korig = prop.GetPropValue('Key')
            kt = _key_after_tdi_id(korig)
            k = kt if kt else korig
            if k == 'Ig':
                src_ig.append('{}:{}:{}'.format(sp_id, korig, prop.GetPropValue('Value')))
            elif k.startswith('Unanswer_'):
                unans.append('{}:{}:{}'.format(sp_id, korig, prop.GetPropValue('Value')))

    res.sort()
    src_ig.sort()
    unans.sort()
    return res + src_ig + unans


def _key_after_tdi_id(key):
    # try remove digits + '_' prefix from key and return it or return None
    # (use format tdi props declared epar@ in SEARCH-983)
    ksize = len(key)
    if ksize and key[0].isdigit():
        for i in range(1, ksize):
            if not key[i].isdigit():
                break
        if i < ksize and key[i] == '_':
            return key[i + 1:]
    return None


def get_bad_search_source(response):
    """
        Returns list of failed search sources
    """

    debug_info = response._nodes.get("DebugInfo")
    if debug_info:
        return debug_info[0]._props.get("FailedPrimusList", None)


def is_answer_full(response):
    """
        Returns true all basesearches was responded
    """

    # NOTE Actually is_answer_full() is not appropriate for basesearch responses.
    # Consequently the other thing TODO is not calling it in cases like that.
    debug_info = response._nodes.get("DebugInfo")
    if debug_info:
        not_respond_count = debug_info[0].GetPropValue("BaseSearchNotRespondCount", required=False)
        if not_respond_count is not None and int(not_respond_count) == 0:
            return True

    return False


def is_answer_complete(response):
    """
        Returns true if response was marked as a complete
    """

    debug_info = response._nodes.get("DebugInfo")
    if debug_info:
        return debug_info[0].GetPropValue("AnswerIsComplete", required=False) == "true"
    return False


def is_answer_not_fetched(response):
    """
        Returns True if some document was not fetched on snippet stage (i.e. answer is not complete)
        Relates SEARCH-556
    """

    for prop in response._nodes.get("SearcherProp", []):
        if _REMOVED_DOC_KEY_PATTERN.search(prop.GetPropValue("Key")):
            return True
    return False


def is_answer_empty(response):
    """
        Returns true if answer doesn't contains documents
    """

    for grouping_list in response._nodes.get("Grouping", []):
        for group_list in grouping_list._nodes.get("Group", []):
            if group_list._nodes.get("Document"):
                return False

    return True


def is_answer_erroneous(response):
    """
        Returns true if answer contains error
    """

    for error_info in response._nodes.get("ErrorInfo", []):
        if error_info.GetPropValue("GotError") != "NO":
            return True

    return False


def has_incomplete_responses(r1, r2):
    """
        False: both responses are complete, True otherwise
    """

    if is_answer_complete(r1) and is_answer_complete(r2):
        return False
    return True
