# -*- coding: utf-8 -*-

import logging

from sandbox.projects.common.base_search_quality.tree import htmldiff
from sandbox.projects.common.base_search_quality.tree.node import WalkPath

"""
    Вспомогательные методы обработки ответа от поиска
"""

KEY_SCHEME = "scheme.json_local_standard.nodump"
KEY_SCHEME_TREE = "SearcherProp-{}".format(KEY_SCHEME)


def sort_val(val, sep, sep2=None):
    """
        Упорядочение значении для стабильности при сравнениях
        (результаты сериализации хеша нестаибильны)
        :return sorted_val: str
    """
    sorted_val = ''
    for tok in sorted(val.split(sep)):
        if len(tok):
            if sep2 is not None:
                tok = sort_val(tok, sep2)
            if sorted_val:
                sorted_val += sep
            sorted_val += tok
    return sorted_val


def sort_search_prop_values(sp, sep, sep2=None):
    """
        Упорядочение данных в узлах типа SearchProp
        :param sp: -  узел дерева содержащий props Key/Value
        :param sep: - разделитель значений в значении Value
    """
    val = sp.GetPropValue("Value", required=False)
    if not val:
        return
    sorted_val = sort_val(val, sep, sep2)
    if val != sorted_val:
        raw = sp.GetPropValue("Value", parse=False)
        if raw[0] == '"' and raw[-1] == '"':
            sp.SetPropValue("Value", '"{}"'.format(sorted_val))
        else:
            sp.SetPropValue("Value", sorted_val)


def convert_scheme_to_tree(response, key_scheme=KEY_SCHEME, key_scheme_tree=None):
    """
        Конвертация json строки со схемой переранжирований в дерево
        (для более удобного отслеживания измененений/патчинга)
    """
    if key_scheme_tree is None:
        key_scheme_tree = "SearcherProp-{}".format(key_scheme)
    searcher_props = response._nodes.get("SearcherProp")
    if searcher_props:
        # convert SearcherProp json value to tree
        tmp_searcher_props = []
        for sp in searcher_props:
            key = sp.GetPropValue("Key")
            if key == key_scheme:
                try:
                    tree = htmldiff.json_dict_to_tree(sp.GetPropValue("Value"))
                    response._nodes[key_scheme_tree] = [tree]
                except Exception as e:
                    # игнорируем (оставляем as is) поля, которые не удалось распарзить
                    # (типа некорректного json который чинится в FRESHNESS-1691)
                    logging.warning('can not convert_scheme_to_tree (totally discard bad field):\n' + str(e))
            else:
                tmp_searcher_props.append(sp)
        searcher_props = tmp_searcher_props
        response._nodes["SearcherProp"] = searcher_props
    return key_scheme_tree


def convert_serp_data_to_tree(response):
    """
        Конвертация json строки с описанием вёрстки колдунщиков в дерево
        (для более удобного чтения diff-ов)
    """
    def process_archive_info(ai):
        oldGtaRelatedAttribute = ai._nodes.get('GtaRelatedAttribute')
        if not oldGtaRelatedAttribute:
            return
        newGtaRelatedAttribute = []
        for gta in oldGtaRelatedAttribute:
            if gta.GetPropValue('Key') == '_SerpData':
                if not gta.GetPropValue('Value'):
                    logging.debug("Empty value of _SerpData (SEARCH-5226)!")
                    t = htmldiff.json_dict_to_tree('{"!EMPTY!":"!EMPTY!"}')  # TODO SEARCH-5226
                else:
                    t = htmldiff.json_dict_to_tree(gta.GetPropValue('Value'))
                if 'GtaRelatedAttribute-_SerpData.json' in ai._nodes:
                    ai._nodes['GtaRelatedAttribute-_SerpData.json'].append(t)
                else:
                    ai._nodes['GtaRelatedAttribute-_SerpData.json'] = [t]
            else:
                newGtaRelatedAttribute.append(gta)
        ai._nodes['GtaRelatedAttribute'] = newGtaRelatedAttribute

    WalkPath(
        response,
        [
            "Grouping",
            "Group",
            "Document",
            "ArchiveInfo",
        ],
        process_archive_info,
    )

    def process_serp_data(ai):
        # TODO(dima-zakharov, mvel): remove this SEARCH-5820
        old_images = ai._nodes.get('images')
        if not old_images:
            return
        new_images = []
        for node in old_images:
            for name in ['thmb_href', 'passages', 'preview_dups']:
                if node.GetPropValue(name, required=False):
                    logging.debug("Removed %s:%s.", name, node.GetPropValue(name))
                    break
            else:
                new_images.append(node)
        ai._nodes['images'] = new_images

    WalkPath(
        response,
        [
            "Grouping",
            "Group",
            "Document",
            "ArchiveInfo",
            "GtaRelatedAttribute-_SerpData.json",
        ],
        process_serp_data,
    )


def purge(node, prop_name):
    """
        Удаление субнод/свойств
        - пустая нода (без субнод/свойств) вырождается в свойство, так что надо чистить обоих
        remove subnodes/property
        - empty node (without subnodes) become props, so need remove both
    """
    node._nodes.pop(prop_name, None)
    node._props.pop(prop_name, None)
